#include "zones_state_coordinator.h"

#include <infra/libs/yp_dns/dynamic_zones/services/bridge.h>

#include <infra/libs/yp_dns/dynamic_zones/helpers/yt/yt.h>

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/util/ypath_join.h>

#include <yp/cpp/yp/token.h>

#include <util/generic/serialized_enum.h>
#include <util/string/join.h>

namespace NYpDns::NDynamicZones {

////////////////////////////////////////////////////////////////////////////////

namespace {

template <TZonesStateCoordinator::EServiceType serviceType>
class ITypedService : public TZonesStateCoordinator::IService {
public:
    void Start(NInfra::TLogFramePtr logFrame) override {
        Y_UNUSED(logFrame);
    }

    void ReopenLogs() override {
    }

    void HandleFeedback(const google::protobuf::Message& /* feedback */) const override {
        ythrow TWithBackTrace<yexception>() << "HandleFeedback method is not implemented for service " << ToString(serviceType); }

    bool NeedStaticZones() const override {
        return false;
    }

    grpc::Service* GetGrpcService() const override {
        return nullptr;
    }
};

template <TZonesStateCoordinator::EServiceType>
class TService;

#define DEFINE_SERVICE(serviceType) \
    template <>                                                                     \
    class TService<TZonesStateCoordinator::EServiceType::serviceType>               \
        : public ITypedService<TZonesStateCoordinator::EServiceType::serviceType>   \

DEFINE_SERVICE(BRIDGE) {
public:
    TService(
        NBridgeService::TStateServiceConfig stateServiceConfig,
        TStateServiceCommonOptions stateServiceCommonOptions
    )
        : StateService_(std::move(stateServiceConfig), std::move(stateServiceCommonOptions))
        , StateGrpcService_(StateService_)
    {
    }

    void Start(NInfra::TLogFramePtr logFrame) override {
        StateService_.Start(logFrame);
    }

    void ReopenLogs() override {
        StateService_.ReopenLogs();
    }

    bool IsInConfiguration(const TZoneId& zoneId, NInfra::TLogFramePtr logFrame) const override {
        return StateService_.AreAllInstancesHasZone(zoneId, logFrame);
    }

    bool IsSuitableState(const TZoneStateEntry& zoneState) const override {
        switch (FromString<EZoneState>(zoneState["target_state"].GetString())) {
            case EZoneState::NOT_EXIST:
                return false;
            case EZoneState::CREATED:
            case EZoneState::PREPARED:
            case EZoneState::SERVING:
                return true;
        }
    }

    grpc::Service* GetGrpcService() const override {
        return &dynamic_cast<grpc::Service&>(StateGrpcService_);
    }

private:
    NBridgeService::TStateService StateService_;
    mutable NBridgeService::TStateGrpcService StateGrpcService_;
};

DEFINE_SERVICE(REPLICATOR) {
public:
    bool IsInConfiguration(const TZoneId& zoneId, NInfra::TLogFramePtr logFrame) const override {
        Y_UNUSED(zoneId, logFrame);
        // Not used
        return false;
    }

    bool IsSuitableState(const TZoneStateEntry& zoneState) const override {
        switch (FromString<EZoneState>(zoneState["target_state"].GetString())) {
            case EZoneState::NOT_EXIST:
                return false;
            case EZoneState::CREATED:
            case EZoneState::PREPARED:
            case EZoneState::SERVING:
                return true;
        }
    }
};

DEFINE_SERVICE(DNS_AUTH) {
public:
    bool IsInConfiguration(const TZoneId& zoneId, NInfra::TLogFramePtr logFrame) const override {
        Y_UNUSED(zoneId, logFrame);
        // TODO
        return false;
    }

    bool IsSuitableState(const TZoneStateEntry& zoneState) const override {
        switch (FromString<EZoneState>(zoneState["target_state"].GetString())) {
            case EZoneState::NOT_EXIST:
            case EZoneState::CREATED:
            case EZoneState::PREPARED:
                return false;
            case EZoneState::SERVING:
                return true;
        }
    }
};

DEFINE_SERVICE(DNS_API) {
public:
    bool IsInConfiguration(const TZoneId& zoneId, NInfra::TLogFramePtr logFrame) const override {
        Y_UNUSED(zoneId, logFrame);
        return false;
    }

    bool IsSuitableState(const TZoneStateEntry& zoneState) const override {
        return IsSuitableCurrentState(zoneState) && IsSuitableTargetState(zoneState);
    }

    bool NeedStaticZones() const override {
        return true;
    }

private:
    bool IsSuitableCurrentState(const TZoneStateEntry& zoneState) const {
        switch (FromString<EZoneState>(zoneState["current_state"].GetString())) {
            case EZoneState::NOT_EXIST:
                return false;
            case EZoneState::CREATED:
            case EZoneState::PREPARED:
            case EZoneState::SERVING:
                return true;
        }
    }

    bool IsSuitableTargetState(const TZoneStateEntry& zoneState) const {
        switch (FromString<EZoneState>(zoneState["target_state"].GetString())) {
            case EZoneState::NOT_EXIST:
                return false;
            case EZoneState::CREATED:
            case EZoneState::PREPARED:
            case EZoneState::SERVING:
                return true;
        }
    }
};

DEFINE_SERVICE(ZONE_PREPARED) {
public:
    bool IsInConfiguration(const TZoneId& zoneId, NInfra::TLogFramePtr logFrame) const override {
        Y_UNUSED(zoneId, logFrame);
        // TODO
        return false;
    }

    // Unused
    bool IsSuitableState(const TZoneStateEntry& zoneState) const override {
        Y_UNUSED(zoneState);
        return false;
    }
};

DEFINE_SERVICE(YP_DNS_EXPORT) {
public:
    bool IsInConfiguration(const TZoneId& zoneId, NInfra::TLogFramePtr logFrame) const override {
        Y_UNUSED(zoneId, logFrame);
        // Not used
        return false;
    }

    bool IsSuitableState(const TZoneStateEntry& zoneState) const override {
        return IsSuitableCurrentState(zoneState) || IsSuitableTargetState(zoneState);
    }

private:
    bool IsSuitableCurrentState(const TZoneStateEntry& zoneState) const {
        switch (FromString<EZoneState>(zoneState["current_state"].GetString())) {
            case EZoneState::NOT_EXIST:
                return false;
            case EZoneState::CREATED:
            case EZoneState::PREPARED:
            case EZoneState::SERVING:
                return true;
        }
    }

    bool IsSuitableTargetState(const TZoneStateEntry& zoneState) const {
        switch (FromString<EZoneState>(zoneState["target_state"].GetString())) {
            case EZoneState::NOT_EXIST:
                return false;
            case EZoneState::CREATED:
            case EZoneState::PREPARED:
            case EZoneState::SERVING:
                return true;
        }
    }
};

} // anonymous namespace

////////////////////////////////////////////////////////////////////////////////

TStateServicesOptions::TStateServicesOptions(TStateServicesConfig config)
    : BridgeStateServiceConfig(std::move(*config.MutableBridgeStateServiceConfig()))
{
}

TZonesStateCoordinatorOptions::TZonesStateCoordinatorOptions(const TZonesStateCoordinatorConfig& config)
    : YtProxy(config.GetYtProxy())
    , CypressRootPath(config.GetCypressRootPath())
    , YpAddress(config.GetYpAddress())
    , YpToken(NYP::NClient::FindToken())
    , GrpcServerConfig(config.GetGrpcServerConfig())
    , StateServicesOptions(config.GetStateServicesConfig())
{
}

////////////////////////////////////////////////////////////////////////////////

TZonesStateCoordinator::TZonesStateCoordinator(TZonesStateCoordinatorOptions options)
    : Options_(std::move(options))
    , StateServiceCommonOptions_(
        TStateServiceCommonOptions()
            .SetYtProxy(Options_.YtProxy)
            .SetYtToken(Options_.YtToken)
            .SetCypressRootPath(NYT::JoinYPaths(Options_.CypressRootPath, "services"))
    )
    , Services_({
        {
            EServiceType::BRIDGE, MakeSimpleShared<TService<EServiceType::BRIDGE>>(
                Options_.StateServicesOptions.BridgeStateServiceConfig,
                StateServiceCommonOptions_
                    .SetCypressRootPath(NYT::JoinYPaths(StateServiceCommonOptions_.CypressRootPath, to_lower(ToString(EServiceType::BRIDGE))))
            )
        },
        {EServiceType::REPLICATOR,          MakeSimpleShared<TService<EServiceType::REPLICATOR>>()},
        {EServiceType::DNS_AUTH,            MakeSimpleShared<TService<EServiceType::DNS_AUTH>>()},
        {EServiceType::DNS_API,             MakeSimpleShared<TService<EServiceType::DNS_API>>()},
        {EServiceType::ZONE_PREPARED,       MakeSimpleShared<TService<EServiceType::ZONE_PREPARED>>()},
        {EServiceType::YP_DNS_EXPORT,       MakeSimpleShared<TService<EServiceType::YP_DNS_EXPORT>>()},
    })
    , YtClient_(NYT::CreateClient(
        Options_.YtProxy,
        NYT::TCreateClientOptions()
            .Token(Options_.YtToken)))
    , YpClient_(NYP::NClient::CreateClient(
        NYP::NClient::TClientOptions()
            .SetAddress(Options_.YpAddress)
            .SetEnableBalancing(true)
            .SetEnableSsl(true)
            .SetToken(Options_.YpToken)))
    , GrpcServer_(Options_.GrpcServerConfig)
{
    ValidateInitialState();
    InitGrpcServer();
}

void TZonesStateCoordinator::InitGrpcServer() {
    for (const auto& [_, service] : Services_) {
        if (grpc::Service* grpcService = service->GetGrpcService()) {
            GrpcServer_.RegisterService(grpcService);
        }
    }
}

void TZonesStateCoordinator::Start(NInfra::TLogFramePtr logFrame) {
    for (const auto& [_, service] : Services_) {
        service->Start(logFrame);
    }
    GrpcServer_.Start(logFrame);
}

void TZonesStateCoordinator::Stop(NInfra::TLogFramePtr logFrame) {
    GrpcServer_.Stop(logFrame);
}

void TZonesStateCoordinator::ReopenLogs() {
    for (const auto& [_, service] : Services_) {
        service->ReopenLogs();
    }
}

TZoneStateEntry TZonesStateCoordinator::GetState(
    const TZoneId& zoneId,
    const TMaybe<NYP::NClient::TDnsZone>& ypZoneData,
    NInfra::TLogFramePtr logFrame
) {
    TZoneStateEntry result;
    if (NJson::TJsonValue targetState; ypZoneData.Defined() &&
            ypZoneData->Labels().GetValueByPath("state.target_state", targetState) &&
            targetState.IsString()
    ) {
        result["target_state"] = targetState.GetString();
    } else {
        result["target_state"] = ToString(EZoneState::NOT_EXIST);
    }
    result["current_state"] = ToString(GetCurrentState(zoneId, logFrame));
    return result;
}

TZoneStateEntry TZonesStateCoordinator::GetState(
    const TZone& zone,
    const TMaybe<NYP::NClient::TDnsZone>& ypZoneData,
    NInfra::TLogFramePtr logFrame
) {
    return GetState(zone.GetId(), ypZoneData, logFrame);
}

THashMap<TZoneId, TZoneStateEntry> TZonesStateCoordinator::AcceptFeedback(const TServiceFeedback& feedback) {
    Y_ENSURE(feedback.Data);
    GetService(feedback.ServiceType).HandleFeedback(*feedback.Data);
    // TODO
    return {};
}

bool TZonesStateCoordinator::HasService(const TServiceType& serviceTypeName) const {
    EServiceType serviceType;
    return TryFromString<EServiceType>(serviceTypeName, serviceType);
}

bool TZonesStateCoordinator::IsZoneReadyForService(const TServiceType& serviceType, const TZoneId& zoneId, const TZoneStateEntry& zoneState) const {
    Y_UNUSED(zoneId);
    return GetService(serviceType).IsSuitableState(zoneState);
}

bool TZonesStateCoordinator::NeedStaticZones(const TServiceType& serviceType) const {
    return GetService(serviceType).NeedStaticZones();
}

TZoneStateEntry TZonesStateCoordinator::OnCreateZone(
    const TZone& zone,
    const TMaybe<NYP::NClient::TDnsZone>& ypZoneData,
    NInfra::TLogFramePtr logFrame
) {
    TZoneStateEntry result = GetState(zone, ypZoneData, logFrame);
    if (FromString<EZoneState>(result["target_state"].GetString()) == EZoneState::NOT_EXIST) {
        result["target_state"] = ToString(EZoneState::CREATED);
    } else {
        ythrow yexception() << "Zone \"" << zone.GetId() << "\" already created";
    }
    return result;
}

TZoneStateEntry TZonesStateCoordinator::OnRemoveZone(
    const TZoneId& zoneId,
    const TMaybe<NYP::NClient::TDnsZone>& ypZoneData,
    NInfra::TLogFramePtr logFrame
) {
    TZoneStateEntry result = GetState(zoneId, ypZoneData, logFrame);
    if (FromString<EZoneState>(result["target_state"].GetString()) != EZoneState::NOT_EXIST) {
        result["target_state"] = ToString(EZoneState::NOT_EXIST);
    } else {
        ythrow yexception() << "Zone \"" << zoneId << "\" << does not exist";
    }
    return result;
}

TZoneStateEntry TZonesStateCoordinator::OnUpdateZone(
    const TZone& zone,
    const TMaybe<NYP::NClient::TDnsZone>& ypZoneData,
    NInfra::TLogFramePtr logFrame
) {
    // State does not change
    return GetState(zone, ypZoneData, logFrame);
}

TZonesStateCoordinator::IService& TZonesStateCoordinator::GetService(const TServiceType& serviceTypeName) const {
    const EServiceType serviceType = FromString<EServiceType>(serviceTypeName);
    return GetService(serviceType);
}

TZonesStateCoordinator::IService& TZonesStateCoordinator::GetService(const EServiceType& serviceType) const {
    return *Services_.at(serviceType);
}

void TZonesStateCoordinator::ValidateInitialState() const {
    TVector<TString> notRegisteredServiceTypes;
    for (const EServiceType serviceType : GetEnumAllValues<EServiceType>()) {
        if (!Services_.contains(serviceType)) {
            notRegisteredServiceTypes.push_back(ToString(serviceType));
        }
    }
    Y_ENSURE(notRegisteredServiceTypes.empty(), "Services are not registered for types " << JoinSeq(", ", notRegisteredServiceTypes));
}

TZonesStateCoordinator::EZoneState TZonesStateCoordinator::GetCurrentState(
    const TZoneId& zoneId,
    NInfra::TLogFramePtr logFrame
) const {
    EZoneState result = EZoneState::NOT_EXIST;

    // Check if zone created
    if (!GetService(EServiceType::BRIDGE).IsInConfiguration(zoneId, logFrame)) {
        return result;
    }
    result = EZoneState::CREATED;

    // Check if zone prepared
    if (!GetService(EServiceType::ZONE_PREPARED).IsInConfiguration(zoneId, logFrame)) {
        return result;
    }
    result = EZoneState::PREPARED;

    // Check if zone is serving by nameservers
    for (const EServiceType nameserverServiceType : {EServiceType::DNS_AUTH}) {
        if (!GetService(nameserverServiceType).IsInConfiguration(zoneId, logFrame)) {
            return result;
        }
    }
    result = EZoneState::SERVING;

    return result;
}

////////////////////////////////////////////////////////////////////////////////

} // namespace NYpDns::NDynamicZones
