#pragma once

#include "dns_zones_replica.h"
#include "service_feedback.h"

#include <infra/libs/yp_dns/dynamic_zones/protos/configs/zones_state_coordinator.pb.h>

#include <infra/libs/yp_dns/dynamic_zones/services/service.h>

#include <infra/libs/yp_dns/dynamic_zones/services/protos/configs/bridge.pb.h>

#include <infra/libs/yp_dns/zone/zone.h>

#include <infra/libs/grpc_server/server.h>
#include <infra/libs/logger/log_frame.h>

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/interface/fwd.h>

#include <yp/cpp/yp/client.h>

#include <library/cpp/yson/node/node.h>

#include <util/generic/noncopyable.h>
#include <util/generic/ptr.h>

namespace grpc {
    class Service;
} // namespace grpc

namespace NYpDns::NDynamicZones {

////////////////////////////////////////////////////////////////////////////////

using TZoneStateEntry = NJson::TJsonValue;

////////////////////////////////////////////////////////////////////////////////

class IZonesStateCoordinator : public TNonCopyable {
public:
    virtual ~IZonesStateCoordinator() = default;

    virtual void ReopenLogs() = 0;

    virtual TZoneStateEntry GetState(
        const TZoneId& zoneId,
        const TMaybe<NYP::NClient::TDnsZone>& ypZoneData,
        NInfra::TLogFramePtr logFrame) = 0;

    virtual THashMap<TZoneId, TZoneStateEntry> AcceptFeedback(const TServiceFeedback& feedback) = 0;

    virtual bool HasService(const TServiceType& serviceType) const = 0;

    virtual bool IsZoneReadyForService(const TServiceType& serviceType, const TZoneId& zoneId, const TZoneStateEntry& zoneState) const = 0;

    virtual bool NeedStaticZones(const TServiceType& serviceType) const = 0;

    /// Callbacks
    virtual TZoneStateEntry OnCreateZone(
        const TZone& zone,
        const TMaybe<NYP::NClient::TDnsZone>& ypZoneData,
        NInfra::TLogFramePtr logFrame
    ) = 0;

    virtual TZoneStateEntry OnRemoveZone(
        const TZoneId& zoneId,
        const TMaybe<NYP::NClient::TDnsZone>& ypZoneData,
        NInfra::TLogFramePtr logFrame
    ) = 0;

    virtual TZoneStateEntry OnUpdateZone(
        const TZone& zone,
        const TMaybe<NYP::NClient::TDnsZone>& ypZoneData,
        NInfra::TLogFramePtr logFrame
    ) = 0;
};

using IZonesStateCoordinatorPtr = TAtomicSharedPtr<IZonesStateCoordinator>;

////////////////////////////////////////////////////////////////////////////////

struct TStateServicesOptions {
    NBridgeService::TStateServiceConfig BridgeStateServiceConfig;

    TStateServicesOptions(TStateServicesConfig config);
};

struct TZonesStateCoordinatorOptions {
    TString YtProxy;
    TString YtToken;
    NYT::TYPath CypressRootPath;

    TString YpAddress;
    TString YpToken;

    NInfra::TGrpcServerConfig GrpcServerConfig;
    TStateServicesOptions StateServicesOptions;

    TZonesStateCoordinatorOptions(const TZonesStateCoordinatorConfig& config);
};

////////////////////////////////////////////////////////////////////////////////

class TZonesStateCoordinator : public IZonesStateCoordinator {
public:
    enum class EZoneState {
        NOT_EXIST = 0,
        CREATED   = 1,
        PREPARED  = 2,
        SERVING   = 20,
    };

    // [BEGIN EServiceType]
    enum class EServiceType {
        BRIDGE            = 1 /* "BRIDGE" */,
        REPLICATOR        = 2 /* "REPLICATOR" */,
        DNS_AUTH          = 3 /* "DNS_AUTH" */,
        DNS_API           = 4 /* "DNS-API" */,
        ZONE_PREPARED     = 5 /* "ZONE_PREPARED" */,
        YP_DNS_EXPORT     = 6 /* "YP-DNS-EXPORT" */,
    };
    // [END EServiceType]

    class IService {
    public:
        using EZoneState = EZoneState;
        using EServiceType = EServiceType;

        virtual void Start(NInfra::TLogFramePtr logFrame) = 0;
        virtual void ReopenLogs() = 0;
        virtual bool IsInConfiguration(const TZoneId& zoneId, NInfra::TLogFramePtr logFrame) const = 0;
        virtual bool IsSuitableState(const TZoneStateEntry& zoneState) const = 0;
        virtual bool NeedStaticZones() const = 0;
        virtual void HandleFeedback(const google::protobuf::Message& /* feedback */) const = 0;

        virtual grpc::Service* GetGrpcService() const = 0;

        virtual ~IService() = default;
    };

public:
    TZonesStateCoordinator(TZonesStateCoordinatorOptions options);

    void Start(NInfra::TLogFramePtr logFrame);
    void Stop(NInfra::TLogFramePtr logFrame);

    void ReopenLogs() override;

    TZoneStateEntry GetState(
        const TZoneId& zoneId,
        const TMaybe<NYP::NClient::TDnsZone>& ypZoneData,
        NInfra::TLogFramePtr logFrame
    ) override;

    TZoneStateEntry GetState(
        const TZone& zone,
        const TMaybe<NYP::NClient::TDnsZone>& ypZoneData,
        NInfra::TLogFramePtr logFrame
    );

    THashMap<TZoneId, TZoneStateEntry> AcceptFeedback(
        const TServiceFeedback& feedback
    ) override;

    bool HasService(const TServiceType& serviceType) const override;

    bool IsZoneReadyForService(
        const TServiceType& serviceTypeName,
        const TZoneId& zoneId,
        const TZoneStateEntry& zoneState
    ) const override;

    bool NeedStaticZones(const TServiceType& serviceType) const override;

    TZoneStateEntry OnCreateZone(
        const TZone& zone,
        const TMaybe<NYP::NClient::TDnsZone>& ypZoneData,
        NInfra::TLogFramePtr logFrame
    ) override;

    TZoneStateEntry OnRemoveZone(
        const TZoneId& zoneId,
        const TMaybe<NYP::NClient::TDnsZone>& ypZoneData,
        NInfra::TLogFramePtr logFrame
    ) override;

    TZoneStateEntry OnUpdateZone(
        const TZone& zone,
        const TMaybe<NYP::NClient::TDnsZone>& ypZoneData,
        NInfra::TLogFramePtr logFrame
    ) override;

    IService& GetService(const TServiceType& serviceTypeName) const;

private:
    void ValidateInitialState() const;

    void InitGrpcServer();

    IService& GetService(const EServiceType& serviceType) const;

    EZoneState GetCurrentState(const TZoneId& zoneId, NInfra::TLogFramePtr logFrame) const;

private:
    const TZonesStateCoordinatorOptions Options_;
    TStateServiceCommonOptions StateServiceCommonOptions_;
    const THashMap<EServiceType, TSimpleSharedPtr<IService>> Services_;
    const NYT::IClientPtr YtClient_;
    const NYP::NClient::TClientPtr YpClient_;

    NInfra::TGrpcServer GrpcServer_;
};

////////////////////////////////////////////////////////////////////////////////

} // namespace NYpDns::NDynamicZones
