#include "record.h"

#include <util/string/cast.h>

namespace NYpDns {

using NYP::NClient::NApi::NProto::TDnsRecordSetSpec;

ERecordType GetRecordType(TDnsRecordSetSpec::TResourceRecord::EType type) {
    switch (type) {
    case TDnsRecordSetSpec::TResourceRecord::A:
        return ERecordType::A;
    case TDnsRecordSetSpec::TResourceRecord::NS:
        return ERecordType::NS;
    case TDnsRecordSetSpec::TResourceRecord::CNAME:
        return ERecordType::CNAME;
    case TDnsRecordSetSpec::TResourceRecord::SOA:
        return ERecordType::SOA;
    case TDnsRecordSetSpec::TResourceRecord::PTR:
        return ERecordType::PTR;
    case TDnsRecordSetSpec::TResourceRecord::MX:
        return ERecordType::MX;
    case TDnsRecordSetSpec::TResourceRecord::TXT:
        return ERecordType::TXT;
    case TDnsRecordSetSpec::TResourceRecord::AAAA:
        return ERecordType::AAAA;
    case TDnsRecordSetSpec::TResourceRecord::SRV:
        return ERecordType::SRV;
    case TDnsRecordSetSpec::TResourceRecord::RRSIG:
        return ERecordType::RRSIG;
    case TDnsRecordSetSpec::TResourceRecord::NSEC:
        return ERecordType::NSEC;
    case TDnsRecordSetSpec::TResourceRecord::DNSKEY:
        return ERecordType::DNSKEY;
    case TDnsRecordSetSpec::TResourceRecord::CAA:
        return ERecordType::CAA;
    }
}

TDnsRecordSetSpec::TResourceRecord::EType GetProtoRecordType(ERecordType type) {   
    switch (type) {
    case ERecordType::A:
        return TDnsRecordSetSpec::TResourceRecord::A;
    case ERecordType::NS:
        return TDnsRecordSetSpec::TResourceRecord::NS;
    case ERecordType::CNAME:
        return TDnsRecordSetSpec::TResourceRecord::CNAME;
    case ERecordType::SOA:
        return TDnsRecordSetSpec::TResourceRecord::SOA;
    case ERecordType::PTR:
        return TDnsRecordSetSpec::TResourceRecord::PTR;
    case ERecordType::MX:
        return TDnsRecordSetSpec::TResourceRecord::MX;
    case ERecordType::TXT:
        return TDnsRecordSetSpec::TResourceRecord::TXT;
    case ERecordType::AAAA:
        return TDnsRecordSetSpec::TResourceRecord::AAAA;
    case ERecordType::SRV:
        return TDnsRecordSetSpec::TResourceRecord::SRV;
    case ERecordType::RRSIG:
        return TDnsRecordSetSpec::TResourceRecord::RRSIG;
    case ERecordType::NSEC:
        return TDnsRecordSetSpec::TResourceRecord::NSEC;
    case ERecordType::DNSKEY:
        return TDnsRecordSetSpec::TResourceRecord::DNSKEY;
    case ERecordType::CAA:
        return TDnsRecordSetSpec::TResourceRecord::CAA;
    }
}

bool IsFieldDelimiterOfEntry(unsigned char c) {
    return c == ' ' || c == '\t';
}

ui32 MakeSerialFromYpTimestamp(const ui64 ypTimestamp) {
    return ypTimestamp >> 30u;
}

ui32 MakeSerialFromYpTimestamps(const THashMap<TString, ui64>& ypTimestamps) {
    ui64 maxYpTimestamp = 0;
    for (const auto& [cluster, ypTimestamp] : ypTimestamps) {
        if (maxYpTimestamp < ypTimestamp) {
            maxYpTimestamp = ypTimestamp;
        }
    }
    return MakeSerialFromYpTimestamp(maxYpTimestamp);
}

TRecord CreateNSRecord(const DNSName& domain, const TString& nameserver, const ui32 ttl) {
    TRecord record;
    record.Name = domain.makeLowerCase();
    record.Class = ERecordClass::IN;
    record.Type = ERecordType::NS;
    record.Ttl = ttl;
    record.Data = nameserver;
    return record;
}

TRecord CreateNSRecord(const TString& domain, const TString& nameserver, const ui32 ttl) {
    return CreateNSRecord(DNSName(domain), nameserver, ttl);
}

NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord CreateNSRecordProto(const TString& nameserver, const ui32 ttl) {
    NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord record;
    record.set_ttl(ttl);
    record.set_class_("IN");
    record.set_type(NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord::NS);
    record.set_data(nameserver);
    return record;
}

TRecord CreateSOARecord(const TZoneConfig& zone, const ui32 serial) {
    TRecord record;
    record.Name = DNSName(zone.GetName()).makeLowerCase();
    record.Class = ERecordClass::IN;
    record.Type = ERecordType::SOA;
    record.Ttl = zone.GetSOARecordTtl();
    record.Data = TSOAData(zone, serial).ToString();
    return record;
}

NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord CreateSOARecordProto(const TZoneConfig& zone, const ui32 serial) {
    NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord record;
    record.set_ttl(zone.GetSOARecordTtl());
    record.set_class_("IN");
    record.set_type(NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord::SOA);
    record.set_data(TSOAData(zone, serial).ToString());
    return record;
}

TRecord CreateRecordFromProto(const DNSName& domain, const NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord& proto, const TMaybe<ui32>& defaultTtl) {
    TRecord record;
    record.Name = domain.makeLowerCase();
    record.Class = ERecordClass::IN;
    record.Type = GetRecordType(proto.type());
    record.Ttl = proto.has_ttl() ? proto.ttl() : defaultTtl.GetOrElse(proto.ttl());
    record.Data = proto.data();
    return record;
}

TRecord CreateRecordFromProto(const DNSName& domain, const NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord& proto) {
    return CreateRecordFromProto(domain, proto, Nothing());
}

TRecord CreateRecordFromProto(const DNSName& domain, const TDnsRecordSetSpec::TResourceRecord& proto, const ui32 defaultTtl) {
    return CreateRecordFromProto(domain, proto, MakeMaybe(defaultTtl));
}

NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord CreateProtoFromRecord(const TRecord& record) {
    NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord proto;
    if (record.Ttl) {
        proto.set_ttl(record.Ttl);
    }
    proto.set_class_(ToString(record.Class));
    proto.set_type(GetProtoRecordType(record.Type));
    proto.set_data(record.Data);
    return proto;
}

TRecord CreateRecordFromString(TStringBuf line, const TMaybe<TString>& origin, ui64 defaultTtl, const TMaybe<DNSName>& prevRecordName) {
    TRecord record;

    bool hasTtl = false;
    bool hasClass = false;
    TString incorrectRecordStringMessage = TString::Join("Incorrect record line: \"", TString(line), "\". ");

    Y_ENSURE_EX(!line.empty(), TIncorrectRecordStringException() << incorrectRecordStringMessage << "Line is empty");

    if (IsFieldDelimiterOfEntry(line[0])) { // Name is empty
        Y_ENSURE_EX(prevRecordName.Defined(), TIncorrectRecordStringException() << incorrectRecordStringMessage << "Name is empty");
        record.Name = *prevRecordName;
    } else { // Name isn`t empty
        TString recordName;
        bool isOkSplit = StringSplitter(line)
            .SplitByFunc(&IsFieldDelimiterOfEntry)
            .SkipEmpty()
            .Limit(2)
            .TryCollectInto(&recordName, &line);

        Y_ENSURE_EX(isOkSplit, TIncorrectRecordStringException() << incorrectRecordStringMessage << "Not enough fields in the record");

        if (recordName.back() != '.') {
            Y_ENSURE_EX(origin.Defined(), TIncorrectRecordStringException() << incorrectRecordStringMessage << "Incorrect Name. $ORIGIN is missing");
            /**
             * Domain names which do not end in a dot are called relative.
             * A relative name is an error when no origin is available.
             * [Page 33] https://datatracker.ietf.org/doc/html/rfc1035
             */
            if (recordName == "@") {
                recordName = *origin;
            } else {
                recordName = Join('.', recordName, *origin);
            }
        }
        record.Name = DNSName(recordName);
    }

    while (true) {
        TStringBuf recordField;

        bool isOkSplit = StringSplitter(line)
            .SplitByFunc(&IsFieldDelimiterOfEntry)
            .SkipEmpty()
            .Limit(2)
            .TryCollectInto(&recordField, &line);
        
        Y_ENSURE_EX(isOkSplit, TIncorrectRecordStringException() << incorrectRecordStringMessage << "Not enough fields in the record");
        
        if (TryFromString(recordField, record.Ttl)) {
            if (hasTtl) {
                ythrow TIncorrectRecordStringException() << incorrectRecordStringMessage << "Ttl was repeated";
            }
            hasTtl = true;
        } else if (TryFromString(recordField, record.Class)) {
            if (hasClass) {
                ythrow TIncorrectRecordStringException() << incorrectRecordStringMessage << "Class was repeated";
            }
            hasClass = true;
        } else if (TryFromString(recordField, record.Type)) {
            break;
        } else if (hasTtl && hasClass) {
            ythrow TIncorrectRecordStringException() << incorrectRecordStringMessage << "The type field is missing";
        } else {
            ythrow TIncorrectRecordStringException() << incorrectRecordStringMessage << "Unknown field before the type";
        }
    }

    Y_ENSURE_EX(hasClass, TIncorrectRecordStringException() << incorrectRecordStringMessage << "The class field is missing");

    if (!hasTtl) {
        record.Ttl = defaultTtl;
    }
    while (IsFieldDelimiterOfEntry(line.back())) {
        line.Chop(1);
    }
    record.Data = line;

    return record;
}

} // namespace NYpDns
