#pragma once

#include <infra/libs/yp_dns/zone/protos/config.pb.h>

#include <infra/contrib/pdns/power_dns/dnsname.hh>

#include <yp/yp_proto/yp/client/api/proto/data_model.pb.h>

#include <util/generic/hash.h>
#include <util/generic/maybe.h>
#include <util/generic/string.h>
#include <util/string/join.h>
#include <util/string/split.h>

namespace NYpDns {

enum class ERecordType {
    A       = 1   /* "A" */,
    NS      = 2   /* "NS" */,
    CNAME   = 5   /* "CNAME" */,
    SOA     = 6   /* "SOA" */,
    PTR     = 12  /* "PTR" */,
    MX      = 15  /* "MX" */,
    TXT     = 16  /* "TXT" */,
    AAAA    = 28  /* "AAAA" */,
    SRV     = 33  /* "SRV" */,
    RRSIG   = 46  /* "RRSIG" */,
    NSEC    = 47  /* "NSEC" */,
    DNSKEY  = 48  /* "DNSKEY" */,
    CAA     = 257 /* "CAA" */,
};

ERecordType GetRecordType(NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord::EType type);

NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord::EType GetProtoRecordType(ERecordType type);

struct TSOAData {
    TString PrimaryNameserver;
    TString HostMaster;
    ui64 Serial = 0;
    ui64 Refresh = 900;
    ui64 Retry = 600;
    ui64 Expire = 604800;
    ui64 NegativeTtl = 120;

    TSOAData(const TZoneConfig& zone, const ui64 serial)
        : PrimaryNameserver(DNSName(zone.GetPrimaryNameserver()).toString())
        , HostMaster(DNSName(zone.GetHostmaster()).toString())
        , Serial(serial)
        , Refresh(zone.GetSOARefresh())
        , Retry(zone.GetSOARetry())
        , Expire(zone.GetSOAExpire())
        , NegativeTtl(zone.GetSOARecordMinimumTtl())
    {
    }

    // data example: "ns3.yandex.ru. sysadmin.yandex.ru. 2019012931 900 600 3600000 300"
    TSOAData(const TStringBuf data) {
        StringSplitter(data)
            .SplitBySet(" \t")
            .SkipEmpty()
            .CollectInto(
                &PrimaryNameserver,
                &HostMaster,
                &Serial,
                &Refresh,
                &Retry,
                &Expire,
                &NegativeTtl);
    }

    TString ToString() const {
        return Join(' ', PrimaryNameserver, HostMaster, Serial, Refresh, Retry, Expire, NegativeTtl);
    }
};


enum class ERecordClass {
    IN = 1 /* IN */,
};

struct TIncorrectRecordStringException : yexception {
};

enum class ERecordDataFormat {
    RAW = 0,
    ZONEFILE = 1,
};

struct TRecord {
    DNSName Name;
    ERecordClass Class;
    ERecordType Type;
    ui64 Ttl = 0;
    TString Data;

    int Compare(const TRecord& rhs) const {
        if (Name != rhs.Name) {
            return Name < rhs.Name ? -1 : 1;
        } else if (Y_UNLIKELY(Class != rhs.Class)) {
            return static_cast<int>(Class) - static_cast<int>(rhs.Class);
        } else if (Type != rhs.Type) {
            return static_cast<int>(Type) - static_cast<int>(rhs.Type);
        }
        return TString::compare(Data, rhs.Data);
    }

    bool operator==(const TRecord& rhs) const {
        return Compare(rhs) == 0;
    }

    bool operator<(const TRecord& rhs) const {
        return Compare(rhs) < 0;
    }

    TString FormatData(ERecordDataFormat format = ERecordDataFormat::ZONEFILE) const {
        switch (format) {
            case ERecordDataFormat::RAW: {
                return Data;
            }
            case ERecordDataFormat::ZONEFILE: {
                TString data;
                switch (Type) {
                    case ERecordType::TXT: {
                        data = TString::Join("\"", Data, "\"");
                        break;
                    }
                    default: {
                        data = Data;
                        break;
                    }
                }
                return data;
            }
        }
    }

    TString ToString(char delimiter = ' ') const {
        return Join(delimiter, Name.toString(), Ttl, Class, Type, FormatData());
    }
};

bool IsFieldDelimiterOfEntry(unsigned char c);

ui32 MakeSerialFromYpTimestamp(const ui64 ypTimestamp);

ui32 MakeSerialFromYpTimestamps(const THashMap<TString, ui64>& ypTimestamps);

TRecord CreateNSRecord(const DNSName& domain, const TString& nameserver, const ui32 ttl);

TRecord CreateNSRecord(const TString& domain, const TString& nameserver, const ui32 ttl);

NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord CreateNSRecordProto(const TString& nameserver, const ui32 ttl);

TRecord CreateSOARecord(const TZoneConfig& zone, const ui32 serial);

NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord CreateSOARecordProto(const TZoneConfig& zone, const ui32 serial);

TRecord CreateRecordFromProto(const DNSName& domain, const NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord& proto);

TRecord CreateRecordFromProto(const DNSName& domain, const NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord& proto, const ui32 defaultTtl);

NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord CreateProtoFromRecord(const TRecord& record);

TRecord CreateRecordFromString(TStringBuf line, const TMaybe<TString>& origin = Nothing(), ui64 defaultTtl = 0, const TMaybe<DNSName>& prevRecordName = Nothing());

} // namespace NYpDns
