#pragma once

#include <infra/libs/yp_dns/changelist/changelist.h>

#include <infra/libs/yp_replica/replica_objects.h>

#include <yp/yp_proto/yp/client/api/proto/data_model.pb.h>
#include <yp/cpp/yp/data_model.h>

#include <google/protobuf/util/message_differencer.h>

#include <util/generic/maybe.h>
#include <util/ysaveload.h>

namespace NYpDns {

class TRecordSet {
public:
    TRecordSet();
    TRecordSet(NYP::NClient::TDnsRecordSet&& recordSet);
    TRecordSet(const ui64 ypTimestamp, NYP::NClient::TDnsRecordSet&& recordSet);
    TRecordSet(const NYP::NClient::TDnsRecordSet& recordSet);
    TRecordSet(const ui64 ypTimestamp, const NYP::NClient::TDnsRecordSet& recordSet);

    NYP::NClient::TDnsRecordSet MakeYpObject() const;

    const TMaybe<ui64>& YpTimestamp() const {
        return YpTimestamp_;
    }

    void SetYpTimestamp(ui64 timestamp) {
        YpTimestamp_ = timestamp;
    }

    decltype(auto) Meta() const {
        return Proto_.meta();
    }

    decltype(auto) MutableMeta() {
        return Proto_.mutable_meta();
    }

    decltype(auto) Spec() const {
        return Proto_.spec();
    }

    decltype(auto) MutableSpec() {
        return Proto_.mutable_spec();
    }

    bool HasChangelist() const {
        return Changelist_.Defined();
    }

    const TChangelist& Changelist() const {
        return Changelist_.GetRef();
    }

    TChangelist* MutableChangelist() {
        if (!Changelist_.Defined()) {
            Changelist_.ConstructInPlace();
        }
        return Changelist_.Get();
    }

    const TMaybe<TString>& Zone() const {
        return Zone_;
    }

    void SetZone(TString zone) {
        Zone_ = std::move(zone);
    }

    const TMaybe<TString>& ShardId() const {
        return ShardId_;
    }

    void SetShardId(TString shardId) {
        ShardId_ = std::move(shardId);
    }

    TString Serialize() const {
        TString result;
        TStringOutput stream(result);
        Save(&stream);
        return result;
    }

    void Deserialize(const TString& raw) {
        TStringInput stream(raw);
        Load(&stream);
    }

    bool operator==(const TRecordSet& rhs) const {
        return Zone_ == rhs.Zone_ &&
               ShardId_ == rhs.ShardId_ &&
               google::protobuf::util::MessageDifferencer::Equals(Proto_, rhs.Proto_) &&
               Changelist_ == rhs.Changelist_;
    }

    Y_SAVELOAD_DEFINE(Proto_, Zone_, Changelist_, ShardId_, YpTimestamp_);

private:
    NYP::NClient::NApi::NProto::TDnsRecordSet Proto_;
    TMaybe<TString> Zone_;
    TMaybe<TChangelist> Changelist_;
    TMaybe<TString> ShardId_;

    TMaybe<ui64> YpTimestamp_;
};

class TSerializedRecordSet {
public:
    TSerializedRecordSet(const TRecordSet& recordSet)
        : Raw_(recordSet.Serialize())
    {
    }

    TSerializedRecordSet(NYP::NClient::TDnsRecordSet recordSet)
        : TSerializedRecordSet(TRecordSet(std::move(recordSet)))
    {
    }

    TSerializedRecordSet(const NYP::NYPReplica::TDnsRecordSetReplicaObject& object)
        : TSerializedRecordSet(object.GetObject())
    {
    }

    TSerializedRecordSet(const ui64 ypTimestamp, NYP::NClient::TDnsRecordSet recordSet)
        : TSerializedRecordSet(TRecordSet(ypTimestamp, std::move(recordSet)))
    {
    }

    TRecordSet GetObject() const {
        TRecordSet result;
        result.Deserialize(Raw_);
        return result;
    }

private:
    TString Raw_;
};

struct TRecordSetReplica {
    TString Cluster;
    TMaybe<TRecordSet> RecordSet;
};

struct TRecordSetReplicas {
    TString Fqdn;
    TVector<TRecordSetReplica> Replicas;
};

} // namespace NYpDns
