#include "record.h"

#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/serialized_enum.h>

using namespace NYpDns;

Y_UNIT_TEST_SUITE(CreateRecordFromString) {
    Y_UNIT_TEST(ValidateName) {
        UNIT_ASSERT_EQUAL(CreateRecordFromString(". IN AAAA RecordData").Name, DNSName("."));
        UNIT_ASSERT_EQUAL(CreateRecordFromString("yandex.net. IN AAAA RecordData").Name, DNSName("yandex.net."));
        UNIT_ASSERT_EQUAL(CreateRecordFromString("yandex.net. \t \t\t IN AAAA RecordData").Name, DNSName("yandex.net."));
        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString("yandex.net. "), TIncorrectRecordStringException, "Not enough fields in the record");
    }

    Y_UNIT_TEST(ValidateEmptyName) {
        DNSName prevRecordName("yandex.net.");

        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString(" IN AAAA RecordData"), TIncorrectRecordStringException, "Name is empty");
        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString("\tIN AAAA RecordData"), TIncorrectRecordStringException, "Name is empty");
        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString(" \t \tIN AAAA RecordData"), TIncorrectRecordStringException, "Name is empty");
        UNIT_ASSERT_EQUAL(CreateRecordFromString(" IN AAAA RecordData", Nothing(), 0, prevRecordName).Name, prevRecordName);

        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString(""), TIncorrectRecordStringException, "Line is empty");
        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString(" \t \t"), TIncorrectRecordStringException, "Name is empty");
    }

    Y_UNIT_TEST(ValidateNameWithOrigin) {
        TString origin = "yandex.net.";

        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString("@ IN AAAA RecordData"),TIncorrectRecordStringException, "$ORIGIN is missing");
        UNIT_ASSERT_EQUAL(CreateRecordFromString("@ IN AAAA RecordData", origin).Name, DNSName(origin));

        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString("qloud-c IN AAAA RecordData"),TIncorrectRecordStringException, "$ORIGIN is missing");
        UNIT_ASSERT_EQUAL(CreateRecordFromString("qloud-c IN AAAA RecordData", origin).Name, DNSName("qloud-c." + origin));
    }

    Y_UNIT_TEST(ValidateTtl) {
        UNIT_ASSERT_EQUAL(CreateRecordFromString("yandex.net. IN AAAA RecordData").Ttl, 0);
        UNIT_ASSERT_EQUAL(CreateRecordFromString("yandex.net. IN AAAA RecordData", Nothing(), 600).Ttl, 600);
        UNIT_ASSERT_EQUAL(CreateRecordFromString("yandex.net. 600 IN AAAA RecordData").Ttl, 600);
        UNIT_ASSERT_EQUAL(CreateRecordFromString("yandex.net. IN 600 AAAA RecordData").Ttl, 600);
        UNIT_ASSERT_EQUAL(CreateRecordFromString("yandex.net. 600 IN AAAA RecordData", Nothing(), 3600).Ttl, 600);

        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString("yandex.net. -600 IN AAAA RecordData"), TIncorrectRecordStringException, "Unknown field before the type");

        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString("yandex.net. 600 600 IN AAAA RecordData"), TIncorrectRecordStringException, "Ttl was repeated");
        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString("yandex.net. 600 IN 600 AAAA RecordData"), TIncorrectRecordStringException, "Ttl was repeated");
        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString("yandex.net. IN 600 600 AAAA RecordData"), TIncorrectRecordStringException, "Ttl was repeated");
    }

    Y_UNIT_TEST(ValidateClass) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString("yandex.net. 600 AAAA RecordData"), TIncorrectRecordStringException, "The class field is missing");
        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString("yandex.net. IN IN AAAA RecordData"), TIncorrectRecordStringException, "Class was repeated");
        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString("yandex.net. IN 600 IN AAAA RecordData"), TIncorrectRecordStringException, "Class was repeated");
        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString("yandex.net. 600 IN IN AAAA RecordData"), TIncorrectRecordStringException, "Class was repeated");
        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString("yandex.net. IN IN 600 AAAA RecordData"), TIncorrectRecordStringException, "Class was repeated");
        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString("yandex.net. 600 In AAAA RecordData"), TIncorrectRecordStringException, "Unknown field before the type");

        for (const auto class_ : GetEnumAllValues<ERecordClass>()) {
            UNIT_ASSERT_EQUAL(CreateRecordFromString("yandex.net. 600 " + ToString(class_) + " AAAA RecordData").Class, class_);
        }
    }

    Y_UNIT_TEST(ValidateType) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString("yandex.net. 600 IN RecordData"), TIncorrectRecordStringException, "Not enough fields in the record");
        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString("yandex.net. 600 IN SoA RecordData"), TIncorrectRecordStringException, "The type field is missing");
        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString("yandex.net. IN SoA RecordData"), TIncorrectRecordStringException, "Unknown field before the type");

        for (const auto type : GetEnumAllValues<ERecordType>()) {
            UNIT_ASSERT_EQUAL(CreateRecordFromString("yandex.net. 600 IN " + ToString(type) + " RecordData").Type, type);
        }
    }

    Y_UNIT_TEST(ValidateData) {
        UNIT_ASSERT_STRINGS_EQUAL(CreateRecordFromString("yandex.net. IN AAAA \t RecordData1 \t \t RecordData2  \t").Data, "RecordData1 \t \t RecordData2");
        UNIT_ASSERT_EXCEPTION_CONTAINS(CreateRecordFromString("yandex.net. IN AAAA \t \t \t"), TIncorrectRecordStringException, "Not enough fields in the record");
    }

    Y_UNIT_TEST(ValidateParse) {
        {
            auto record = CreateRecordFromString("yandex.net. \t 600\tIN   SOA\tRecordData1 \tRecordData2 ");
            UNIT_ASSERT_EQUAL(record.Name, DNSName("yandex.net."));
            UNIT_ASSERT_EQUAL(record.Ttl, 600);
            UNIT_ASSERT_EQUAL(record.Class, ERecordClass::IN);
            UNIT_ASSERT_EQUAL(record.Type, ERecordType::SOA);
            UNIT_ASSERT_STRINGS_EQUAL(record.Data, "RecordData1 \tRecordData2");
        }
        {
            auto record = CreateRecordFromString("ext-root-i.qloud-c.yandex.net. IN AAAA 2a02:6b8:d01:201:0:1478:e9d6:686f");
            UNIT_ASSERT_EQUAL(record.Name, DNSName("ext-root-i.qloud-c.yandex.net."));
            UNIT_ASSERT_EQUAL(record.Ttl, 0);
            UNIT_ASSERT_EQUAL(record.Class, ERecordClass::IN);
            UNIT_ASSERT_EQUAL(record.Type, ERecordType::AAAA);
            UNIT_ASSERT_STRINGS_EQUAL(record.Data, "2a02:6b8:d01:201:0:1478:e9d6:686f");
        }
        {
            auto record = CreateRecordFromString("0.0.0.0.0.0.0.0.3.f.9.d.b.0.1.0.3.9.0.0.b.0.c.0.8.b.6.0.2.0.a.2.ip6.arpa. IN PTR man1-3247-man-abacaba-8-0.gencfg-c.yandex.net");
            UNIT_ASSERT_EQUAL(record.Name, DNSName("0.0.0.0.0.0.0.0.3.f.9.d.b.0.1.0.3.9.0.0.b.0.c.0.8.b.6.0.2.0.a.2.ip6.arpa."));
            UNIT_ASSERT_EQUAL(record.Ttl, 0);
            UNIT_ASSERT_EQUAL(record.Class, ERecordClass::IN);
            UNIT_ASSERT_EQUAL(record.Type, ERecordType::PTR);
            UNIT_ASSERT_STRINGS_EQUAL(record.Data, "man1-3247-man-abacaba-8-0.gencfg-c.yandex.net");
        }
    }
}
