#include "apply_request.h"

#include <util/generic/maybe.h>

namespace NYpDns {

using NYP::NClient::NApi::NProto::TDnsRecordSetSpec;

////////////////////////////////////////////////////////////////////////////////

void RemoveElement(NProtoBuf::RepeatedPtrField<TDnsRecordSetSpec::TResourceRecord>& records, int idx) {
    Y_ENSURE(0 <= idx && idx < records.size());
    records.SwapElements(idx, records.size() - 1);
    records.RemoveLast();
}

////////////////////////////////////////////////////////////////////////////////

bool ChangeMatchesRecord(const NProto::TRecordUpdateRequest& updateRequest, const TDnsRecordSetSpec::TResourceRecord& record) {
    TDnsRecordSetSpec::TResourceRecord::EType updateRecordType = static_cast<TDnsRecordSetSpec::TResourceRecord::EType>(
            updateRequest.content().type());

    if (updateRecordType != record.type()) {
        return false;
    }

    switch (updateRecordType) {
        case TDnsRecordSetSpec::TResourceRecord::SOA:
            return true;
        default:
            return record.data() == updateRequest.content().data();
    }
}

////////////////////////////////////////////////////////////////////////////////

bool AddChangelistEntry(TMaybe<TRecordSet>& recordSet, TChange change, bool incrementVersion) {
    Y_ENSURE(recordSet.Defined());
    *recordSet->MutableChangelist()->add_changes() = std::move(change);
    if (incrementVersion) {
        recordSet->MutableChangelist()->set_version(recordSet->Changelist().version() + 1);
    }
    return true;
}

////////////////////////////////////////////////////////////////////////////////

bool ApplyUpdateRequest(
    TMaybe<TRecordSet>& recordSet,
    const TChange& change,
    bool writeToChangelist,
    const TRecordChangeCallback& beforeUpdateCallback,
    const TRecordChangeCallback& afterUpdateCallback,
    const TNoArgCallback& notFoundCallback
) {
    Y_ENSURE(recordSet.Defined());

    const NProto::TRecordUpdateRequest& recordUpdateRequest = change.record_update_request();
    TDnsRecordSetSpec::TResourceRecord::EType recordType = static_cast<TDnsRecordSetSpec::TResourceRecord::EType>(recordUpdateRequest.content().type());

    bool recordExists = false;
    bool recordSetChanged = false;
    int cnamesRecordsCount = 0;
    for (TDnsRecordSetSpec::TResourceRecord& record : *recordSet->MutableSpec()->mutable_records()) {
        if (record.type() == TDnsRecordSetSpec::TResourceRecord::CNAME) {
            ++cnamesRecordsCount;
        }

        if (ChangeMatchesRecord(recordUpdateRequest, record)) {
            recordExists = true;

            if (beforeUpdateCallback) {
                beforeUpdateCallback(record);
            }

            if (recordUpdateRequest.content().ttl() && record.ttl() != recordUpdateRequest.content().ttl()) {
                record.set_ttl(recordUpdateRequest.content().ttl());
                recordSetChanged = true;
            }

            if (!recordUpdateRequest.content().class_().empty() && record.class_() != recordUpdateRequest.content().class_()) {
                record.set_class_(recordUpdateRequest.content().class_());
                recordSetChanged = true;
            }

            if (!recordUpdateRequest.content().data().empty() && record.data() != recordUpdateRequest.content().data()) {
                record.set_data(recordUpdateRequest.content().data());
                recordSetChanged = true;
            }

            if (afterUpdateCallback) {
                afterUpdateCallback(record);
            }
        }
    }

    if (!recordExists) {
        // Handle CNAMEs as it stated in DISCOVERY-120
        if (recordType == TDnsRecordSetSpec::TResourceRecord::CNAME) {
            recordSet->MutableSpec()->clear_records();
        } else if (cnamesRecordsCount > 0) {
            for (int i = 0, cnamesRemoved = 0; i < recordSet->Spec().records().size() && cnamesRemoved < cnamesRecordsCount; ++i) {
                const TDnsRecordSetSpec::TResourceRecord& record = recordSet->Spec().records(i);
                if (record.type() == TDnsRecordSetSpec::TResourceRecord::CNAME) {
                    RemoveElement(*recordSet->MutableSpec()->mutable_records(), i);
                    --i;
                    ++cnamesRemoved;
                }
            }
        }

        TDnsRecordSetSpec::TResourceRecord& record = *recordSet->MutableSpec()->add_records();

        if (notFoundCallback) {
            notFoundCallback();
        }

        record.set_ttl(recordUpdateRequest.content().ttl());
        const TString& recordClass = !recordUpdateRequest.content().class_().empty() ? recordUpdateRequest.content().class_() : "IN";
        record.set_class_(recordClass);
        record.set_type(recordType);
        record.set_data(recordUpdateRequest.content().data());
        recordSetChanged = true;

        if (afterUpdateCallback) {
            afterUpdateCallback(record);
        }
    }

    if (writeToChangelist) {
        recordSetChanged |= AddChangelistEntry(recordSet, change, /* incrementVersion */ true);
    }

    return recordSetChanged;
}

bool ApplyUpdateRequest(TMaybe<TRecordSet>& recordSet, const TChange& change, bool writeToChangelist) {
    return ApplyUpdateRequest(recordSet, change, writeToChangelist, {}, {}, {});
}

////////////////////////////////////////////////////////////////////////////////

bool ApplyRemoveRequest(TMaybe<TRecordSet>& recordSet, const TChange& change, bool writeToChangelist, const TRecordChangeCallback& removeRecordCallback) {
    Y_ENSURE(recordSet.Defined());

    const NProto::TRecordUpdateRequest& recordUpdateRequest = change.record_update_request();

    bool recordSetChanged = false;
    for (int i = 0; i < recordSet->Spec().records().size(); ++i) {
        const TDnsRecordSetSpec::TResourceRecord& record = recordSet->Spec().records(i);

        if (ChangeMatchesRecord(recordUpdateRequest, record)) {
            if (removeRecordCallback) {
                removeRecordCallback(record);
            }

            RemoveElement(*recordSet->MutableSpec()->mutable_records(), i);
            --i;
            recordSetChanged = true;
        }
    }

    if (writeToChangelist) {
        recordSetChanged |= AddChangelistEntry(recordSet, change, /* incrementVersion */ true);
    }

    return recordSetChanged;
}

bool ApplyRemoveRequest(TMaybe<TRecordSet>& recordSet, const TChange& change, bool writeToChangelist) {
    return ApplyRemoveRequest(recordSet, change, writeToChangelist, {});
}

////////////////////////////////////////////////////////////////////////////////

bool ApplyRequest(TMaybe<TRecordSet>& recordSet, const TChange& change, bool writeToChangelist) {
    switch (change.record_update_request().type()) {
        case NProto::TRecordUpdateRequest::UPDATE:
            return ApplyUpdateRequest(recordSet, change, writeToChangelist);
        case NProto::TRecordUpdateRequest::REMOVE:
            return ApplyRemoveRequest(recordSet, change, writeToChangelist);
        default:
            return false;
    }
}

////////////////////////////////////////////////////////////////////////////////

} // namespace NYpDns
