#include "merge.h"

#include "apply_request.h"

#include <util/generic/algorithm.h>
#include <util/generic/guid.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/map.h>
#include <util/generic/maybe.h>
#include <util/generic/set.h>
#include <util/generic/vector.h>

namespace NYpDns {

namespace {

using TRequestKey = std::pair<TString, ui32>;
using TRequestValue = std::pair<ui64, TString>;

TRequestKey GetRequestKey(const TChange& change) {
    return {change.record_update_request().content().data(), change.record_update_request().content().type()};
}

TRequestValue GetRequestValue(const TChange& change) {
    return {change.timestamp(), change.uuid()};
}

TClassifiedChanges ClassifyChanges(const TVector<TVector<TChange>>& changesList) {
    THashSet<TString> replicatedOnce; // uuids of changes, marked as replicated at least in one cluster
    THashSet<TString> notReplicatedOnce; // uuids of changes, marked as not replicated at least in one cluster
    THashMap<TString, size_t> occuranceCounter;
    THashMap<TString, TChange> changes;
    THashMap<TRequestKey, TRequestValue> latestRequest;
    for (const auto& singleClusterChangelist : changesList) {
        THashSet<TString> existingRequests;
        for (const auto& change : singleClusterChangelist) {
            if (change.replicated()) {
                replicatedOnce.insert(change.uuid());
            } else {
                notReplicatedOnce.insert(change.uuid());
            }

            if (auto [it, notExisted] = latestRequest.emplace(GetRequestKey(change), GetRequestValue(change)); !notExisted) {
                if (TRequestValue candidate = GetRequestValue(change); candidate > it->second) {
                    it->second = candidate;
                }
            }

            changes.emplace(change.uuid(), change);
            size_t& currentRequestCounter = occuranceCounter[change.uuid()];
            ++currentRequestCounter;
            auto [ignore, notExisted] = existingRequests.emplace(change.uuid());
            currentRequestCounter -= !notExisted;
        }
    }

    TClassifiedChanges result;
    auto& [toBeApplied, toSetReplicated, toBeRemoved] = result;
    for (auto& [uuid, change] : changes) {
        if (!notReplicatedOnce.count(uuid)) {
            toBeRemoved.emplace_back(std::move(change));
        } else if (replicatedOnce.count(uuid) ||
                   occuranceCounter[uuid] == changesList.size() ||
                   latestRequest[GetRequestKey(change)] != GetRequestValue(change))
        {
            toSetReplicated.emplace_back(std::move(change));
        } else {
            toBeApplied.emplace_back(std::move(change));
        }
    }
    return result;
}

THashMap<TString, TMaybe<TRecordSet>>::const_iterator FindRecordSet(
    const THashMap<TString, TMaybe<TRecordSet>>& recordSets,
    std::function<bool(const TRecordSet&, const TRecordSet&)> less,
    std::function<bool(const TRecordSet&)> predicate
) {
    auto result = recordSets.cend();
    for (auto it = recordSets.cbegin(); it != recordSets.cend(); ++it) {
        const auto& [cluster, recordSet] = *it;
        if (recordSet.Defined() && (!predicate || predicate(*recordSet))) {
            if (result == recordSets.cend() || less(*result->second, *recordSet)) {
                result = it;
            }
        }
    }
    return result;
}

TBaseVersions CreateBaseVersionsMap(const THashMap<TString, TMaybe<TRecordSet>>& recordSets, const TString& recordSetHash) {
    TBaseVersions result;
    for (const auto& [cluster, recordSet] : recordSets) {
        if (recordSet.Defined() && recordSet->HasChangelist() && recordSet->Changelist().MatchHashExact(recordSetHash)) {
            result[cluster] = recordSet->Changelist().version();
        }
    }
    return result;
}

THashMap<TString, TMaybe<TRecordSet>> GetTargetRecordSets(const THashMap<TString, TMaybe<TRecordSet>>& baseRecordSets, const TMaybe<TRecordSet>& targetRecordSet) {
    Y_ENSURE(!targetRecordSet.Defined() || targetRecordSet->HasChangelist());

    THashMap<TString, TMaybe<TRecordSet>> result;
    result.reserve(baseRecordSets.size());
    for (const auto& [cluster, baseRecordSet] : baseRecordSets) {
        auto& clusterTargetRecordSet = result[cluster] = targetRecordSet;
        if (!clusterTargetRecordSet.Defined()) {
            continue;
        }

        if (!baseRecordSet.Defined() || !baseRecordSet->HasChangelist() ||
            clusterTargetRecordSet->Changelist().record_set_hash() != baseRecordSet->Changelist().record_set_hash())
        {
            clusterTargetRecordSet->MutableChangelist()->set_version(0);
        } else {
            clusterTargetRecordSet->MutableChangelist()->set_version(baseRecordSet->Changelist().version());
        }
    }
    return result;
}

} // anonymous namespace

TMergeOptions::TMergeOptions(const TMergeConfig& config)
    : FormChangelist(config.GetFormChangelist())
    , MergeAcls(config.GetMergeAcls())
{
}

TMaybe<TRecordSet> MergeRecords(const THashMap<TString, TMaybe<TRecordSet>>& recordSets, const TMergeOptions& options) {
    TString newestRecordSetHash;
    if (auto it = FindRecordSet(recordSets,
                                [](const TRecordSet& lhs, const TRecordSet& rhs) {
                                        return lhs.YpTimestamp() < rhs.YpTimestamp();
                                },
                                [](const TRecordSet& recordSet) {
                                    return recordSet.HasChangelist() && !recordSet.Changelist().record_set_hash().empty();
                                }); it != recordSets.cend())
    {
        newestRecordSetHash = it->second->Changelist().record_set_hash();
    }

    TMaybe<TRecordSet> result;
    TString baseCluster;
    if (auto it = FindRecordSet(recordSets,
                                [](const TRecordSet& lhs, const TRecordSet& rhs) {
                                    if (!lhs.HasChangelist() || !rhs.HasChangelist()) {
                                        return lhs.HasChangelist() < rhs.HasChangelist();
                                    }
                                    return *lhs.Changelist().BaseVersions() < *rhs.Changelist().BaseVersions();
                                },
                                [&newestRecordSetHash](const TRecordSet& recordSet) {
                                    return (!recordSet.HasChangelist() && newestRecordSetHash.empty()) || (recordSet.HasChangelist() && recordSet.Changelist().MatchHashExact(newestRecordSetHash));
                                }); it != recordSets.cend())
    {
        baseCluster = it->first;
        result.ConstructInPlace(*it->second);
    }

    if (!result.Defined()) {
        return Nothing();
    }

    TVector<TVector<TChange>> changesList;
    const TBaseVersions* baseVersions = result->HasChangelist() ? result->Changelist().BaseVersions() : nullptr;
    for (const auto& [cluster, recordSet] : recordSets) {
        TVector<TChange>& changes = changesList.emplace_back();
        if (recordSet.Defined() && recordSet->HasChangelist() && recordSet->Changelist().MatchHash(newestRecordSetHash)) {
            if (baseVersions && cluster != baseCluster && recordSet->Changelist().MatchHashExact(newestRecordSetHash) && baseVersions->Get(cluster) > recordSet->Changelist().version()) {
                continue;
            }

            for (const auto& change : recordSet->Changelist().changes()) {
                changes.emplace_back(change);
            }
        }
    }

    auto [toBeApplied, toSetReplicated, toBeRemoved] = ClassifyChanges(changesList);
    SortBy(toBeApplied, [](const TChange& change) {
        return change.timestamp();
    });

    for (const auto& change : toBeApplied) {
        ApplyRequest(result, change, /* formChangelist */ false);
    }

    if (options.FormChangelist) {
        result->MutableChangelist()->clear_changes();

        for (auto& change : toSetReplicated) {
            change.set_replicated(true);
            AddChangelistEntry(result, change);
        }

        for (const auto& change : toBeApplied) {
            AddChangelistEntry(result, change);
        }

        if (result->Spec().records().empty() && (!result->HasChangelist() || result->Changelist().changes().empty())) {
            return Nothing();
        }

        if (!result->HasChangelist() || result->Changelist().record_set_hash().empty()) {
            result->MutableChangelist()->set_record_set_hash(CreateGuidAsString());
        }

        *result->MutableChangelist()->MutableBaseVersions() = CreateBaseVersionsMap(recordSets, result->Changelist().record_set_hash());
    } else {
        if (result->Spec().records().empty()) {
            return Nothing();
        }
    }

    return result;
}

void MergeAcls(const THashMap<TString, TMaybe<TRecordSet>>& recordSets, TMaybe<TRecordSet>& result) {
    using namespace NYP::NClient::NApi::NProto;
    using TAttribute = TString;

    if (!result) {
        return;
    }

    TMap<EAccessControlAction, TMap<std::pair<EAccessControlPermission, TAttribute>, TSet<TString>>> accessControlMapByAction;
    for (const auto& [cluster, recordSet] : recordSets) {
        if (!recordSet) {
            continue;
        }
        for (const TAccessControlEntry& ace : recordSet->Meta().acl()) {
            for (const int permission : ace.permissions()) {
                if (!ace.attributes().empty()) {
                    for (const TString& attribute : ace.attributes()) {
                        auto& resultSubjects = accessControlMapByAction[ace.action()][{static_cast<EAccessControlPermission>(permission), attribute}];
                        resultSubjects.insert(ace.subjects().begin(), ace.subjects().end());
                    }
                } else {
                    accessControlMapByAction[ace.action()][{static_cast<EAccessControlPermission>(permission), TString{}}].insert(ace.subjects().begin(), ace.subjects().end());
                }
            }
        }
    }

    result->MutableMeta()->mutable_acl()->Clear();
    for (const auto& [action, accessControlMap] : accessControlMapByAction) {
        TMap<std::tuple<TAttribute, TSet<TString>>, TVector<EAccessControlPermission>> accessParamsBySubjects;
        for (const auto& [accessParams, subjects] : accessControlMap) {
            const auto& [permission, attribute] = accessParams;
            accessParamsBySubjects[{attribute, subjects}].push_back(permission);
        }

        for (const auto& [attributeAndSubjects, permissions] : accessParamsBySubjects) {
            const auto& [attribute, subjects] = attributeAndSubjects;

            TAccessControlEntry* ace = result->MutableMeta()->add_acl();

            ace->set_action(action);

            ace->mutable_permissions()->Reserve(permissions.size());
            for (const EAccessControlPermission permission : permissions) {
                ace->add_permissions(permission);
            }

            ace->mutable_subjects()->Reserve(subjects.size());
            for (const TString& subject : subjects) {
                ace->add_subjects(subject);
            }

            if (attribute) {
                ace->add_attributes(attribute);
            }
        }
    }

    bool hasAnyoneWriteAccess = false;
    for (const auto& ace : result->Meta().acl()) {
        if (ace.action() == EAccessControlAction::ACA_ALLOW &&
            ace.attributes().empty() &&
            !ace.subjects().empty() &&
            IsIn(ace.permissions(), static_cast<int>(EAccessControlPermission::ACA_WRITE))
        ) {
            hasAnyoneWriteAccess = true;
            break;
        }
    }
    Y_ENSURE(hasAnyoneWriteAccess);
}

TMaybe<TRecordSet> MergeInOne(const THashMap<TString, TMaybe<TRecordSet>>& recordSets, const TMergeOptions& options) {
    TMaybe<TRecordSet> result = MergeRecords(recordSets, options);
    if (options.MergeAcls) {
        MergeAcls(recordSets, result);
    }
    return result;
}

TMaybe<TRecordSet> MergeInOne(TVector<TRecordSetReplica> recordSets, const TMergeOptions& options) {
    THashMap<TString, TMaybe<TRecordSet>> recordSetsMap(recordSets.size());
    for (TRecordSetReplica& replica : recordSets) {
        recordSetsMap.emplace(std::move(replica.Cluster), std::move(replica.RecordSet));
    }
    return MergeInOne(recordSetsMap, options);
}

THashMap<TString, TMaybe<TRecordSet>> ReplicateChanges(const THashMap<TString, TMaybe<TRecordSet>>& recordSets) {
    TMergeOptions mergeOptions;
    mergeOptions.SetFormChangelist(true)
                .SetMergeAcls(true);

    TMaybe<TRecordSet> targetRecordSet = MergeInOne(recordSets, mergeOptions);
    return GetTargetRecordSets(recordSets, targetRecordSet);
}

} // namespace NYpDns

