#include "zone.h"

#include <library/cpp/protobuf/yt/proto2yt.h>
#include <library/cpp/protobuf/yt/yt2proto.h>

namespace NYpDns {

namespace {

TZoneConfig YtNodeToConfig(const NYT::TNode& configNode, const TParseConfig& parseConfig) {
    TZoneConfig result;
    YtNodeToProto(configNode, result, parseConfig);
    return result;
}

TZoneConfig YpObjectToConfig(const NYP::NClient::TDnsZone& ypObject) {
    TZoneConfig result;
    result.SetName(ypObject.Meta().id());

    const auto& config = ypObject.Spec().config();
    if (config.has_default_soa_record() && config.default_soa_record().has_soa()) {
        if (config.default_soa_record().soa().has_primary_nameserver()) {
            result.SetPrimaryNameserver(
                config.default_soa_record().soa().primary_nameserver());
        }
        if (config.default_soa_record().soa().has_email()) {
            result.SetHostmaster(config.default_soa_record().soa().email());
        }
        if (config.default_soa_record().has_ttl()) {
            result.SetSOARecordTtl(config.default_soa_record().ttl());
        }
        if (config.default_soa_record().soa().has_serial()) {
            result.SetSOASerial(config.default_soa_record().soa().serial());
        }
        if (config.default_soa_record().soa().has_negative_ttl()) {
            result.SetSOARecordMinimumTtl(config.default_soa_record().soa().negative_ttl());
        }
        if (config.default_soa_record().soa().has_refresh()) {
            result.SetSOARefresh(config.default_soa_record().soa().refresh());
        }
        if (config.default_soa_record().soa().has_expire()) {
            result.SetSOAExpire(config.default_soa_record().soa().expire());
        }
        if (config.default_soa_record().soa().has_retry()) {
            result.SetSOARetry(config.default_soa_record().soa().retry());
        }
    }

    for (const auto& defaultNSRecord : config.default_ns_records()) {
        if (!defaultNSRecord.has_ns()) {
            continue;
        }
        if (defaultNSRecord.ns().has_nameserver()) {
            result.AddNameservers(defaultNSRecord.ns().nameserver());
        }
        if (defaultNSRecord.has_ttl()) {
            result.SetNSRecordTtl(defaultNSRecord.ttl());
        }
    }

    if (config.has_store_config()) {
        for (const TString& ypCluster : config.store_config().clusters()) {
            result.AddYPClusters(ypCluster);
        }

        if (config.store_config().has_algorithm()) {
            ESelectRecordSetMode mode;
            ESelectRecordSetMode_Parse(
                to_upper(config.store_config().algorithm()),
                &mode);
            result.SetSelectRecordSetMode(mode);
        }
    }

    if (config.has_default_records_ttl()) {
        result.SetDefaultTtl(config.default_records_ttl());
    }

    if (config.has_response_policy()) {
        using TResponsePolicy = NYP::NClient::NApi::NProto::TDnsZoneSpec::TDnsZoneConfig::TResponsePolicy;
        auto convertResponsePolicy = [](const TResponsePolicy::TOptions& policy) {
            TTypeResponsePolicy result;
            if (policy.has_order()) {
                switch (policy.order()) {
                    case TResponsePolicy::TOptions::SEQUENTIAL:
                        result.SetOrder(TTypeResponsePolicy::SEQUENTIAL);
                        break;
                    case TResponsePolicy::TOptions::RANDOM:
                        result.SetOrder(TTypeResponsePolicy::RANDOM);
                        break;
                    default:
                        break;
                }
            }
            result.SetMaxRecordsNumber(
                policy.has_max_records_number()
                    ? static_cast<i32>(policy.max_records_number())
                    : -1);
            return result;
        };

        const auto& rp = config.response_policy();

        if (rp.has_a()) {
            *result.MutableResponsePolicy()->MutableA() = convertResponsePolicy(rp.a());
        }
        if (rp.has_ns()) {
            *result.MutableResponsePolicy()->MutableNS() = convertResponsePolicy(rp.ns());
        }
        if (rp.has_cname()) {
            *result.MutableResponsePolicy()->MutableCNAME() = convertResponsePolicy(rp.cname());
        }
        if (rp.has_soa()) {
            *result.MutableResponsePolicy()->MutableSOA() = convertResponsePolicy(rp.soa());
        }
        if (rp.has_ptr()) {
            *result.MutableResponsePolicy()->MutablePTR() = convertResponsePolicy(rp.ptr());
        }
        if (rp.has_txt()) {
            *result.MutableResponsePolicy()->MutableTXT() = convertResponsePolicy(rp.txt());
        }
        if (rp.has_aaaa()) {
            *result.MutableResponsePolicy()->MutableAAAA() = convertResponsePolicy(rp.aaaa());
        }
        if (rp.has_srv()) {
            *result.MutableResponsePolicy()->MutableSRV() = convertResponsePolicy(rp.srv());
        }
    }

    if (config.has_backend_store_policy()) {
        using TBackendStoreRecordsPolicy = NYP::NClient::NApi::NProto::TDnsZoneSpec::TDnsZoneConfig::TBackendStoreRecordsPolicy;
        auto convertBackendStorePolicy = [](const TBackendStoreRecordsPolicy::TOptions& policy) {
            TFilterStoredRecordsPolicy result;
            if (policy.has_order()) {
                switch (policy.order()) {
                    case TBackendStoreRecordsPolicy::TOptions::ORIGINAL:
                        result.SetOrder(TFilterStoredRecordsPolicy::ORIGINAL);
                        break;
                    case TBackendStoreRecordsPolicy::TOptions::RANDOM:
                        result.SetOrder(TFilterStoredRecordsPolicy::RANDOM);
                        break;
                    default:
                        break;
                }
            }
            result.SetMaxRecordsNumber(
                policy.has_max_records_number()
                    ? static_cast<i32>(policy.max_records_number())
                    : -1);
            return result;
        };

        const auto& bsp = config.backend_store_policy();

        if (bsp.has_default_()) {
            *result.MutableFilterStoredRecordsPolicy() =
                convertBackendStorePolicy(bsp.default_());
        }
    }

    if (config.has_transfer_config()) {
        if (config.transfer_config().has_enabled()) {
            result.MutableTransferConfig()->SetEnabled(config.transfer_config().enabled());
        }
        *result.MutableTransferConfig()->MutableAxfrAllowList() =
            config.transfer_config().axfr_allow_list();
        *result.MutableTransferConfig()->MutableNotifyAddresses() =
            config.transfer_config().notify_addresses();
    }

    if (config.has_serial_generate_mode()) {
        using EYpProtoDnsZoneConfig = NYP::NClient::NApi::NProto::TDnsZoneSpec::TDnsZoneConfig;
        switch (config.serial_generate_mode()) {
            case EYpProtoDnsZoneConfig::ORIGINAL:
                result.SetSerialGenerateMode(ESerialNumberGenerateMode::ORIGINAL);
                break;
            case EYpProtoDnsZoneConfig::YP_TIMESTAMP_BASED:
                result.SetSerialGenerateMode(ESerialNumberGenerateMode::YP_TIMESTAMP_BASED);
                break;
            default:
                break;
        }
    }

    for (const NJson::TJsonValue& owner : ypObject.Labels()["owners"].GetArray()) {
        if (owner.IsString()) {
            result.AddOwners(owner.GetString());
        }
    }

    return result;
}

} // anoonymous namespace

TParseConfig TZone::YSON_PARSE_CONFIG = TParseConfig()
    .SetSkipEmptyOptionalFields(true);

TZone::TZone(TZoneConfig config)
    : Config_(std::move(config))
    , Name_(Config_.GetName())
{
}

TZone::TZone(const NYT::TNode& config)
    : TZone(YtNodeToConfig(config, YSON_PARSE_CONFIG))
{
}

TZone::TZone(const NYP::NClient::TDnsZone& ypObject)
    : TZone(YpObjectToConfig(ypObject))
{
}

const TZoneId& TZone::GetId() const {
    return Config_.GetName();
}

const TString& TZone::GetName() const {
    return Config_.GetName();
}

const TDnsNameView& TZone::GetDnsName() const {
    return Name_;
}

const TZoneConfig& TZone::Config() const {
    return Config_;
}

TZoneConfig* TZone::MutableConfig() {
    return &Config_;
}

NYT::TNode TZone::ConfigToYtNode() const {
    return ProtoToYtNode(Config_, YSON_PARSE_CONFIG);
}

NYP::NClient::TDnsZone TZone::ConfigToYpDnsZoneObject() const {
    NYP::NClient::TDnsZone result;
    result.MutableMeta()->set_id(GetName());

    auto& config = *result.MutableSpec()->mutable_config();
    if (Config_.HasDefaultTtl()) {
        config.set_default_records_ttl(Config_.GetDefaultTtl());
    }

    auto& defaultSOARecord = *config.mutable_default_soa_record();
    if (Config_.HasSOARecordTtl()) {
        defaultSOARecord.set_ttl(Config_.GetSOARecordTtl());
    }
    defaultSOARecord.set_class_("IN");
    if (Config_.HasPrimaryNameserver()) {
        defaultSOARecord.mutable_soa()->set_primary_nameserver(Config_.GetPrimaryNameserver());
    }
    if (Config_.HasHostmaster()) {
        defaultSOARecord.mutable_soa()->set_email(Config_.GetHostmaster());
    }
    if (Config_.HasSOASerial()) {
        defaultSOARecord.mutable_soa()->set_serial(Config_.GetSOASerial());
    }
    if (Config_.HasSOARefresh()) {
        defaultSOARecord.mutable_soa()->set_refresh(Config_.GetSOARefresh());
    }
    if (Config_.HasSOARetry()) {
        defaultSOARecord.mutable_soa()->set_retry(Config_.GetSOARetry());
    }
    if (Config_.HasSOAExpire()) {
        defaultSOARecord.mutable_soa()->set_expire(Config_.GetSOAExpire());
    }
    if (Config_.HasSOARecordMinimumTtl()) {
        defaultSOARecord.mutable_soa()->set_negative_ttl(Config_.GetSOARecordMinimumTtl());
    }

    for (const TString& nameserver : Config_.GetNameservers()) {
        auto& defaultNSRecord = *config.add_default_ns_records();
        if (Config_.HasNSRecordTtl()) {
            defaultNSRecord.set_ttl(Config_.GetNSRecordTtl());
        }
        defaultNSRecord.set_class_("IN");
        defaultNSRecord.mutable_ns()->set_nameserver(nameserver);
    }

    for (const TString& ypCluster : Config_.GetYPClusters()) {
        config.mutable_store_config()->add_clusters(ypCluster);
    }
    config.mutable_store_config()->set_algorithm(
        to_lower(NYpDns::ESelectRecordSetMode_Name(Config_.GetSelectRecordSetMode())));

    if (Config_.HasResponsePolicy()) {
        auto convertResponsePolicy = [](const TTypeResponsePolicy& policy) {
            using TResponsePolicy = NYP::NClient::NApi::NProto::TDnsZoneSpec::TDnsZoneConfig::TResponsePolicy;
            TResponsePolicy::TOptions result;
            if (policy.HasOrder()) {
                switch (policy.GetOrder()) {
                    case TTypeResponsePolicy::SEQUENTIAL:
                        result.set_order(TResponsePolicy::TOptions::SEQUENTIAL);
                        break;
                    case TTypeResponsePolicy::RANDOM:
                        result.set_order(TResponsePolicy::TOptions::RANDOM);
                        break;
                    default:
                        break;
                }
            }
            if (policy.HasMaxRecordsNumber() && policy.GetMaxRecordsNumber() != -1) {
                result.set_max_records_number(static_cast<ui32>(policy.GetMaxRecordsNumber()));
            }
            return result;
        };

        if (Config_.GetResponsePolicy().HasA()) {
            *config.mutable_response_policy()->mutable_a() =
                convertResponsePolicy(Config_.GetResponsePolicy().GetA());
        }
        if (Config_.GetResponsePolicy().HasNS()) {
            *config.mutable_response_policy()->mutable_ns() =
                convertResponsePolicy(Config_.GetResponsePolicy().GetNS());
        }
        if (Config_.GetResponsePolicy().HasCNAME()) {
            *config.mutable_response_policy()->mutable_cname() =
                convertResponsePolicy(Config_.GetResponsePolicy().GetCNAME());
        }
        if (Config_.GetResponsePolicy().HasSOA()) {
            *config.mutable_response_policy()->mutable_soa() =
                convertResponsePolicy(Config_.GetResponsePolicy().GetSOA());
        }
        if (Config_.GetResponsePolicy().HasPTR()) {
            *config.mutable_response_policy()->mutable_ptr() =
                convertResponsePolicy(Config_.GetResponsePolicy().GetPTR());
        }
        if (Config_.GetResponsePolicy().HasTXT()) {
            *config.mutable_response_policy()->mutable_txt() =
                convertResponsePolicy(Config_.GetResponsePolicy().GetTXT());
        }
        if (Config_.GetResponsePolicy().HasAAAA()) {
            *config.mutable_response_policy()->mutable_aaaa() =
                convertResponsePolicy(Config_.GetResponsePolicy().GetAAAA());
        }
        if (Config_.GetResponsePolicy().HasSRV()) {
            *config.mutable_response_policy()->mutable_srv() =
                convertResponsePolicy(Config_.GetResponsePolicy().GetSRV());
        }
    }

    if (Config_.HasFilterStoredRecordsPolicy()) {
        auto convertBackendStorePolicy = [](const TFilterStoredRecordsPolicy& policy) {
            using TBackendStoreRecordsPolicy = NYP::NClient::NApi::NProto::TDnsZoneSpec::TDnsZoneConfig::TBackendStoreRecordsPolicy;
            TBackendStoreRecordsPolicy::TOptions result;
            if (policy.HasOrder()) {
                switch (policy.GetOrder()) {
                    case TFilterStoredRecordsPolicy::ORIGINAL:
                        result.set_order(TBackendStoreRecordsPolicy::TOptions::ORIGINAL);
                        break;
                    case TFilterStoredRecordsPolicy::RANDOM:
                        result.set_order(TBackendStoreRecordsPolicy::TOptions::RANDOM);
                        break;
                    default:
                        break;
                }
            }
            if (policy.HasMaxRecordsNumber() && policy.GetMaxRecordsNumber() != -1) {
                result.set_max_records_number(static_cast<ui32>(policy.GetMaxRecordsNumber()));
            }
            return result;
        };

        *config.mutable_backend_store_policy()->mutable_default_() =
            convertBackendStorePolicy(Config_.GetFilterStoredRecordsPolicy());
    }

    if (Config_.HasTransferConfig()) {
        config.mutable_transfer_config()->set_enabled(
            Config_.GetTransferConfig().GetEnabled());

        *config.mutable_transfer_config()->mutable_axfr_allow_list() =
            Config_.GetTransferConfig().GetAxfrAllowList();

        *config.mutable_transfer_config()->mutable_notify_addresses() =
            Config_.GetTransferConfig().GetNotifyAddresses();
    }

    if (Config_.HasSerialGenerateMode()) {
        using EYpProtoDnsZoneConfig = NYP::NClient::NApi::NProto::TDnsZoneSpec::TDnsZoneConfig;
        switch (Config_.GetSerialGenerateMode()) {
            case ESerialNumberGenerateMode::ORIGINAL:
                config.set_serial_generate_mode(EYpProtoDnsZoneConfig::ORIGINAL);
                break;
            case ESerialNumberGenerateMode::YP_TIMESTAMP_BASED:
                config.set_serial_generate_mode(EYpProtoDnsZoneConfig::YP_TIMESTAMP_BASED);
                break;
            default:
                break;
        }
    }

    for (const TString& owner : Config_.GetOwners()) {
        (*result.MutableLabels())["owners"].AppendValue(owner);
    }

    return result;
}

} // namespace NYpDns
