#include "yp_replica.h"

#include <infra/libs/yp_dns/record_set/record.h>
#include <infra/libs/yp_dns/record_set/record_set.h>
#include <infra/libs/yp_dns/replication/iterate.h>
#include <infra/libs/yp_dns/replication/merge.h>

#include <library/cpp/threading/future/async.h>

namespace NYpDns {

////////////////////////////////////////////////////////////////////////////////

using NYP::NClient::NApi::NProto::TDnsRecordSetSpec;

////////////////////////////////////////////////////////////////////////////////

TListRecordSetsFromReplicaResult ListRecordSetsFromReplica(
    const TYpReplica& replica,
    TFromReplicaListOptions listOptions
) {
    TVector<std::pair<TString, TVector<TYpReplicaStorageElement>>> elements = replica.ListElements<TYpReplicaRecordSetObject>(listOptions);

    TListRecordSetsFromReplicaResult result;
    result.YpTimestamp = *replica.GetYpTimestamp(listOptions.Snapshot);
    result.RecordSets.reserve(elements.size());
    for (auto&& [key, values] : elements) {
        Y_ENSURE(values.size() == 1,
            "Number of record sets by key " << key << " is " << values.size() << " (must be 1)");
        result.RecordSets.emplace_back(std::move(values.front().ReplicaObject));
    }

    if (!result.RecordSets.empty()) {
        listOptions.SeekType = NYP::NYPReplica::ESeekType::Next;
        listOptions.SeekKey = result.RecordSets.back().GetObject().Meta().id();
    }
    result.HasReachedEnd = !listOptions.Limit || result.RecordSets.size() < listOptions.Limit;
    result.ContinuationOptions = std::move(listOptions);

    return result;
}

////////////////////////////////////////////////////////////////////////////////

TListRecordSetsFromReplicasResult ListRecordSetsFromReplicas(
    const TReplicasList& replicas,
    THashMap<TString, TFromReplicaListOptions> clustersListOptions,
    IThreadPool& threadPool,
    const ui64 limit
) {
    THashMap<TString, NThreading::TFuture<TListRecordSetsFromReplicaResult>> listResultsFutures(replicas.size());
    for (const auto& [cluster, replica] : replicas) {
        TFromReplicaListOptions& listOptions = clustersListOptions[cluster];
        listOptions.Limit = limit ? (limit + replicas.size() - 1) : 0;

        listResultsFutures[cluster] = NThreading::Async(
            [replica = replica, listOptions] {
                return ListRecordSetsFromReplica(*replica, std::move(listOptions));
            },
            threadPool
        );
    }

    THashMap<TString, TListRecordSetsFromReplicaResult> listResults(replicas.size());
    THashMap<TString, TVector<TSerializedRecordSet>> clustersRecordSets(replicas.size());
    for (auto&& [cluster, resultFuture] : listResultsFutures) {
        listResults.emplace(cluster, std::move(resultFuture.ExtractValueSync()));
        clustersRecordSets.emplace(cluster, std::move(listResults[cluster].RecordSets));
    }

    TListRecordSetsFromReplicasResult result;
    TString lastId;
    bool hasReachedLimit = false;
    Iterate(clustersRecordSets, [limit, &result, &lastId, &hasReachedLimit](const TRecordSetReplicas& recordSetReplicas) {
        if (limit && result.RecordSets.size() == limit) {
            hasReachedLimit = true;
            return false;
        }

        TMaybe<TRecordSet> mergedRecordSet = MergeInOne(
            recordSetReplicas.Replicas,
            TMergeOptions()
                .SetFormChangelist(false)
                .SetMergeAcls(false)
        );

        if (mergedRecordSet.Defined()) {
            result.RecordSets.emplace_back(std::move(*mergedRecordSet));
        }

        lastId = recordSetReplicas.Fqdn;
        return true;
    });

    result.HasReachedEnd = !hasReachedLimit && AllOf(listResults, [](const auto& it) { return it.second.HasReachedEnd; });

    result.ClustersContinuationOptions.reserve(listResults.size());
    for (auto&& [cluster, listResult] : listResults) {
        result.YpTimestamps[cluster] = listResult.YpTimestamp;

        listResult.ContinuationOptions.SeekType = NYP::NYPReplica::ESeekType::Next;
        listResult.ContinuationOptions.SeekKey = lastId;
        result.ClustersContinuationOptions.emplace(cluster, std::move(listResult.ContinuationOptions));
    }

    return result;
}

////////////////////////////////////////////////////////////////////////////////

TListRecordSetsResult ListMulticlusterZone(
    const TZone& zone,
    const TReplicasList& replicas,
    TListMulticlusterZoneOptions listOptions,
    IThreadPool& threadPool,
    NInfra::TLogFramePtr logFrame,
    const NInfra::TSensorGroup& sensorGroup
) {
    Y_UNUSED(logFrame, sensorGroup);

    THashMap<TString, TFromReplicaListOptions> clustersListOptions(replicas.size());
    for (const auto& [cluster, replica] : replicas) {
        TFromReplicaListOptions& clusterListOptions = clustersListOptions[cluster];
        switch (listOptions.SeekType) {
            case ESeekType::ToFirst:
                clusterListOptions.SeekType = NYP::NYPReplica::ESeekType::ToFirst;
                break;
            case ESeekType::ToLast:
                clusterListOptions.SeekType = NYP::NYPReplica::ESeekType::ToLast;
                break;
            case ESeekType::AtOrNext:
                clusterListOptions.SeekType = NYP::NYPReplica::ESeekType::AtOrNext;
                break;
            case ESeekType::Next:
                clusterListOptions.SeekType = NYP::NYPReplica::ESeekType::Next;
                break;
        }
        clusterListOptions.SeekKey = listOptions.SeekKey;
        clusterListOptions.Snapshot = replica->GetReplicaSnapshot();
        clusterListOptions.Filter = [&zone](const TStringBuf key, const TVector<TYpReplicaStorageElement>&) {
            return TDnsNameView(key).IsPartOf(zone.GetDnsName());
        };
    }

    TListRecordSetsResult result;
    while (!listOptions.Limit || result.RecordSets.size() < listOptions.Limit) {
        TListRecordSetsFromReplicasResult listResultBatch = ListRecordSetsFromReplicas(replicas, clustersListOptions, threadPool, listOptions.Limit);

        result.YpTimestamps = std::move(listResultBatch.YpTimestamps);

        const size_t mergeSize = listOptions.Limit
            ? Min(listOptions.Limit - result.RecordSets.size(), listResultBatch.RecordSets.size())
            : listResultBatch.RecordSets.size();
        std::move(listResultBatch.RecordSets.begin(), listResultBatch.RecordSets.begin() + mergeSize, std::back_inserter(result.RecordSets));

        clustersListOptions = std::move(listResultBatch.ClustersContinuationOptions);
        if (mergeSize < listResultBatch.RecordSets.size() && !result.RecordSets.empty()) {
            const TString& lastId = result.RecordSets.back().Meta().id();
            for (auto& [cluster, clusterListOptions] : clustersListOptions) {
                clusterListOptions.SeekKey = lastId;
            }
        }

        if (listResultBatch.HasReachedEnd) {
            break;
        }
    }

    TVector<TRecordSet>::iterator zoneRecordSetIt = result.RecordSets.end();
    if (listOptions.BuildSOARecordFromConfigIfNotFound || listOptions.BuildNSRecordsFromConfigIfNotFound) {
        auto it = LowerBound(
            result.RecordSets.begin(), result.RecordSets.end(),
            zone.GetName(), [](const TRecordSet& recordSet, const TString& zoneName) {
                return recordSet.Meta().id() < zoneName;
            }
        );
        if (it != result.RecordSets.end() && it->Meta().id() == zone.GetName()) {
            zoneRecordSetIt = it;
        } else {
            NYP::NClient::TDnsRecordSet zoneRecordSet;
            zoneRecordSet.MutableMeta()->set_id(zone.GetName());
            result.RecordSets.emplace_back(std::move(zoneRecordSet));
            zoneRecordSetIt = std::prev(result.RecordSets.end());
        }
    }

    auto hasRecordOfType = [](const TRecordSet& recordSet, TDnsRecordSetSpec::TResourceRecord::EType type) {
        return AnyOf(recordSet.Spec().records(), [type](const TDnsRecordSetSpec::TResourceRecord& record) { return record.type() == type; });
    };

    if (listOptions.BuildSOARecordFromConfigIfNotFound && !hasRecordOfType(*zoneRecordSetIt, TDnsRecordSetSpec::TResourceRecord::SOA)) {
        *zoneRecordSetIt->MutableSpec()->add_records() = CreateSOARecordProto(zone.Config(), MakeSerialFromYpTimestamps(result.YpTimestamps));
    }

    if (listOptions.BuildNSRecordsFromConfigIfNotFound && !hasRecordOfType(*zoneRecordSetIt, TDnsRecordSetSpec::TResourceRecord::NS)) {
        for (const TString& nameserver : zone.Config().GetNameservers()) {
            *zoneRecordSetIt->MutableSpec()->add_records() = CreateNSRecordProto(nameserver, zone.Config().GetNSRecordTtl());
        }
    }

    result.ContinuationOptions = std::move(listOptions);
    auto& clusterContinuationOptions = clustersListOptions.begin()->second;
    switch (clusterContinuationOptions.SeekType) {
        case NYP::NYPReplica::ESeekType::ToFirst:
            result.ContinuationOptions.SeekType = ESeekType::ToFirst;
            break;
        case NYP::NYPReplica::ESeekType::ToLast:
            result.ContinuationOptions.SeekType = ESeekType::ToLast;
            break;
        case NYP::NYPReplica::ESeekType::AtOrNext:
            result.ContinuationOptions.SeekType = ESeekType::AtOrNext;
            break;
        case NYP::NYPReplica::ESeekType::Next:
            result.ContinuationOptions.SeekType = ESeekType::Next;
            break;
    }
    result.ContinuationOptions.SeekKey = std::move(clusterContinuationOptions.SeekKey);

    return result;
}

////////////////////////////////////////////////////////////////////////////////

} // namespace NYpDns
