#include "reader.h"

#include "writer.h"

#include <util/string/cast.h>
#include <util/string/split.h>

namespace NYpDns {

bool TZoneReader::ReadRecord(TRecord& record) {
    try {
        record = CreateRecordFromString(ReadNextLineWithRecord(), ZoneOrigin_, ZoneTtl_, NamePrevRecord_);
        NamePrevRecord_ = record.Name;
        return true;
    } catch (const TEndOfFileException& e) {
        return false;
    }
}

bool TZoneReader::ParseDirective(TStringBuf line, EDirectiveType& type, TString& value) {
    if (!line.SkipPrefix("$")) {
        return false;
    }

    bool isOkSplit = StringSplitter(line)
        .SplitByFunc(&IsFieldDelimiterOfEntry)
        .SkipEmpty()
        .Take(2)
        .TryCollectInto(&line, &value);

    Y_ENSURE_EX(isOkSplit && TryFromString(line, type), TIncorrectDirectiveException() << "Incorrect directive: " << line);

    return true;
}

void TZoneReader::SetDirective(EDirectiveType type, const TString& value) {
    switch (type) {
        case EDirectiveType::ORIGIN:
            ZoneOrigin_ = value;
            break;
        case EDirectiveType::TTL:
            ZoneTtl_ = FromString(value);
            break;
        default:
            ythrow TDirectiveNotSupportedException() << "Directive not supported: " << ToString(type);
    }
}

TString TZoneReader::ReadNextLineWithRecord() {
    while (true) {
        TString line;

        Y_ENSURE_EX(Input_.ReadLine(line), TEndOfFileException() << "End of file");

        // Deleting comments in the record
        TString lineWithoutComment(TStringBuf(line).Before(';'));

        if (AllOf(lineWithoutComment, &IsFieldDelimiterOfEntry)) {
            continue;
        }

        EDirectiveType type;
        TString directiveValue;
        if (ParseDirective(lineWithoutComment, type, directiveValue)) {
            SetDirective(type, directiveValue);
        } else {
            return lineWithoutComment;
        }
    }
}

} // namespace NYpDns
