#include "reader.h"

#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/serialized_enum.h>

using namespace NYpDns;

Y_UNIT_TEST_SUITE(TZoneReader) {
    Y_UNIT_TEST(ValidateParseDirective) {
        {
            EDirectiveType type(EDirectiveType::TTL);
            TString value("value");

            UNIT_ASSERT(!TZoneReader::ParseDirective("ORIGIN DirectiveValue", type, value));
            UNIT_ASSERT_EQUAL(type, EDirectiveType::TTL);
            UNIT_ASSERT_STRINGS_EQUAL(value, "value");

            UNIT_ASSERT(!TZoneReader::ParseDirective("NotDirective DirectiveValue", type, value));
            UNIT_ASSERT_EQUAL(type, EDirectiveType::TTL);
            UNIT_ASSERT_STRINGS_EQUAL(value, "value");

            UNIT_ASSERT(!TZoneReader::ParseDirective("NotDirective", type, value));
            UNIT_ASSERT_EQUAL(type, EDirectiveType::TTL);
            UNIT_ASSERT_STRINGS_EQUAL(value, "value");

            UNIT_ASSERT(!TZoneReader::ParseDirective("", type, value));
            UNIT_ASSERT_EQUAL(type, EDirectiveType::TTL);
            UNIT_ASSERT_STRINGS_EQUAL(value, "value");
        }

        for (const auto type : GetEnumAllValues<EDirectiveType>()) {
            EDirectiveType typeAfterParse;
            TString value;
            UNIT_ASSERT(TZoneReader::ParseDirective("$" + ToString(type) + " \t \t DirectiveValue", typeAfterParse, value));
            UNIT_ASSERT_EQUAL(typeAfterParse, type);
            UNIT_ASSERT_STRINGS_EQUAL(value, "DirectiveValue");
        }

        {
            EDirectiveType type;
            TString value;
            UNIT_ASSERT_EXCEPTION(TZoneReader::ParseDirective("$IncorrectDirective DirectiveValue", type, value), TIncorrectDirectiveException);
            UNIT_ASSERT_EXCEPTION(TZoneReader::ParseDirective("$IncorrectDirective", type, value), TIncorrectDirectiveException);
            UNIT_ASSERT_EXCEPTION(TZoneReader::ParseDirective("$", type, value), TIncorrectDirectiveException);                                    
        }
    }

    Y_UNIT_TEST(ValidateSetDirective) {
        {
            TStringBuilder zoneFile;
            zoneFile << "$ORIGIN \t \t yandex.net. \t\n";
            zoneFile << "$TTL \t \t\t 600 \t \n";
            zoneFile << "@ IN AAAA RecordData\n";
            zoneFile << "$ORIGIN\t\tnet.\t\t\n";
            zoneFile << "$TTL 200   \n";
            zoneFile << "@ IN AAAA RecordData \n";
            TStringInput input(zoneFile);
            TZoneReader zoneReader(input);
            
            TRecord record;
            UNIT_ASSERT(zoneReader.ReadRecord(record));
            UNIT_ASSERT_EQUAL(record.Name, DNSName("yandex.net."));
            UNIT_ASSERT_EQUAL(record.Ttl, 600);

            UNIT_ASSERT(zoneReader.ReadRecord(record));
            UNIT_ASSERT_EQUAL(record.Name, DNSName("net."));
            UNIT_ASSERT_EQUAL(record.Ttl, 200);
        }
        {
            TStringBuilder zoneFile;
            zoneFile << "$INCLUDE \t \t RecordData \t\n";
            zoneFile << "yandex.net. IN AAAA RecordData\n";
            TStringInput input(zoneFile);
            TZoneReader zoneReader(input);
            
            TRecord record;
            UNIT_ASSERT_EXCEPTION(zoneReader.ReadRecord(record), TDirectiveNotSupportedException);
        }
        {
            TStringBuilder zoneFile;
            zoneFile << "$GENERATE \t \t RecordData \t\n";
            zoneFile << "yandex.net. IN AAAA RecordData\n";
            TStringInput input(zoneFile);
            TZoneReader zoneReader(input);
            
            TRecord record;
            UNIT_ASSERT_EXCEPTION(zoneReader.ReadRecord(record), TDirectiveNotSupportedException);
        }
        {
            TStringBuilder zoneFile;
            zoneFile << "$TTL \t \t -600 \t\n";
            zoneFile << "yandex.net. IN AAAA RecordData\n";
            TStringInput input(zoneFile);
            TZoneReader zoneReader(input);

            TRecord record;
            UNIT_ASSERT_EXCEPTION(zoneReader.ReadRecord(record), TFromStringException);
        }
    }

    Y_UNIT_TEST(ValidateReadNextLineWithRecord) {
        {
            TStringBuilder zoneFile;
            zoneFile << "\n";
            zoneFile << "  ; comments aren't needed\n";
            zoneFile << "$TTL 3600 ; TTL1 \n";
            zoneFile << "$ORIGIN yandex.net. ; ORIGIN1\n";
            zoneFile << "palindrome IN AAAA RecordData\n";
            zoneFile << "$ORIGIN palindrome.yandex.net. ; ORIGIN2\n";
            zoneFile << "$TTL 200 ; T T L 2 \n";
            zoneFile << "abacaba IN AAAA RecordData\n";
            TStringInput input(zoneFile);
            TZoneReader zoneReader(input);

            TRecord record;
            UNIT_ASSERT(zoneReader.ReadRecord(record));
            UNIT_ASSERT_EQUAL(record.Name, DNSName("palindrome.yandex.net."));
            UNIT_ASSERT_EQUAL(record.Ttl, 3600);

            UNIT_ASSERT(zoneReader.ReadRecord(record));
            UNIT_ASSERT_EQUAL(record.Name, DNSName("abacaba.palindrome.yandex.net."));
            UNIT_ASSERT_EQUAL(record.Ttl, 200);

            UNIT_ASSERT(!zoneReader.ReadRecord(record));
        }
        {
            TString zoneFile;
            TStringInput input(zoneFile);
            TZoneReader zoneReader(input);

            TRecord record;
            UNIT_ASSERT(!zoneReader.ReadRecord(record));
        }
    }

    Y_UNIT_TEST(ValidateReadRecord) {
        TStringBuilder zoneFile;
        zoneFile << "$ORIGIN \t \t yandex.net. \t\n";
        zoneFile << "@ IN AAAA RecordData ; xm \n";
        zoneFile << " IN AAAA RecordData ; xmm \n";
        zoneFile << " \t \t\t\t IN AAAA RecordData;xmmmmm\n";
        TStringInput input(zoneFile);
        TZoneReader zoneReader(input);

        for (int i = 0; i < 3; ++i) {
            TRecord record;
            UNIT_ASSERT(zoneReader.ReadRecord(record));
            UNIT_ASSERT_EQUAL(record.Name, DNSName("yandex.net."));
        }

        TRecord record;
        UNIT_ASSERT(!zoneReader.ReadRecord(record));
    }
}
