#pragma once

#include <infra/libs/yp_dns/record_set/record.h>
#include <infra/libs/yp_dns/record_set/record_set.h>

#include <infra/libs/yp_dns/zone/protos/config.pb.h>

#include <infra/contrib/pdns/power_dns/dnsname.hh>

#include <yp/yp_proto/yp/client/api/proto/data_model.pb.h>

#include <util/folder/path.h>
#include <util/generic/maybe.h>
#include <util/stream/file.h>
#include <util/stream/fwd.h>
#include <util/string/ascii.h>
#include <util/string/join.h>

namespace NYpDns {

class IZoneWriter {
public:
    virtual ~IZoneWriter() = default;

    virtual void Finish() = 0;

    virtual void Validate() const = 0;

    virtual void WriteRecordSetRecords(const TRecordSet& recordSet) = 0;

    virtual void WriteRecord(TRecord record) = 0;
};

static inline IZoneWriter& operator<<(IZoneWriter& writer, const TRecordSet& recordSet) {
    writer.WriteRecordSetRecords(recordSet);
    return writer;
}

struct TZoneWriterOptions {
    bool SetOrigin = false;
    bool SetDefaultTtl = true;

    bool ValidateOnFinish = true;
    bool ValidateSOARecordExistence = true;
    bool ValidateNSRecordsExistence = true;

    TZoneConfig ZoneConfig;
};

enum class EDirectiveType {
    ORIGIN    /* "ORIGIN" */,
    INCLUDE   /* "INCLUDE" */,
    TTL       /* "TTL" */,
    GENERATE  /* "GENERATE" */
};

class TZoneWriter : public IZoneWriter {
    friend class TLineWriter;

    class TLineWriter {
    public:
        TLineWriter(TZoneWriter& fileWriter)
            : ZoneWriter_(fileWriter)
        {
        }

        ~TLineWriter() {
            ZoneWriter_.Output_ << std::move(Line_) << '\n';
        }

        template <typename TValue>
        void Write(const TValue& value) {
            Line_ << (!Line_.empty() && !IsAsciiSpace(Line_.back()) ? " " : "") << value;
        }

    private:
        TZoneWriter& ZoneWriter_;
        TStringBuilder Line_;
    };

public:
    TZoneWriter(IOutputStream& output, TZoneWriterOptions zoneFileOptions)
        : Output_(output)
        , Options_(std::move(zoneFileOptions))
        , ZoneName_(Options_.ZoneConfig.GetName())
    {
        WriteDirectives();
    }

    void Finish() override {
        if (Options_.ValidateOnFinish) {
            Validate();
        }
        Output_.Finish();
    }

    void Validate() const override {
        if (Options_.ValidateSOARecordExistence) {
            Y_ENSURE(HasSOARecord_, "No SOA record for zone " << ZoneName_.toString());
        }

        if (Options_.ValidateNSRecordsExistence) {
            Y_ENSURE(HasNSRecords_, "No NS records for zone " << ZoneName_.toString());
        }
    }

    void WriteRecordSetRecords(const TRecordSet& recordSet) override {
        const DNSName domain(recordSet.Meta().id());
        for (const auto& record : recordSet.Spec().records()) {
            WriteRecord(CreateRecordFromProto(domain, record));
        }
    }

    void WriteRecord(TRecord record) override {
        if (Options_.SetOrigin && record.Name != ZoneName_) {
            record.Name.makeUsRelative(ZoneName_);
            if (record.Name.empty()) {
                return;
            }
        }

        TLineWriter lineWriter(*this);
        lineWriter.Write(record.Name.toString());
        lineWriter.Write("");  // print whitespace after domain
        if (record.Ttl) {
            lineWriter.Write(record.Ttl);
        } else if (!Options_.SetDefaultTtl) {
            lineWriter.Write(Options_.ZoneConfig.GetDefaultTtl());
        }
        lineWriter.Write(record.Class);
        lineWriter.Write(record.Type);
        lineWriter.Write(record.FormatData());

        switch (record.Type) {
            case ERecordType::SOA:
                HasSOARecord_ = true;
                break;
            case ERecordType::NS:
                HasNSRecords_ = true;
                break;
            default:
                break;
        }
    }

private:
    void WriteDirectives() {
        if (Options_.SetOrigin) {
            WriteDirective(EDirectiveType::ORIGIN, ZoneName_.toString());
        }
        if (Options_.SetDefaultTtl) {
            WriteDirective(EDirectiveType::TTL, Options_.ZoneConfig.GetDefaultTtl());
        }
    }

    template <typename TValue>
    void WriteDirective(EDirectiveType type, const TValue& value) {
        WriteLine("$" + ToString(type), value);
    }

    template <typename... TArgs>
    void WriteLine(TArgs&&... args) {
        TLineWriter lineWriter(*this);
        (lineWriter.Write(std::forward<TArgs>(args)), ...);
    }

private:
    IOutputStream& Output_;
    const TZoneWriterOptions Options_;
    const DNSName ZoneName_;
    bool Finished_ = false;
    bool HasSOARecord_ = false;
    bool HasNSRecords_ = false;
};

class TZoneFileWriter : public IZoneWriter {
public:
    TZoneFileWriter(const TFsPath& zoneFilePath, TZoneWriterOptions zoneFileOptions)
        : FileOutput_(zoneFilePath)
        , ZoneWriter_(FileOutput_, std::move(zoneFileOptions))
    {
    }

    void Finish() override {
        ZoneWriter_.Finish();
    }

    void Validate() const override {
        ZoneWriter_.Validate();
    }

    void WriteRecordSetRecords(const TRecordSet& recordSet) override {
        ZoneWriter_.WriteRecordSetRecords(recordSet);;
    }

    void WriteRecord(TRecord record) override {
        ZoneWriter_.WriteRecord(std::move(record));
    }

private:
    TFileOutput FileOutput_;
    TZoneWriter ZoneWriter_;
};

} // namespace NYpDns
