#include "backup.h"

#include <contrib/libs/rocksdb/include/rocksdb/utilities/backup_engine.h>

namespace NYP::NYPReplica {
    TBackupEngineOptions::TBackupEngineOptions(const TStringBuf name, const TBackupConfig& config)
        : Path(TFsPath(config.GetPath()) / name)
        , BackupLogFiles(config.GetBackupLogFiles())
    {
    }

    TBackupInfo::TBackupInfo(const rocksdb::BackupInfo& info)
        : Id(info.backup_id)
        , Timestamp(info.timestamp)
        , Size(info.size)
        , NumberFiles(info.number_files)
    {
        IsCorrupted = !Meta.ParseFromArray(info.app_metadata.data(), info.app_metadata.size());
    }

    class TBackupEngine::TImpl {
    public:
        TImpl(TBackupEngineOptions options)
            : Options_(std::move(options))
            , BackupEngineOptions_(Options_.Path.GetPath())
        {
            ResetBackupEngineOptions();
        }

        bool IsOpened() const {
            return static_cast<bool>(BackupEngine_);
        }

        TStatus Open() {
            if (IsOpened()) {
                return TStatus::Ok();
            }

            return Reopen();
        }

        TStatus Reopen() {
            Options_.Path.Parent().MkDirs();

            BackupEngine_.Reset();

            rocksdb::BackupEngine* backupEngine = nullptr;
            if (TStatus status = rocksdb::BackupEngine::Open(rocksdb::Env::Default(), BackupEngineOptions_, &backupEngine); !status) {
                return TStatus(std::move(status));
            }

            BackupEngine_.Reset(backupEngine);

            return TStatus::Ok();
        }

        void Drop() {
            if (IsOpened()) {
                PurgeOldBackups(0);
            }

            BackupEngine_.Reset();

            try {
                Options_.Path.ForceDelete();
            } catch (...) {
            }
        }

        TStatus CreateNewBackup(const TCreateBackupOptions& options, TStorage* storage) {
            if (!storage || !storage->IsOpened()) {
                return TStatus::DatabaseNotOpened();
            }

            if (!IsOpened()) {
                TStatus reopenStatus = Reopen();
                if (!reopenStatus) {
                    return reopenStatus;
                }
            }

            TString meta = options.Meta.SerializeAsString();
            rocksdb::Status status = BackupEngine_->CreateNewBackupWithMetadata(storage->InternalDB(), std::move(meta), options.FlushBeforeBackup);
            return TStatus(std::move(status));
        }

        TStatus DeleteBackup(ui64 backupId) {
            if (!IsOpened()) {
                TStatus reopenStatus = Reopen();
                if (!reopenStatus) {
                    return reopenStatus;
                }
            }

            rocksdb::Status status = BackupEngine_->DeleteBackup(backupId);
            return TStatus(std::move(status));
        }

        TStatus PurgeOldBackups(ui32 backupsToKeep) {
            if (!IsOpened()) {
                TStatus reopenStatus = Reopen();
                if (!reopenStatus) {
                    return reopenStatus;
                }
            }
            rocksdb::Status status = BackupEngine_->PurgeOldBackups(backupsToKeep);
            return TStatus(std::move(status));
        }

        void StopBackup() {
            Y_ENSURE(BackupEngine_);

            BackupEngine_->StopBackup();
        }

        TVector<TBackupInfo> ListBackups() {
            if (!IsOpened() && !Reopen()) {
                return {};
            }

            std::vector<rocksdb::BackupInfo> backupInfos;
            BackupEngine_->GetBackupInfo(&backupInfos);

            TVector<TBackupInfo> result;
            result.reserve(backupInfos.size());
            for (const rocksdb::BackupInfo& info : backupInfos) {
                result.emplace_back(info);
            }
            return result;
        }

    private:
        void ResetBackupEngineOptions() {
            BackupEngineOptions_.backup_dir = Options_.Path.GetPath();
            BackupEngineOptions_.backup_log_files = Options_.BackupLogFiles;
        }

    private:
        TBackupEngineOptions Options_;

        rocksdb::BackupEngineOptions BackupEngineOptions_;

        THolder<rocksdb::BackupEngine> BackupEngine_;
    };

    TBackupEngine::TBackupEngine(TBackupEngineOptions options)
        : Impl_(MakeHolder<TImpl>(std::move(options)))
    {
    }

    TBackupEngine::~TBackupEngine() = default;

    bool TBackupEngine::IsOpened() const {
        return Impl_->IsOpened();
    }

    TStatus TBackupEngine::Open() {
        return Impl_->Open();
    }

    TStatus TBackupEngine::Reopen() {
        return Impl_->Reopen();
    }

    void TBackupEngine::Drop() {
        Impl_->Drop();
    }

    TStatus TBackupEngine::CreateNewBackup(const TCreateBackupOptions& options, TStorage* storage) {
        return Impl_->CreateNewBackup(options, storage);
    }

    TStatus TBackupEngine::DeleteBackup(ui64 backupId) {
        return Impl_->DeleteBackup(backupId);
    }

    TStatus TBackupEngine::PurgeOldBackups(ui32 backupsToKeep) {
        return Impl_->PurgeOldBackups(backupsToKeep);
    }

    void TBackupEngine::StopBackup() {
        Impl_->StopBackup();
    }

    TVector<TBackupInfo> TBackupEngine::ListBackups() {
        return Impl_->ListBackups();
    }
} // namespace NYP::NYPReplica
