#include "cache.h"
#include "util.h"

#include <contrib/libs/rocksdb/include/rocksdb/db.h>
#include <contrib/libs/rocksdb/include/rocksdb/write_batch.h>

namespace NYP::NYPReplica {

namespace {

class TBatchUpdatesHandler : public rocksdb::WriteBatch::Handler {
public:
    TBatchUpdatesHandler(TCache& cache)
        : Cache_(cache)
    {
    }

    rocksdb::Status PutCF(ui32 cfId, const rocksdb::Slice& key, const rocksdb::Slice& value) override {
        Cache_.Update(cfId, ToString(key), ToString(value));
        return rocksdb::Status::OK();
    };

    rocksdb::Status DeleteCF(ui32 cfId, const rocksdb::Slice& key) override {
        Cache_.Delete(cfId, ToString(key));
        return rocksdb::Status::OK();
    };

private:
    TCache& Cache_;
};

} // anonymous namespace

TCache::TCache(const THashMap<TString, TAtomicSharedPtr<rocksdb::ColumnFamilyHandle>>& stableColumnFamilies,
               const THashMap<TString, TAtomicSharedPtr<rocksdb::ColumnFamilyHandle>>& dynamicColumnFamilies
)
    : StableCFLocks_(stableColumnFamilies.size())
    , DynamicCFLocks_(dynamicColumnFamilies.size())
    , StableCFData_(stableColumnFamilies.size())
    , DynamicCFData_(dynamicColumnFamilies.size())
{
    for (const auto& [cfName, cfHolder] : stableColumnFamilies) {
        Y_ENSURE(cfHolder.Get());
        StableCFLocks_[cfHolder->GetID()];
        StableCFData_[cfHolder->GetID()];
    }

    for (const auto& [cfName, cfHolder] : dynamicColumnFamilies) {
        Y_ENSURE(cfHolder.Get());
        DynamicCFLocks_[cfHolder->GetID()];
        DynamicCFData_[cfHolder->GetID()];
    }
}

void TCache::CreateColumnFamilies(const TVector<TAtomicSharedPtr<rocksdb::ColumnFamilyHandle>>& columnFamilies) {
    TWriteGuard guard(MapLock_);
    for (const auto& cf : columnFamilies) {
        DynamicCFLocks_[cf->GetID()];
        DynamicCFData_[cf->GetID()];
    }
}

void TCache::DeleteColumnFamilies(const TVector<TAtomicSharedPtr<rocksdb::ColumnFamilyHandle>>& columnFamilies) {
    TWriteGuard guard(MapLock_);
    for (const auto& cf : columnFamilies) {
        DynamicCFLocks_.erase(cf->GetID());
        DynamicCFData_.erase(cf->GetID());
    }
}

void TCache::Update(ui32 columnFamilyId, TString key, TString value) {
    if (auto lockIt = StableCFLocks_.find(columnFamilyId); lockIt != StableCFLocks_.end()) {
        TWriteGuard guardCF(lockIt->second);
        UpdateStableCFNoLock(columnFamilyId, std::move(key), std::move(value));
        return;
    }

    TReadGuard guardMap(MapLock_);
    if (auto lockIt = DynamicCFLocks_.find(columnFamilyId); Y_LIKELY(lockIt != DynamicCFLocks_.end())) {
        TWriteGuard guardCF(lockIt->second);
        UpdateDynamicCFNoLock(columnFamilyId, std::move(key), std::move(value));
    }
}

void TCache::Update(const rocksdb::WriteBatch& batch) {
    TBatchUpdatesHandler handler(*this);
    rocksdb::Status status = batch.Iterate(&handler);
    Y_ENSURE(status.ok());
}

void TCache::UpdateStableCFNoLock(ui32 columnFamilyId, TString key, TString value) {
    StableCFData_.at(columnFamilyId)[std::move(key)] = std::move(value);
}

void TCache::UpdateDynamicCFNoLock(ui32 columnFamilyId, TString key, TString value) {
    DynamicCFData_.at(columnFamilyId)[std::move(key)] = std::move(value);
}

void TCache::Delete(ui32 columnFamilyId, const TString& key) {
    if (auto lockIt = StableCFLocks_.find(columnFamilyId); lockIt != StableCFLocks_.end()) {
        TWriteGuard guardCF(lockIt->second);
        DeleteStableCFNoLock(columnFamilyId, std::move(key));
        return;
    }

    TReadGuard guardMap(MapLock_);
    auto lockIt = DynamicCFLocks_.find(columnFamilyId);
    if (Y_LIKELY(lockIt != DynamicCFLocks_.end())) {
        TWriteGuard guardCF(lockIt->second);
        DeleteDynamicCFNoLock(columnFamilyId, key);
    }
}

void TCache::DeleteStableCFNoLock(ui32 columnFamilyId, const TString& key) {
    StableCFData_.at(columnFamilyId).erase(key);
}

void TCache::DeleteDynamicCFNoLock(ui32 columnFamilyId, const TString& key) {
    DynamicCFData_.at(columnFamilyId).erase(key);
}

TExpected<TString, TCache::EGetError> TCache::Get(ui32 columnFamilyId, const TString& key) {
    if (auto lockIt = StableCFLocks_.find(columnFamilyId); lockIt != StableCFLocks_.end()) {
        if (TTryReadGuard guardCF(lockIt->second); guardCF) {
            return GetFromCF(StableCFData_.at(columnFamilyId), key);
        } else {
            return EGetError::BUSY;
        }
    }

    if (TTryReadGuard guardMap(MapLock_); !guardMap) {
        return EGetError::BUSY;
    } else if (auto lockIt = DynamicCFLocks_.find(columnFamilyId); Y_UNLIKELY(lockIt == DynamicCFLocks_.end())) {
        return EGetError::CF_REMOVED;
    } else if (TTryReadGuard guardCF(lockIt->second); !guardCF) {
        return EGetError::BUSY;
    } else {
        return GetFromCF(DynamicCFData_.at(columnFamilyId), key);
    }
}

TExpected<TString, TCache::EGetError> TCache::GetFromCF(const THashMap<TString, TString>& data, const TString& key) {
    if (auto* it = data.FindPtr(key)) {
        return *it;
    } else {
        return EGetError::NOT_FOUND;
    }
}

void TCache::FillNoLock(const THashMap<TString, TAtomicSharedPtr<rocksdb::ColumnFamilyHandle>>& stableColumnFamilies,
                        const THashMap<TString, TAtomicSharedPtr<rocksdb::ColumnFamilyHandle>>& dynamicColumnFamilies,
                        std::function<THolder<rocksdb::Iterator>(rocksdb::ColumnFamilyHandle*)> getIterator) {
    for (const auto& [cfName, cfHolder] : stableColumnFamilies) {
        Y_ENSURE(cfHolder.Get());

        THolder<rocksdb::Iterator> it = getIterator(cfHolder.Get());

        it->SeekToFirst();
        for (; it->Valid(); it->Next()) {
            UpdateStableCFNoLock(
                cfHolder->GetID(),
                ToString(it->key()),
                ToString(it->value()));
        }
    }

    for (const auto& [cfName, cfHolder] : dynamicColumnFamilies) {
        Y_ENSURE(cfHolder.Get());

        THolder<rocksdb::Iterator> it = getIterator(cfHolder.Get());

        it->SeekToFirst();
        for (; it->Valid(); it->Next()) {
            UpdateDynamicCFNoLock(
                cfHolder->GetID(),
                ToString(it->key()),
                ToString(it->value()));
        }
    }
}

} // namespace NYP::NYPReplica
