#pragma once

#include <infra/libs/outcome/result.h>

#include <util/generic/hash.h>
#include <util/system/rwlock.h>

namespace rocksdb {
    class ColumnFamilyHandle;
    class Iterator;
    class WriteBatch;
}

namespace NYP::NYPReplica {

class TCache {
public:
    enum class EGetError  {
        NOT_FOUND = 0,
        BUSY = 1,
        CF_REMOVED = 2,
    };

    TCache(const THashMap<TString, TAtomicSharedPtr<rocksdb::ColumnFamilyHandle>>& stableColumnFamilies,
           const THashMap<TString, TAtomicSharedPtr<rocksdb::ColumnFamilyHandle>>& dynamicColumnFamilies);

    void CreateColumnFamilies(const TVector<TAtomicSharedPtr<rocksdb::ColumnFamilyHandle>>& columnFamilies);
    void DeleteColumnFamilies(const TVector<TAtomicSharedPtr<rocksdb::ColumnFamilyHandle>>& columnFamilies);

    void Update(ui32 columnFamilyId, TString key, TString value);
    void Update(const rocksdb::WriteBatch& batch);

    void Delete(ui32 columnFamilyId, const TString& key);

    TExpected<TString, EGetError> Get(ui32 columnFamilyId, const TString& key);

    void FillNoLock(const THashMap<TString, TAtomicSharedPtr<rocksdb::ColumnFamilyHandle>>& stableColumnFamilies,
                    const THashMap<TString, TAtomicSharedPtr<rocksdb::ColumnFamilyHandle>>& dynamicColumnFamilies,
                    std::function<THolder<rocksdb::Iterator>(rocksdb::ColumnFamilyHandle*)> getIterator);
private:
    void UpdateStableCFNoLock(ui32 columnFamilyId, TString key, TString value);
    void UpdateDynamicCFNoLock(ui32 columnFamilyId, TString key, TString value);

    void DeleteStableCFNoLock(ui32 columnFamilyId, const TString& key);
    void DeleteDynamicCFNoLock(ui32 columnFamilyId, const TString& key);

    TExpected<TString, EGetError> GetFromCF(const THashMap<TString, TString>& data, const TString& key);

private:
    TRWMutex MapLock_;
    THashMap<ui32, TRWMutex> StableCFLocks_;
    THashMap<ui32, TRWMutex> DynamicCFLocks_;
    THashMap<ui32, THashMap<TString, TString>> StableCFData_;
    THashMap<ui32, THashMap<TString, TString>> DynamicCFData_;
};

using TCacheHolder = THolder<TCache>;

} // namespace NYP::NYPReplica
