#pragma once

#include <yp/cpp/yp/selector_result.h>

#include <util/generic/algorithm.h>
#include <util/generic/vector.h>

namespace NYP::NYPReplica {
    enum class EObjectType {
        PUT = 0,
        DELETE = 1
    };

    template <typename TReplicaObject>
    struct TStorageElementRef;

    template <typename TReplicaObject>
    struct TStorageElement {
        TReplicaObject ReplicaObject;
        EObjectType Type = EObjectType::PUT;

        TStorageElement() = default;
        TStorageElement(TReplicaObject replicaObject, EObjectType type = EObjectType::PUT)
            : ReplicaObject(std::move(replicaObject))
            , Type(type)
        {
        }

        explicit TStorageElement(const TStorageElementRef<TReplicaObject>& storageElementRef)
            : ReplicaObject(storageElementRef.ReplicaObject.get())
            , Type(storageElementRef.Type)
        {
        }

        bool operator<(const TStorageElement& rhs) const {
            return ReplicaObject < rhs.ReplicaObject;
        }

        bool operator<(const TStorageElementRef<TReplicaObject>& rhs) const {
            return ReplicaObject < rhs.ReplicaObject.get();
        }

        bool operator==(const TStorageElement& rhs) const {
            return ReplicaObject == rhs.ReplicaObject;
        }

        bool operator==(const TStorageElementRef<TReplicaObject>& rhs) const {
            return ReplicaObject == rhs.ReplicaObject.get();
        }

        bool Equals(const TStorageElement& rhs) const {
            return ReplicaObject.Equals(rhs.ReplicaObject);
        }

        bool Equals(const TStorageElementRef<TReplicaObject>& rhs) const {
            return ReplicaObject.Equals(rhs.ReplicaObject.get());
        }
    };

    template <typename TReplicaObject>
    struct TStorageElementRef {
        std::reference_wrapper<const TReplicaObject> ReplicaObject;
        EObjectType Type = EObjectType::PUT;

        TStorageElementRef(const TReplicaObject& replicaObject, EObjectType type = EObjectType::PUT)
            : ReplicaObject(replicaObject)
            , Type(type)
        {
        }

        TStorageElementRef(const TStorageElement<TReplicaObject>& storageElement)
            : ReplicaObject(storageElement.ReplicaObject)
            , Type(storageElement.Type)
        {
        }

        bool operator<(const TStorageElementRef& rhs) const {
            return ReplicaObject.get() < rhs.ReplicaObject.get();
        }

        bool operator==(const TStorageElementRef& rhs) const {
            return ReplicaObject.get() == rhs.ReplicaObject.get();
        }

        bool Equals(const TStorageElementRef<TReplicaObject>& rhs) const {
            return ReplicaObject.get().Equals(rhs.ReplicaObject.get());
        }
    };

    template <typename TReplicaObject>
    TVector<TStorageElementRef<TReplicaObject>> ApplyDiff(TVector<TStorageElement<TReplicaObject>>& elements,
                                                          TVector<TStorageElementRef<TReplicaObject>> diff) {
        SortUnique(elements);
        SortUnique(diff);

        TVector<TStorageElementRef<TReplicaObject>> result;
        result.reserve(elements.size() + diff.size());

        auto elementsIt = elements.begin();
        auto diffIt = diff.begin();
        while (elementsIt != elements.end() || diffIt != diff.end()) {
            if (elementsIt == elements.end()) {
                if (diffIt->Type == EObjectType::PUT) {
                    result.emplace_back(diffIt->ReplicaObject, EObjectType::PUT);
                }
                ++diffIt;
            } else if (diffIt == diff.end()) {
                result.emplace_back(elementsIt->ReplicaObject, EObjectType::PUT);
                ++elementsIt;
            } else {
                if (*elementsIt < *diffIt) {
                    result.emplace_back(elementsIt->ReplicaObject, EObjectType::PUT);
                    ++elementsIt;
                } else if (*diffIt < *elementsIt) {
                    if (diffIt->Type == EObjectType::PUT) {
                        result.emplace_back(diffIt->ReplicaObject, EObjectType::PUT);
                    }
                    ++diffIt;
                } else {
                    if (diffIt->Type == EObjectType::PUT) {
                        result.emplace_back(diffIt->ReplicaObject, EObjectType::PUT);
                    }
                    ++elementsIt;
                    ++diffIt;
                }
            }
        }

        Y_ASSERT(IsSorted(result.begin(), result.end()));

        return result;
    }

}
