#include "data_model.h"
#include "replica_objects.h"

#include <yp/cpp/yp/data_model.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/ymath.h>
#include <util/string/builder.h>

using namespace NYP;
using namespace NYPReplica;

using TEndpointReplicaObject = TEndpointReplicaObject;
using TEndpointStorageElement = TStorageElement<TEndpointReplicaObject>;
using TEndpointStorageElementRef = TStorageElementRef<TEndpointReplicaObject>;

#define UNIT_ASSERT_ENDPOINT_ID(endpoint, expectedId) \
    UNIT_ASSERT_VALUES_EQUAL(endpoint.ReplicaObject.GetObject().Meta().id(), expectedId)

#define UNIT_ASSERT_FQDN(endpoint, expectedFQDN) \
    UNIT_ASSERT_VALUES_EQUAL(endpoint.ReplicaObject.GetObject().Spec().fqdn(), expectedFQDN)

namespace {

    const TString DEFAULT_FQDN = "fqdn";

    TString GetEndpointId(int i) {
        return TStringBuilder() << "endpoint-id-" << i;
    }

    TString GetFQDN(int i) {
        return TStringBuilder() << "fqdn-" << i;
    }

    void AddEndpoint(TVector<TEndpointStorageElement>& storage, const TString& id, const TString& fqdn = DEFAULT_FQDN, EObjectType type = EObjectType::PUT) {
        NClient::TEndpoint endpoint;
        endpoint.MutableMeta()->set_id(id);
        endpoint.MutableSpec()->set_fqdn(fqdn);

        TEndpointReplicaObject replicaObject(endpoint);

        storage.emplace_back(std::move(replicaObject), type);
    }

    template <typename TReplicaObject>
    TVector<TStorageElement<TReplicaObject>> TestApplyDiff(TVector<TStorageElement<TReplicaObject>>& elements,
                                                           const TVector<TStorageElement<TReplicaObject>>& diff) {
        auto result = ApplyDiff(elements, TVector<TStorageElementRef<TReplicaObject>>(diff.begin(), diff.end()));
        return TVector<TStorageElement<TReplicaObject>>(result.begin(), result.end());
    }

} // namespace

Y_UNIT_TEST_SUITE(DataModelTests) {
    Y_UNIT_TEST(ApplyDiffTest) {
        TVector<TEndpointStorageElement> elements;
        for (size_t i = 0; i < 5; ++i) {
            AddEndpoint(elements, GetEndpointId(i));
        }

        TVector<TEndpointStorageElement> diff;
        for (size_t i = 5; i < 10; ++i) {
            AddEndpoint(diff, GetEndpointId(i));
        }

        {
            const auto result = TestApplyDiff(elements, diff);

            UNIT_ASSERT_VALUES_EQUAL(result.size(), 10);

            for (int i = 0; i < 10; ++i) {
                UNIT_ASSERT(result[i].Type == EObjectType::PUT);
                UNIT_ASSERT_ENDPOINT_ID(result[i], GetEndpointId(i));
            }
        }

        for (size_t i = 0; i < 5; ++i) {
            AddEndpoint(diff, GetEndpointId(i), DEFAULT_FQDN, EObjectType::DELETE);
        }

        Sort(diff);
        {
            const auto result = TestApplyDiff(elements, diff);

            UNIT_ASSERT_VALUES_EQUAL(result.size(), 5);

            for (int i = 5; i < 10; ++i) {
                UNIT_ASSERT(result[i - 5].Type == EObjectType::PUT);
                UNIT_ASSERT_ENDPOINT_ID(result[i - 5], GetEndpointId(i));
            }

            auto copy_result = result;

            const auto result1 = TestApplyDiff(copy_result, diff);

            UNIT_ASSERT_VALUES_EQUAL(result.size(), result1.size());
            for (size_t i = 0; i < result.size(); ++i) {
                UNIT_ASSERT(result[i].ReplicaObject == result1[i].ReplicaObject);
            }
        }

        diff.clear();
        for (int i = 0; i < 3; ++i) {
            AddEndpoint(diff, GetEndpointId(i), DEFAULT_FQDN, EObjectType::PUT);
        }
        for (int i = 3; i < 10; ++i) {
            AddEndpoint(diff, GetEndpointId(i), GetFQDN(i), EObjectType::PUT);
        }

        {
            const auto result = TestApplyDiff(elements, diff);

            UNIT_ASSERT_VALUES_EQUAL(result.size(), 10);

            for (size_t i = 0; i < elements.size(); ++i) {
                UNIT_ASSERT_FQDN(elements[i], DEFAULT_FQDN);
            }
            for (int i = 0; i < 3; ++i) {
                UNIT_ASSERT(result[i].Type == EObjectType::PUT);
                UNIT_ASSERT_ENDPOINT_ID(result[i], GetEndpointId(i));
                UNIT_ASSERT_FQDN(result[i], DEFAULT_FQDN);
            }
            for (int i = 3; i < 10; ++i) {
                UNIT_ASSERT(result[i].Type == EObjectType::PUT);
                UNIT_ASSERT_ENDPOINT_ID(result[i], GetEndpointId(i));
                UNIT_ASSERT_FQDN(result[i], GetFQDN(i));
            }
        }
    }

    Y_UNIT_TEST(CornerCases) {
        TVector<TEndpointStorageElement> elements;

        UNIT_ASSERT(TestApplyDiff(elements, elements).empty());

        for (int i = 3; i < 7; ++i) {
            const auto oldSize = elements.size();

            TVector<TEndpointStorageElement> diff;
            AddEndpoint(elements, GetEndpointId(i));

            elements = TestApplyDiff(elements, diff);
            UNIT_ASSERT_VALUES_EQUAL(elements.size(), oldSize + 1);
            UNIT_ASSERT(elements.back().Type == EObjectType::PUT);
            UNIT_ASSERT_ENDPOINT_ID(elements.back(), GetEndpointId(i));

            UNIT_ASSERT_VALUES_EQUAL(TestApplyDiff(elements, {}).size(), elements.size());
        }
        for (int i = 0; i < 3; ++i) {
            TVector<TEndpointStorageElement> diff;
            AddEndpoint(elements, GetEndpointId(i));

            UNIT_ASSERT_VALUES_EQUAL(TestApplyDiff(elements, diff).size(), elements.size());
        }
        for (int i = 7; i < 10; ++i) {
            TVector<TEndpointStorageElement> diff;
            AddEndpoint(elements, GetEndpointId(i));

            UNIT_ASSERT_VALUES_EQUAL(TestApplyDiff(elements, diff).size(), elements.size());
        }
        for (int i = 3; i < 7; ++i) {
            const auto oldSize = elements.size();

            TVector<TEndpointStorageElement> diff;
            AddEndpoint(diff, GetEndpointId(i), DEFAULT_FQDN, EObjectType::DELETE);

            elements = TestApplyDiff(elements, diff);

            UNIT_ASSERT_VALUES_EQUAL(elements.size(), oldSize - 1);
        }
    }
}
