#pragma once

#include "data_model.h"

#include <infra/libs/yp_replica/protos/events_decl.ev.pb.h>

#include <yp/cpp/yp/response_model.h>

namespace NYP::NYPReplica {

template <typename TReplicaObject>
struct TUpdate;

template <typename TReplicaObject>
THashMap<TString, TReplicaFoundUpdatedObject> MakeReplicaFoundUpdatedObjectEvents(const TVector<TUpdate<TReplicaObject>>& updates) {
    THashMap<TString, TReplicaFoundUpdatedObject> events;

    for (const auto& [oldValue, newValue] : updates) {
        if (newValue.Type == EObjectType::PUT) {
            TReplicaFoundUpdatedObject& event = events[newValue.ReplicaObject.GetKey()];
            TDiffElement& diffElement = *event.AddDiffs();
            diffElement.SetObjectId(newValue.ReplicaObject.GetObjectId());
            diffElement.SetAction(TDiffElement::PUT);
        } else {
            const TReplicaObject& oldValueRef = oldValue.GetRef();
            TReplicaFoundUpdatedObject& event = events[oldValueRef.GetKey()];
            TDiffElement& diffElement = *event.AddDiffs();
            diffElement.SetObjectId(oldValueRef.GetObjectId());
            diffElement.SetAction(TDiffElement::DELETE);
        }
    }

    for (auto& [key, event] : events) {
        event.SetKey(key);
        event.SetDiffSize(event.GetDiffs().size());
    }
    return events;
}

template <typename TReplicaObject>
TReplicaStorageUpdateObject MakeReplicaStorageUpdateObjectEvent(
    const TStringBuf columnFamily,
    const TString& objectKey,
    const TVector<TStorageElementRef<TReplicaObject>>& value
) {
    TReplicaStorageUpdateObject event;
    event.SetReplicaObjectType(TString{TReplicaObject::NAME});
    event.SetColumnFamily(TString{columnFamily});
    event.SetKey(objectKey);
    if (value.empty()) {
        event.SetAction(TReplicaStorageUpdateObject::DELETE);
    } else {
        event.SetAction(TReplicaStorageUpdateObject::UPDATE);
        event.MutableObjectIds()->Reserve(value.size());
        for (const TStorageElementRef<TReplicaObject>& element : value) {
            event.AddObjectIds(element.ReplicaObject.get().GetObjectId());
        }
    }
    return event;
}

template <typename TReplicaObject>
TReplicaWatchObjectsResult MakeReplicaWatchObjectsResultEvent(const NYP::NClient::TWatchObjectsResult& watchObjectsResult) {
    TReplicaWatchObjectsResult logEvent;
    logEvent.SetReplicaObjectType(TString{TReplicaObject::NAME});
    logEvent.MutableWatchEvents()->Reserve(watchObjectsResult.Events.size());
    for (const NYP::NClient::NApi::NProto::TEvent& event : watchObjectsResult.Events) {
        TWatchEvent& watchEvent = *logEvent.AddWatchEvents();
        watchEvent.SetObjectId(event.object_id());
        switch (event.event_type()) {
            case NYP::NClient::NApi::NProto::ET_OBJECT_CREATED:
                watchEvent.SetType(TWatchEvent::CREATED);
                break;
            case NYP::NClient::NApi::NProto::ET_OBJECT_UPDATED:
                watchEvent.SetType(TWatchEvent::UPDATED);
                break;
            case NYP::NClient::NApi::NProto::ET_OBJECT_REMOVED:
                watchEvent.SetType(TWatchEvent::REMOVED);
                break;
            default:
                ythrow yexception() << "Unexpected event type: " << NYP::NClient::NApi::NProto::EEventType_Name(event.event_type());
        }
    }
    return logEvent;
}

TReplicaUpdateColumnFamilies MakeReplicaUpdateColumnFamiliesEvent(const TVector<TString>& columnFamilies, TReplicaUpdateColumnFamilies_EAction action);

} // namespace NYP::NYPReplica
