#pragma once

#include <yp/cpp/yp/data_model.h>
#include <yp/cpp/yp/selector_result.h>

#include <util/digest/murmur.h>
#include <util/generic/maybe.h>
#include <util/generic/strbuf.h>
#include <util/generic/string.h>
#include <util/stream/str.h>
#include <util/string/builder.h>
#include <util/ysaveload.h>

namespace NYP::NYPReplica {
    template<typename TReplicaObject>
    concept KeyIsNotObjectId = !TReplicaObject::KEY_IS_OBJECT_ID;

    template<typename... TReplicaObjects>
    concept EmptySet = sizeof...(TReplicaObjects) == 0;

    template <class TObject>
    class IReplicaObject {
    public:
        IReplicaObject() = default;
        IReplicaObject(TObject object)
            : Object_(std::move(object))
        {
        }
        virtual ~IReplicaObject() = default;

        virtual void FromSelectorResult(const NYP::NClient::TSelectorResult& selectorResult) = 0;
        virtual const TString& GetKey() const = 0;
        virtual const TString& GetObjectId() const = 0;
        virtual void FillBorderFilter(TString& filter) const = 0;

        virtual TString ToString() const {
            return Object_.Serialize();
        }

        virtual void FromString(const TString& rawObject) {
            Object_.Deserialize(rawObject);
        }

        const TObject& GetObject() const {
            return Object_;
        }

        bool Equals(const IReplicaObject& rhs) const {
            return Object_ == rhs.Object_;
        }

    protected:
        TObject Object_;
    };

    class TEndpointSetReplicaObject: public IReplicaObject<NClient::TEndpointSet> {
    public:
        using TObject = NClient::TEndpointSet;

        static constexpr TStringBuf NAME = "ENDPOINT_SET";
        static constexpr TStringBuf COLUMN_FAMILY_NAME = "endpoint_set";
        static constexpr bool KEY_IS_OBJECT_ID = true;

        static const TVector<TString> ATTRIBUTE_SELECTORS;
        static const TVector<TString> WATCH_SELECTORS;

        TEndpointSetReplicaObject() = default;
        TEndpointSetReplicaObject(NClient::TEndpointSet object)
            : IReplicaObject<NClient::TEndpointSet>(std::move(object))
        {
        }

        void FromSelectorResult(const NYP::NClient::TSelectorResult& selectorResult) override final {
            selectorResult.Fill(Object_.MutableMeta(), Object_.MutableSpec(), Object_.MutableLabels(), Object_.MutableStatus());
        }

        const TString& GetKey() const override {
            return Object_.Meta().id();
        }

        const TString& GetObjectId() const override {
            return Object_.Meta().id();
        }

        void FillBorderFilter(TString& filter) const override {
            filter = TObject::TTraits::GetFilterExpression(Object_.Meta().id());
        }

        bool operator<(const TEndpointSetReplicaObject& rhs) const {
            return Object_.Meta().id() < rhs.Object_.Meta().id();
        }

        bool operator==(const TEndpointSetReplicaObject& rhs) const {
            return Object_.Meta().id() == rhs.Object_.Meta().id();
        }
    };

    class TEndpointReplicaObject: public IReplicaObject<NClient::TEndpoint> {
    public:
        using TObject = NClient::TEndpoint;

        static constexpr TStringBuf NAME = "ENDPOINT";
        static constexpr TStringBuf COLUMN_FAMILY_NAME = "endpoint";
        static constexpr bool KEY_IS_OBJECT_ID = false;

        static const TVector<TString> ATTRIBUTE_SELECTORS;
        static const TVector<TString> WATCH_SELECTORS;

        TEndpointReplicaObject() = default;
        TEndpointReplicaObject(NClient::TEndpoint object)
            : IReplicaObject<NClient::TEndpoint>(std::move(object))
        {
        }

        void FromSelectorResult(const NYP::NClient::TSelectorResult& selectorResult) override final {
            selectorResult.Fill(Object_.MutableMeta(), Object_.MutableSpec(), Object_.MutableLabels(), Object_.MutableStatus());
        }

        const TString& GetKey() const override {
            return this->Object_.Meta().Getendpoint_set_id();
        }

        const TString& GetObjectId() const override {
            return Object_.Meta().id();
        }

        void FillBorderFilter(TString& filter) const override {
            filter = TObject::TTraits::GetFilterExpression(this->Object_.Meta().endpoint_set_id(), this->Object_.Meta().id());
        }

        bool operator<(const TEndpointReplicaObject& rhs) const {
            return this->Object_.Meta().id() < rhs.Object_.Meta().id();
        }

        bool operator==(const TEndpointReplicaObject& rhs) const {
            return this->Object_.Meta().id() == rhs.Object_.Meta().id();
        }
    };

    class TPodReplicaObject: public IReplicaObject<NClient::TPod> {
    public:
        using TObject = NClient::TPod;

        static constexpr TStringBuf NAME = "POD";
        static constexpr TStringBuf COLUMN_FAMILY_NAME = "pod";
        static constexpr bool KEY_IS_OBJECT_ID = false;

        static const TVector<TString> ATTRIBUTE_SELECTORS;
        static const TVector<TString> WATCH_SELECTORS;

        TPodReplicaObject() = default;
        TPodReplicaObject(NClient::TPod object)
            : IReplicaObject<NClient::TPod>(std::move(object))
        {
        }

        void FromSelectorResult(const NYP::NClient::TSelectorResult& selectorResult) override final {
            NJson::TJsonValue shardId;
            selectorResult.Fill(
                Object_.MutableMeta()->mutable_id(),
                Object_.MutableMeta()->mutable_pod_set_id(),
                Object_.MutableSpec()->mutable_node_id(),
                Object_.MutableStatus()->mutable_ip6_address_allocations(),
                Object_.MutableStatus()->mutable_ip6_subnet_allocations(),
                Object_.MutableStatus()->mutable_dns(),
                Object_.MutableStatus()->mutable_iss_conf_summaries(),
                Object_.MutableStatus()->mutable_agent()->mutable_iss_summary(),
                Object_.MutableStatus()->mutable_agent()->mutable_pod_agent_payload()->mutable_status()->mutable_boxes(),
                &shardId
            );

            if (shardId.IsDefined()) {
                (*Object_.MutableLabels())["shard_id"] = shardId;
            }
        }

        const TString& GetKey() const override {
            return this->Object_.Meta().pod_set_id();
        }

        const TString& GetObjectId() const override {
            return Object_.Meta().id();
        }

        void FillBorderFilter(TString& filter) const override {
            filter = TObject::TTraits::GetFilterExpression(this->Object_.Meta().pod_set_id(), this->Object_.Meta().id());
        }

        bool operator<(const TPodReplicaObject& rhs) const {
            return this->Object_.Meta().id() < rhs.Object_.Meta().id();
        }

        bool operator==(const TPodReplicaObject& rhs) const {
            return this->Object_.Meta().id() == rhs.Object_.Meta().id();
        }
    };

    class TPodWithNodeIdKeyReplicaObject: public IReplicaObject<NClient::TPod> {
    public:
        using TObject = NClient::TPod;

        static constexpr TStringBuf NAME = "POD_WITH_NODE_ID_KEY";
        static constexpr TStringBuf COLUMN_FAMILY_NAME = "pod_with_node_id_key";
        static constexpr bool KEY_IS_OBJECT_ID = false;

        static const TVector<TString> ATTRIBUTE_SELECTORS;
        static const TVector<TString> WATCH_SELECTORS;

        TPodWithNodeIdKeyReplicaObject() = default;
        TPodWithNodeIdKeyReplicaObject(NClient::TPod object)
            : IReplicaObject<NClient::TPod>(std::move(object))
        {
        }

        void FromSelectorResult(const NYP::NClient::TSelectorResult& selectorResult) override final {
            selectorResult.Fill(
                Object_.MutableMeta()->mutable_id(),
                Object_.MutableMeta()->mutable_pod_set_id(),
                Object_.MutableSpec()->mutable_node_id(),
                Object_.MutableStatus()->mutable_dns()->mutable_persistent_fqdn()
            );
        }

        const TString& GetKey() const override {
            return this->Object_.Spec().node_id();
        }

        const TString& GetObjectId() const override {
            return Object_.Meta().id();
        }

        void FillBorderFilter(TString& filter) const override {
            filter = TObject::TTraits::GetFilterExpression(this->Object_.Meta().pod_set_id(), this->Object_.Meta().id());
        }

        bool operator<(const TPodWithNodeIdKeyReplicaObject& rhs) const {
            return this->Object_.Meta().id() < rhs.Object_.Meta().id();
        }

        bool operator==(const TPodWithNodeIdKeyReplicaObject& rhs) const {
            return this->Object_.Meta().id() == rhs.Object_.Meta().id();
        }
    };

    class TDnsRecordSetReplicaObject {
    public:
        using TObject = NClient::TDnsRecordSet;

        static constexpr TStringBuf NAME = "DNS_RECORD_SET";
        static constexpr TStringBuf COLUMN_FAMILY_NAME = "dns_record_set";
        static constexpr bool KEY_IS_OBJECT_ID = true;

        static const TVector<TString> ATTRIBUTE_SELECTORS;
        static const TVector<TString> WATCH_SELECTORS;

        TDnsRecordSetReplicaObject() = default;
        TDnsRecordSetReplicaObject(const NClient::TDnsRecordSet& object)
            : ObjectId_(object.Meta().id())
            , Spec_(object.Spec().SerializeAsString())
        {
            NJson::TJsonValue zone;
            object.Labels().GetValueByPath("zone", zone);
            if (zone.IsDefined()) {
                Zone_ = zone.GetStringSafe();
            }
            object.Labels().GetValueByPath("changelist", Changelist_);
        }

        void FromSelectorResult(const NYP::NClient::TSelectorResult& selectorResult) {
            NYP::NClient::NApi::NProto::TDnsRecordSetSpec spec;
            NYT::TNode zone;
            selectorResult.Fill(
                &ObjectId_,
                &spec,
                &zone,
                &Changelist_);
            if (zone.HasValue()) {
                Zone_ = zone.AsString();
            }
            Spec_ = spec.SerializeAsStringOrThrow();
        }

        bool Equals(const TDnsRecordSetReplicaObject& obj) const {
            return ObjectId_ == obj.ObjectId_ && Spec_ == obj.Spec_ && Zone_ == obj.Zone_ && Changelist_ == obj.Changelist_;
        }

        const TString& GetKey() const {
            return ObjectId_;
        }

        const TString& GetObjectId() const {
            return ObjectId_;
        }

        const TString& GetSpec() const {
            return Spec_;
        }

        bool HasZone() const {
            return Zone_.Defined();
        }

        TMaybe<TString> GetZone() const {
            return Zone_;
        }

        const TString ToString() const {
            TStringStream outputStream;
            SaveMany(&outputStream, ObjectId_, Spec_, Zone_, Changelist_);
            return outputStream.Str();
        }

        TObject GetObject() const {
            TObject object;
            object.MutableMeta()->set_id(ObjectId_);
            object.MutableSpec()->ParseFromStringOrThrow(Spec_);
            if (Zone_.Defined()) {
                object.MutableLabels()->SetValueByPath("zone", Zone_.GetRef());
            }
            if (Changelist_.IsDefined()) {
                object.MutableLabels()->SetValueByPath("changelist", Changelist_);
            }
            return object;
        }

        void FromString(const TString& rawObject) {
            TStringInput inputStream(rawObject);
            LoadMany(&inputStream, ObjectId_, Spec_, Zone_, Changelist_);
        }

        void FillBorderFilter(TString& filter) const {
            filter = TObject::TTraits::GetFilterExpression(ObjectId_);
        }

        bool operator<(const TDnsRecordSetReplicaObject& rhs) const {
            return ObjectId_ < rhs.ObjectId_;
        }

        bool operator==(const TDnsRecordSetReplicaObject& rhs) const {
            return ObjectId_ == rhs.ObjectId_;
        }

    private:
        TString ObjectId_;
        TString Spec_;
        TMaybe<TString> Zone_;
        NJson::TJsonValue Changelist_;
    };

    class TDnsZoneReplicaObject: public IReplicaObject<NClient::TDnsZone> {
    public:
        using TObject = NClient::TDnsZone;

        static constexpr TStringBuf NAME = "DNS_ZONE";
        static constexpr TStringBuf COLUMN_FAMILY_NAME = "dns_zone";
        static constexpr bool KEY_IS_OBJECT_ID = true;

        static const TVector<TString> ATTRIBUTE_SELECTORS;
        static const TVector<TString> WATCH_SELECTORS;

        TDnsZoneReplicaObject() = default;
        TDnsZoneReplicaObject(NClient::TDnsZone object)
            : IReplicaObject<NClient::TDnsZone>(std::move(object))
        {
        }

        void FromSelectorResult(const NYP::NClient::TSelectorResult& selectorResult) override final {
            selectorResult.Fill(
                Object_.MutableMeta()->mutable_id(),
                Object_.MutableSpec(),
                Object_.MutableStatus(),
                Object_.MutableLabels()
            );
        }

        const TString& GetKey() const override {
            return Object_.Meta().id();
        }

        const TString& GetObjectId() const override {
            return Object_.Meta().id();
        }

        void FillBorderFilter(TString& filter) const override {
            filter = TObject::TTraits::GetFilterExpression(Object_.Meta().id());
        }

        bool operator<(const TDnsZoneReplicaObject& rhs) const {
            return Object_.Meta().id() < rhs.Object_.Meta().id();
        }

        bool operator==(const TDnsZoneReplicaObject& rhs) const {
            return Object_.Meta().id() == rhs.Object_.Meta().id();
        }
    };
} // namespace NYP::NYPReplica
