#include "sensors.h"

#include <contrib/libs/rocksdb/include/rocksdb/statistics.h>

namespace rocksdb {
    extern const std::vector<std::pair<Tickers, std::string>> TickersNameMap;
    extern const std::vector<std::pair<Histograms, std::string>> HistogramsNameMap;
}

namespace NYP::NYPReplica::NSensors {

void TRocksDBSensorsUpdater::UpdateTickerSensors() {
    if (Statistics_->get_stats_level() <= rocksdb::StatsLevel::kExceptTickers) {
        return;
    }

    for (const auto& [tickerType, tickerName] : rocksdb::TickersNameMap) {
        TRocksDBSensorsUpdater::ESensorType sensorType = TRocksDBSensorsUpdater::SelectorSensor(tickerName, /* is ticker */ true);

        if (sensorType == TRocksDBSensorsUpdater::ESensorType::SKIPPED) {
            continue;
        }

        const ui64 value = Statistics_->getTickerCount(tickerType);
        if (sensorType == TRocksDBSensorsUpdater::ESensorType::INT_GAUGE) {
            NInfra::TIntGaugeSensor(SensorGroup_, tickerName).Set(value);
        } else {
            NInfra::TRateSensor rateSensor(SensorGroup_, tickerName);
            rateSensor.Add(value - rateSensor.Get());
        }
    }
}

void TRocksDBSensorsUpdater::UpdateHistogramSensors() {
    if (Statistics_->get_stats_level() <= rocksdb::StatsLevel::kExceptHistogramOrTimers) {
        return;
    }

    for (const auto& [histogramType, histogramName] : rocksdb::HistogramsNameMap) {
        TRocksDBSensorsUpdater::ESensorType sensorType = TRocksDBSensorsUpdater::SelectorSensor(histogramName, /* is ticker */ false);

        if (sensorType == TRocksDBSensorsUpdater::ESensorType::SKIPPED) {
            continue;
        }

        rocksdb::HistogramData histogramData;
        Statistics_->histogramData(histogramType, &histogramData);
        NInfra::TGaugeSensor(SensorGroup_, histogramName, {{"histogram", "median"}}).Set(histogramData.median);
        NInfra::TGaugeSensor(SensorGroup_, histogramName, {{"histogram", "percentile95"}}).Set(histogramData.percentile95);
        NInfra::TGaugeSensor(SensorGroup_, histogramName, {{"histogram", "percentile99"}}).Set(histogramData.percentile99);
        NInfra::TGaugeSensor(SensorGroup_, histogramName, {{"histogram", "average"}}).Set(histogramData.average);
        NInfra::TGaugeSensor(SensorGroup_, histogramName, {{"histogram", "standard_deviation"}}).Set(histogramData.standard_deviation);
        NInfra::TGaugeSensor(SensorGroup_, histogramName, {{"histogram", "max"}}).Set(histogramData.max);
        NInfra::TGaugeSensor(SensorGroup_, histogramName, {{"histogram", "count"}}).Set(histogramData.count);
        NInfra::TGaugeSensor(SensorGroup_, histogramName, {{"histogram", "sum"}}).Set(histogramData.sum);
        NInfra::TGaugeSensor(SensorGroup_, histogramName, {{"histogram", "min"}}).Set(histogramData.min);
    }
}

TRocksDBSensorsUpdater::ESensorType TRocksDBSensorsUpdater::SelectorSensor(const TStringBuf sensor, bool isTicker) {
    // BlobDB statistics are ignored because BlobDB isn't used in YPReplica
    if (sensor.StartsWith("rocksdb.blobdb")) {
        return TRocksDBSensorsUpdater::ESensorType::SKIPPED;
    }

    if (isTicker) {
        return TRocksDBSensorsUpdater::ESensorType::RATE;
    } else {
        return TRocksDBSensorsUpdater::ESensorType::GAUGE;
    }
}

} // namespace NYP::NYPReplica::NSensors
