#include "status.h"

#include <contrib/libs/rocksdb/include/rocksdb/status.h>

namespace NYP::NYPReplica {
    namespace {
        TString BuildFullMessage(TError::ECode code, TError::ESubCode subCode, const TString& message) {
            TString codeString = TString::Join("Code: [", ToString(code), "] ");
            TString subCodeString;
            if (subCode != TError::ESubCode::None) {
                subCodeString = TString::Join("Subcode: [", ToString(subCode), "] ");
            }

            return TStringBuilder() << codeString << subCodeString << message;
        }

        TString BuildVersionErrorMessage(ui64 actualVersion, ui64 expectedVersion) {
            return TString::Join("Version mismatch: expected [", ToString(expectedVersion), "], got [", ToString(actualVersion), "]");
        }

        TString BuildAgeMismatch(TDuration age, TDuration maxAge) {
            return TString::Join("Too old data: age: [", age.ToString(), "] > maxAge: [", maxAge.ToString(), "]");
        }

        TString BuildDeserializationFailedMessage(const TStringBuf objectName, const TStringBuf object) {
            return TString::Join("Failed to deserialize the ", objectName, ". Object: [", object, "]");
        }

        TString BuildColumnFamilyErrorMessage(const TStringBuf columnFamilyName, const TStringBuf message) {
            return TString::Join("Column family: [", columnFamilyName, "]. ", message);
        }

        TString BuildSemaphoreAcquireErrorMessage(const TStringBuf semaphoreSetId, const TStringBuf ypError) {
            return TString::Join("Semaphore set id : [", semaphoreSetId, "]. YpError: [", ypError, "]");
        }

        TString BuildSemaphorePingErrorMessage(const TStringBuf semaphoreSetId) {
            return TString::Join("Semaphore set id : [", semaphoreSetId, "]");
        }
    } // anonymous namespace

    TError::TError(const TError& error)
        : Code_(error.Code_)
        , SubCode_(error.SubCode_)
        , Message_(error.Message_)
        , FullMessage_(error.FullMessage_)
    {
    }

    TError::TError(const rocksdb::Status& status) {
        Y_ENSURE(!status.ok(), "Trying to construct TError from rocksdb::Status with status OK");
        Code_ = static_cast<ECode>(status.code());
        SubCode_ = static_cast<ESubCode>(status.subcode());
        Message_ = status.ToString();
        FullMessage_ = BuildFullMessage(Code_, SubCode_, Message_);
    }

    TError::TError(rocksdb::Status&& status)
        : TError(status)
    {
    }

    TError::TError(ECode code, const TStringBuf message)
        : Code_(code)
        , SubCode_(ESubCode::None)
        , Message_(ToString(message))
        , FullMessage_(BuildFullMessage(Code_, SubCode_, Message_))
    {
    }

    TError::TError(ECode code, ESubCode subCode, const TStringBuf message)
        : Code_(code)
        , SubCode_(subCode)
        , Message_(ToString(message))
        , FullMessage_(BuildFullMessage(Code_, SubCode_, Message_))
    {
    }

    TError::~TError() = default;

    bool TError::operator==(const TError& rhs) const {
        return this->Code() == rhs.Code() && this->SubCode() == rhs.SubCode();
    }

    const char* TError::what() const noexcept {
        return FullMessage_.c_str();
    }

    TError::ECode TError::Code() const {
        return Code_;
    }

    TError::ESubCode TError::SubCode() const {
        return SubCode_;
    }

    const TString& TError::Message() const {
        return Message_;
    }

    const TString& TError::FullMessage() const {
        return FullMessage_;
    }

    TStatus::TStatus()
        : TExpected<void, TError>()
    {
    }

    TStatus::TStatus(const rocksdb::Status& status)
        : TExpected<void, TError>(
            status.ok()
            ? TExpected<void, TError>::DefaultSuccess()
            : TExpected<void, TError>(TError(status))
        )
    {
    }

    TStatus::TStatus(const TError& error)
        : TExpected<void, TError>(error)
    {
    }

    TStatus::TStatus(TError&& error)
        : TExpected<void, TError>(std::move(error))
    {
    }

    TStatus::TStatus(TError::ECode code, const TStringBuf message)
        : TExpected<void, TError>(TError(code, message))
    {
    }

    TStatus::TStatus(TError::ECode code, TError::ESubCode subCode, const TStringBuf message)
        : TExpected<void, TError>(TError(code, subCode, message))
    {
    }

    bool TStatus::operator==(const TStatus& rhs) const {
        if (static_cast<bool>(*this) && static_cast<bool>(rhs)) {
            return true;
        }

        if (static_cast<bool>(*this) != static_cast<bool>(rhs)) {
            return false;
        }

        return this->Error() == rhs.Error();
    }

    IOutputStream& operator<<(IOutputStream& stream, const TStatus& status) {
        if (status) {
            stream << "OK";
        } else {
            stream << status.Error().FullMessage();
        }
        return stream;
    }

    void TStatus::ThrowIfError() const {
        if (!(*this)) {
            throw Error();
        }
    }

    TStatus TStatus::ValidateVersionFail(ui64 actualVersion, ui64 expectedVersion) {
        return TStatus(TError::ValidationError, TError::VersionMismatch, BuildVersionErrorMessage(actualVersion, expectedVersion));
    }

    TStatus TStatus::ValidateAgeFail(TDuration age, TDuration maxAge) {
        return TStatus(TError::ValidationError, TError::OldData, BuildAgeMismatch(age, maxAge));
    }

    TStatus TStatus::InvalidArgument(const TStringBuf message) {
        return TStatus(TError::InvalidArgument, message);
    }

    TStatus TStatus::DeserializationFailed(const TStringBuf objectName, const TStringBuf object) {
        return TStatus(TError::DeserializationFailed, BuildDeserializationFailedMessage(objectName, object));
    }

    TStatus TStatus::ColumnFamilyError(const TStringBuf columnFamilyName, const TStringBuf message) {
        return TStatus(TError::ColumnFamilyError, BuildColumnFamilyErrorMessage(columnFamilyName, message));
    }

    TStatus TStatus::SemaphoreAcquireError(const TStringBuf semaphoreSetId, const TStringBuf ypError) {
        return TStatus(TError::SemaphoreAcquireError, BuildSemaphoreAcquireErrorMessage(semaphoreSetId, ypError));
    }

    TStatus TStatus::SemaphorePingError(const TStringBuf semaphoreSetId) {
        return TStatus(TError::SemaphorePingError, BuildSemaphorePingErrorMessage(semaphoreSetId));
    }
} // namespace NYP::NYPReplica
