#pragma once

#include <infra/libs/outcome/result.h>

#include <util/datetime/base.h>
#include <util/generic/fwd.h>
#include <util/generic/ptr.h>
#include <util/generic/string.h>
#include <util/generic/yexception.h>
#include <util/string/builder.h>
#include <util/string/cast.h>
#include <util/system/types.h>


namespace rocksdb {
    class Status;
}

namespace NYP::NYPReplica {
    constexpr TStringBuf DATABASE_IS_NOT_OPENED = "Trying to use database but not opened before";
    constexpr TStringBuf UPDATES_DISABLED = "Trying to apply updates but updates disabled";
    constexpr TStringBuf RESTORE_OPENED_DB_ERROR_MESSAGE = "Trying to restore database but not closed before";
    constexpr TStringBuf FAILED_TO_ACQUIRE_EXCLUSIVE_LOCK_ON_DATABASE = "Failed to acquire exclusive lock on database";
    constexpr TStringBuf BACKUP_ENGINE_NOT_INITIALIZED = "BackupEngine not initialized";
    constexpr TStringBuf CHECKPOINT_NOT_CREATED_ERROR_MESSAGE = "Checkpoint is still NULL after calling create method";
    constexpr TStringBuf ROLLBACK_CONFIG_IS_NOT_SET = "RollbackConfig is not set";
    constexpr TStringBuf CANNOT_CHOOSE_SUITABLE_BACKUP_ERROR_MESSAGE = "Cannot choose suitable backup for passed arguments";
    constexpr TStringBuf UNKNOWN_ROLLBACK_OPTIONS = "Failed to rollback to backup";

    class TError : public yexception {
    public:
        enum ECode : unsigned char {
            NotFound = 1,
            Corruption = 2,
            NotSupported = 3,
            InvalidArgument = 4,
            IOError = 5,
            MergeInProgress = 6,
            Incomplete = 7,
            ShutdownInProgress = 8,
            TimedOut = 9,
            Aborted = 10,
            Busy = 11,
            Expired = 12,
            TryAgain = 13,
            CompactionTooLarge = 14,
            ColumnFamilyDropped = 15,
            DBNotOpened = 16,
            UpdatesDisabled = 17,
            DBIsOpened = 18,
            AcquireExclusiveLockFailed = 19,
            BackupEngineNotInitialized = 20,
            CheckpointNotCreated = 21,
            RollbackConfigNotSet = 22,
            NoSuitableBackup = 23,
            UnknownRollbackOptions = 24,
            ValidationError = 25,
            ReplicaStopped = 26,
            DeserializationFailed = 27,
            ColumnFamilyError = 28,
            SemaphoreAcquireError = 29,
            SemaphorePingError = 30
        };

        enum ESubCode : unsigned char {
            None = 0,
            MutexTimeout = 1,
            LockTimeout = 2,
            LockLimit = 3,
            NoSpace = 4,
            Deadlock = 5,
            StaleFile = 6,
            MemoryLimit = 7,
            SpaceLimit = 8,
            PathNotFound = 9,
            MergeOperandsInsufficientCapacity = 10,
            ManualCompactionPaused = 11,
            VersionMismatch = 12,
            OldData = 13
        };

        TError(const TError& error);

        TError(rocksdb::Status&& status);

        TError(const rocksdb::Status& status);

        TError(ECode code, const TStringBuf message = "");

        TError(ECode code, ESubCode subCode, const TStringBuf message = "");

        ~TError();

        bool operator==(const TError& rhs) const;

        const char* what() const noexcept override;

        ECode Code() const;

        ESubCode SubCode() const;

        const TString& Message() const;

        const TString& FullMessage() const;

    private:
        ECode Code_;
        ESubCode SubCode_;
        TString Message_;
        TString FullMessage_;
    };

    class TStatus : public TExpected<void, TError> {
    public:
        TStatus();

        TStatus(const rocksdb::Status& status);

        TStatus(const TError& error);

        TStatus(TError&& error);

        TStatus(TError::ECode code, const TStringBuf message = "");

        TStatus(TError::ECode code, TError::ESubCode subCode, const TStringBuf message = "");

        bool operator==(const TStatus& rhs) const;

        void ThrowIfError() const;

        static TStatus Ok() {
            return TStatus();
        }

        static TStatus NotFound() {
            return TStatus(TError::NotFound);
        }

        static TStatus DatabaseNotOpened() {
            return TStatus(TError::DBNotOpened, DATABASE_IS_NOT_OPENED);
        }

        static TStatus UpdatesDisabled() {
            return TStatus(TError::UpdatesDisabled, UPDATES_DISABLED);
        }

        static TStatus AcquireExclusiveLockFailed() {
            return TStatus(TError::AcquireExclusiveLockFailed, FAILED_TO_ACQUIRE_EXCLUSIVE_LOCK_ON_DATABASE);
        }

        static TStatus BackupEngineNotInitialized() {
            return TStatus(TError::BackupEngineNotInitialized, BACKUP_ENGINE_NOT_INITIALIZED);
        }

        static TStatus RollbackConfigNotSet() {
            return TStatus(TError::RollbackConfigNotSet, ROLLBACK_CONFIG_IS_NOT_SET);
        }

        static TStatus NoSuitableBackup() {
            return TStatus(TError::NoSuitableBackup, CANNOT_CHOOSE_SUITABLE_BACKUP_ERROR_MESSAGE);
        }

        static TStatus UnknownRollbackOptions() {
            return TStatus(TError::UnknownRollbackOptions, UNKNOWN_ROLLBACK_OPTIONS);
        }

        static TStatus CheckpointNotCreated() {
            return TStatus(TError::CheckpointNotCreated, CHECKPOINT_NOT_CREATED_ERROR_MESSAGE);
        }

        static TStatus DatabaseIsOpened() {
            return TStatus(TError::DBIsOpened, RESTORE_OPENED_DB_ERROR_MESSAGE);
        }

        static TStatus ValidateVersionFail(ui64 actualVersion, ui64 expectedVersion);

        static TStatus ValidateAgeFail(TDuration age, TDuration maxAge);

        static TStatus InvalidArgument(const TStringBuf message);

        static TStatus DeserializationFailed(const TStringBuf objectName, const TStringBuf object);

        static TStatus ColumnFamilyError(const TStringBuf columnFamilyName, const TStringBuf message);

        static TStatus SemaphoreAcquireError(const TStringBuf semaphoreSetId, const TStringBuf ypError);

        static TStatus SemaphorePingError(const TStringBuf semaphoreSetId);
    };

    IOutputStream& operator<<(IOutputStream& stream, const TStatus& status);
} //namespace NYP::NYPReplica
