#include "storage.h"

#include "data_model.h"
#include "event_listener.h"
#include "sensors.h"
#include "util.h"

#include <infra/libs/sensors/sensor.h>

#include <library/cpp/string_utils/base64/base64.h>
#include <library/cpp/threading/light_rw_lock/lightrwlock.h>

#include <contrib/libs/rocksdb/include/rocksdb/advanced_options.h>
#include <contrib/libs/rocksdb/include/rocksdb/db.h>
#include <contrib/libs/rocksdb/include/rocksdb/filter_policy.h>
#include <contrib/libs/rocksdb/include/rocksdb/slice_transform.h>
#include <contrib/libs/rocksdb/include/rocksdb/sst_file_manager.h>
#include <contrib/libs/rocksdb/include/rocksdb/table.h>
#include <contrib/libs/rocksdb/include/rocksdb/utilities/backup_engine.h>
#include <contrib/libs/rocksdb/include/rocksdb/utilities/checkpoint.h>

#include <util/generic/maybe.h>
#include <util/stream/file.h>
#include <util/stream/output.h>
#include <util/system/file_lock.h>

namespace NYP::NYPReplica {
    namespace {
        template <typename T>
        rocksdb::Slice MakeSlice(const T& value) {
            return rocksdb::Slice(value.data(), value.size());
        }
    } // anonymous namespace

    IColumnFamilyManager::~IColumnFamilyManager() = default;

    class TColumnFamilyManager final : public IColumnFamilyManager {
       friend class TStorage;
    private:
        TColumnFamilyManager(const std::vector<rocksdb::ColumnFamilyDescriptor>& columnFamilyDescriptors,
                             const std::vector<rocksdb::ColumnFamilyHandle*>& columnFamilyHandles,
                             size_t stableCFNumber
        )
            : InternalColumnFamily_(columnFamilyHandles[0])
            , DefaultColumnFamily_(columnFamilyHandles[1])
        {
            size_t cfPos = 2;

            for (; cfPos < 2 + stableCFNumber; ++cfPos) {
                StableColumnFamilies_.emplace(columnFamilyDescriptors[cfPos].name, columnFamilyHandles[cfPos]);
            }

            for (; cfPos < columnFamilyHandles.size(); ++cfPos) {
                DynamicColumnFamilies_.emplace(columnFamilyDescriptors[cfPos].name, columnFamilyHandles[cfPos]);
            }
        }

    public:
        ~TColumnFamilyManager() = default;

        TExpected<bool, TStatus> ContainsColumnFamily(const TStringBuf name) const override {
            if (StableColumnFamilies_.contains(name)) {
                return true;
            }
            TReadGuard guard(MapLock_);
            return DynamicColumnFamilies_.contains(name);
        }

        TExpected<TColumnFamilyPtr, TStatus> GetColumnFamily(const TStringBuf name) const override {
            if (auto stableCFIt = StableColumnFamilies_.FindPtr(name); stableCFIt) {
                return *stableCFIt;
            }
            TReadGuard guard(MapLock_);
            if (auto dynamicCFIt = DynamicColumnFamilies_.FindPtr(name); dynamicCFIt) {
                return *dynamicCFIt;
            } else {
                return TStatus::ColumnFamilyError(name, "Trying to get a handle on a non-existent column family");
            }
        }

        TExpected<TColumnFamilyPtr, TStatus> GetColumnFamilyNoLock(const TStringBuf name) const override {
            if (auto stableCFIt = StableColumnFamilies_.FindPtr(name); stableCFIt) {
                return *stableCFIt;
            } else if (auto dynamicCFIt = DynamicColumnFamilies_.FindPtr(name); dynamicCFIt) {
                return *dynamicCFIt;
            } else {
                return TStatus::ColumnFamilyError(name, "Trying to get a handle on a non-existent column family");
            }
        }

        TExpected<TVector<TString>, TStatus> ListColumnFamilies() const override {
            TVector<TString> columnFamilies;
            TReadGuard guard(MapLock_);
            columnFamilies.reserve(StableColumnFamilies_.size() + DynamicColumnFamilies_.size());
            for (const auto& [cfName, cfHandle] : StableColumnFamilies_) {
                columnFamilies.push_back(cfName);
            }
            for (const auto& [cfName, cfHandle] : DynamicColumnFamilies_) {
                columnFamilies.push_back(cfName);
            }
            return columnFamilies;
        }

        TExpected<rocksdb::ColumnFamilyHandle*, TStatus> GetInternalColumnFamily() const override {
            return InternalColumnFamily_.Get();
        }

    private:
        TExpected<std::pair<TVector<TColumnFamilyPtr>, TVector<TString>>, TStatus> CreateColumnFamilies(const TVector<TColumnFamilyOptions>& columnFamilies, bool ignoreExisting, const rocksdb::Options& dbOptions, rocksdb::DB& database) override {
            std::vector<rocksdb::ColumnFamilyHandle*> cfHandles;
            std::vector<std::string> createdCF;
            std::vector<rocksdb::ColumnFamilyDescriptor> columnFamilyDescriptors;
            for (TReadGuard guard(MapLock_); const auto& columnFamily : columnFamilies) {
                if (DynamicColumnFamilies_.contains(columnFamily.Name) || StableColumnFamilies_.contains(columnFamily.Name)) {
                    if (ignoreExisting) {
                        continue;
                    } else {
                        return TStatus::ColumnFamilyError(columnFamily.Name, "Trying to create existent stable or dynamic column family");
                    }
                } else {
                    createdCF.push_back(columnFamily.Name);
                    columnFamilyDescriptors.push_back(CreateColumnFamilyDescriptor(columnFamily, dbOptions));
                }
            }

            if (createdCF.empty()) {
                return std::pair<TVector<TColumnFamilyPtr>, TVector<TString>>{};
            }

            if (rocksdb::Status status = database.CreateColumnFamilies(columnFamilyDescriptors, &cfHandles); !status.ok()) {
                database.DropColumnFamilies(cfHandles);
                for (rocksdb::ColumnFamilyHandle* handle : cfHandles) {
                    database.DestroyColumnFamilyHandle(handle);
                }
                return TStatus(std::move(status));
            }

            TVector<TColumnFamilyPtr> cfPtrs;
            cfPtrs.reserve(createdCF.size());
            TVector<TString> newColumnFamilies;
            newColumnFamilies.reserve(createdCF.size());
            TWriteGuard guard(MapLock_);
            for (size_t i = 0; i < cfHandles.size(); ++i) {
                cfPtrs.push_back(DynamicColumnFamilies_.emplace(createdCF[i], cfHandles[i]).first->second);
                newColumnFamilies.push_back(std::move(createdCF[i]));
            }
            return std::make_pair(std::move(cfPtrs), std::move(newColumnFamilies));
        }

        TExpected<std::pair<TVector<TColumnFamilyPtr>, TVector<TString>>, TStatus> DeleteColumnFamilies(
            const TVector<TString>& names,
            bool ignoreNonExisting,
            std::function<TExpected<THashSet<TString>, TStatus>()> listInternalColumnFamilies,
            rocksdb::DB& database
        ) override {
            std::vector<rocksdb::ColumnFamilyHandle*> deletedCF;
            TVector<TString> deletedCFNames;
            for (TReadGuard guard(MapLock_); const auto& name : names) {
                auto cfIt = DynamicColumnFamilies_.find(name);
                if (cfIt == DynamicColumnFamilies_.end()) {
                    if (ignoreNonExisting) {
                        continue;
                    } else {
                        return TStatus::ColumnFamilyError(name, "Trying to delete non-existent dynamic column family");
                    }
                } else {
                    deletedCF.push_back(cfIt->second.Get());
                    deletedCFNames.push_back(name);
                }
            }

            if (deletedCF.empty()) {
                return std::pair<TVector<TColumnFamilyPtr>, TVector<TString>>{};
            }

            if (rocksdb::Status status = database.DropColumnFamilies(deletedCF); !status.ok()) {
                OUTCOME_TRYV(SyncDynamicColumnFamilies(listInternalColumnFamilies));
                return TStatus(std::move(status));
            }

            TVector<TColumnFamilyPtr> cfPtrs;
            cfPtrs.reserve(deletedCF.size());
            TWriteGuard guard(MapLock_);
            for (const TString& cf : deletedCFNames) {
                auto cfIt = DynamicColumnFamilies_.find(cf);
                cfPtrs.push_back(std::move(cfIt->second));
                DynamicColumnFamilies_.erase(cfIt);
            }
            return std::make_pair(std::move(cfPtrs), std::move(deletedCFNames));
        }

        TExpected<TVector<TString>, TStatus> SyncDynamicColumnFamilies(std::function<TExpected<THashSet<TString>, TStatus>()> listInternalColumnFamilies) {
            THashSet<TString> columnFamilies;
            if (auto result = listInternalColumnFamilies(); result.IsError()) {
                if (TStatus status = result.Error(); status != TStatus(TError::IOError, TError::PathNotFound)) {
                    return status;
                }
            } else {
                columnFamilies = std::move(result.Success());
            }
            TVector<TString> deletedColumnFamilies;
            for (const auto& [dynamicCFName, _] : DynamicColumnFamilies_) {
                if (!columnFamilies.contains(dynamicCFName)) {
                    deletedColumnFamilies.push_back(dynamicCFName);
                }
            }
            for (const TString& deletedCF : deletedColumnFamilies) {
                DynamicColumnFamilies_.erase(deletedCF);
            }
            return deletedColumnFamilies;
        }

    private:
        TStatus Flush(const rocksdb::FlushOptions& options, rocksdb::DB& database) const override {
            std::vector<rocksdb::ColumnFamilyHandle*> columnFamilies;

            TReadGuard guard(MapLock_);
            columnFamilies.reserve(StableColumnFamilies_.size() + DynamicColumnFamilies_.size());
            for (const auto& [cfName, cfHolder] : StableColumnFamilies_) {
                columnFamilies.push_back(cfHolder.Get());
            }
            for (const auto& [cfName, cfHolder] : DynamicColumnFamilies_) {
                columnFamilies.push_back(cfHolder.Get());
            }
            rocksdb::Status status = database.Flush(options, columnFamilies);
            return TStatus(std::move(status));
        }

    private:
        TExpected<THolder<TCache>, TStatus> CreateCache(std::function<THolder<rocksdb::Iterator>(rocksdb::ColumnFamilyHandle*)> getIterator) const override {
            TReadGuard guard(MapLock_);
            THolder<TCache> cache = MakeHolder<TCache>(StableColumnFamilies_, DynamicColumnFamilies_);
            cache->FillNoLock(StableColumnFamilies_, DynamicColumnFamilies_, getIterator);
            return cache;
        }

    private:
        TRWMutex MapLock_;
        THolder<rocksdb::ColumnFamilyHandle> InternalColumnFamily_;
        THolder<rocksdb::ColumnFamilyHandle> DefaultColumnFamily_;
        TColumnFamilyMap StableColumnFamilies_;
        TColumnFamilyMap DynamicColumnFamilies_;
    };

    class TColumnFamilyManagerReturningStatus final : public IColumnFamilyManager {
        friend class TStorage;
    public:
        ~TColumnFamilyManagerReturningStatus() = default;

        TExpected<bool, TStatus> ContainsColumnFamily(const TStringBuf /* name */) const override {
            return Status_;
        }

        TExpected<TColumnFamilyPtr, TStatus> GetColumnFamily(const TStringBuf /* name */) const override {
            return Status_;
        }

        TExpected<TColumnFamilyPtr, TStatus> GetColumnFamilyNoLock(const TStringBuf /* name */) const override {
            return Status_;
        }

        TExpected<TVector<TString>, TStatus> ListColumnFamilies() const override {
            return Status_;
        }

        TExpected<rocksdb::ColumnFamilyHandle*, TStatus> GetInternalColumnFamily() const override {
            return Status_;
        }

    private:
        TColumnFamilyManagerReturningStatus(TStatus status)
            : Status_(std::move(status))
        {
        }

    private:
        TExpected<std::pair<TVector<TColumnFamilyPtr>, TVector<TString>>, TStatus> CreateColumnFamilies(const TVector<TColumnFamilyOptions>& /* columnFamilies */, bool /* ignoreExisting */, const rocksdb::Options& /* dbOptions */, rocksdb::DB& /* database */) override {
            return Status_;
        }

        TExpected<std::pair<TVector<TColumnFamilyPtr>, TVector<TString>>, TStatus> DeleteColumnFamilies(const TVector<TString>& /* names */, bool /* ignoreNonExisting */, std::function<TExpected<THashSet<TString>, TStatus>()> /* listInternalColumnFamilies */, rocksdb::DB& /* database */) override {
            return Status_;
        }

    private:
        TStatus Flush(const rocksdb::FlushOptions& /* option */, rocksdb::DB& /* database */) const override {
            return Status_;
        }

    private:
        TExpected<THolder<TCache>, TStatus> CreateCache(std::function<THolder<rocksdb::Iterator>(rocksdb::ColumnFamilyHandle*)> /* getIterator */) const override {
            return Status_;
        }

    private:
        const TStatus Status_;
    };

    class TWriteBatch::TImpl {
    public:
        TImpl(const IColumnFamilyManager& columnFamilyManager)
            : ColumnFamilyManager_(columnFamilyManager)
        {
        }

        TStatus Put(const TStringBuf columnFamily, const TStringBuf key, const TString& value) {
            return TStatus(Batch_.Put(OUTCOME_TRYX(ColumnFamilyManager_.GetColumnFamilyNoLock(columnFamily)).Get(), MakeSlice(key), MakeSlice(value)));
        }

        TStatus Delete(const TStringBuf columnFamily, const TStringBuf key) {
            return TStatus(Batch_.Delete(OUTCOME_TRYX(ColumnFamilyManager_.GetColumnFamilyNoLock(columnFamily)).Get(), MakeSlice(key)));
        }

        TStatus UpdateTimestamp(TInstant timestamp) {
            return TStatus(Batch_.Put(OUTCOME_TRYX(ColumnFamilyManager_.GetInternalColumnFamily()), MakeSlice(ROCKSDB_TIMESTAMP_KEY), MakeSlice(ToString(timestamp.GetValue()))));
        }

        TStatus UpdateYpTimestamp(ui64 timestamp) {
            return TStatus(Batch_.Put(OUTCOME_TRYX(ColumnFamilyManager_.GetInternalColumnFamily()), MakeSlice(ROCKSDB_YP_TIMESTAMP_KEY), MakeSlice(ToString(timestamp))));
        }

        rocksdb::WriteBatch& InternalBatch() {
            return Batch_;
        }

    private:
        rocksdb::WriteBatch Batch_;
        const IColumnFamilyManager& ColumnFamilyManager_;
    };

    TWriteBatch::TWriteBatch(const IColumnFamilyManager& columnFamilyManager)
        : Impl_(MakeHolder<TImpl>(columnFamilyManager))
    {
    }

    TWriteBatch::~TWriteBatch() = default;

    TStatus TWriteBatch::UpdateTimestamp(TInstant timestamp) {
        return Impl_->UpdateTimestamp(timestamp);
    }

    TStatus TWriteBatch::UpdateYpTimestamp(ui64 timestamp) {
        return Impl_->UpdateYpTimestamp(timestamp);
    }

    TStatus TWriteBatch::Delete(const TStringBuf columnFamily, const TStringBuf key) {
        return Impl_->Delete(columnFamily, key);
    }

    TStatus TWriteBatch::Put(const TStringBuf columnFamily, const TStringBuf key, const TString& value) {
        return Impl_->Put(columnFamily, key, value);
    }

    rocksdb::WriteBatch& TWriteBatch::InternalBatch() {
        return Impl_->InternalBatch();
    }

    class TSnapshot::TImpl {
    public:
        TImpl(rocksdb::DB* db, TRWMutex& closeStorageMutex)
            : CloseStorageGuard_(closeStorageMutex)
            , Snapshot_(db)
        {
        }

        const rocksdb::Snapshot* InternalSnapshot() {
            return Snapshot_.snapshot();
        }

        void SetCachedYpTimestamp(ui64 timestamp) {
            TLightWriteGuard guard(CachedYpTimestampLock_);
            CachedYpTimestamp_ = timestamp;
        }

        TMaybe<ui64> CachedYpTimestamp() const {
            TLightReadGuard guard(CachedYpTimestampLock_);
            return CachedYpTimestamp_;
        }

    private:
        TReadGuard CloseStorageGuard_;
        rocksdb::ManagedSnapshot Snapshot_;

        TLightRWLock CachedYpTimestampLock_;
        TMaybe<ui64> CachedYpTimestamp_;
    };

    TSnapshot::TSnapshot(rocksdb::DB* db, TRWMutex& closeStorageMutex)
        : Impl_(MakeHolder<TImpl>(db, closeStorageMutex))
    {
    }

    TSnapshot::~TSnapshot() = default;

    const rocksdb::Snapshot* TSnapshot::InternalSnapshot() {
        return Impl_->InternalSnapshot();
    }

    void TSnapshot::SetCachedYpTimestamp(ui64 timestamp) {
        Impl_->SetCachedYpTimestamp(timestamp);
    }

    TMaybe<ui64> TSnapshot::CachedYpTimestamp() const {
        return Impl_->CachedYpTimestamp();
    }

    class TIterator::TImpl {
    public:
        TImpl(TColumnFamilyPtr cfHandle, rocksdb::Iterator* it)
            : ColumnFamily_(std::move(cfHandle))
            , Iterator_(it)
        {
        }

        TImpl(TColumnFamilyPtr cfHandle, TStatus status)
            : ColumnFamily_(std::move(cfHandle))
            , Status_(std::move(status))
        {
        }

        void Seek(ESeekType type, const TStringBuf key) {
            if (!Iterator_) {
                return;
            }

            switch (type) {
                case ESeekType::ToFirst:
                    SeekToFirst();
                    break;
                case ESeekType::ToLast:
                    SeekToLast();
                    break;
                case ESeekType::AtOrNext:
                    Seek(key);
                    break;
                case ESeekType::Next:
                    Seek(key);
                    if (Key() == key) {
                        Next();
                    }
                    break;
            }
        }

        void SeekToFirst() {
            if (Iterator_) {
                Iterator_->SeekToFirst();
            }
        }

        void SeekToLast() {
            if (Iterator_) {
                Iterator_->SeekToLast();
            }
        }

        void Seek(const TStringBuf key) {
            if (Iterator_) {
                Iterator_->Seek(MakeSlice(key));
            }
        }

        bool Valid() const {
            return Iterator_ && Iterator_->Valid() && Status_.IsSuccess();
        }

        TStatus Status() const {
            OUTCOME_TRYV(Status_);
            return TStatus(Iterator_->status());
        }

        void Next() {
            if (!Valid()) {
                return;
            }

            Iterator_->Next();
        }

        TStringBuf Key() const {
            if (!Valid()) {
                return {};
            }

            rocksdb::Slice key = Iterator_->key();
            return TStringBuf(key.data(), key.size());
        }

        TStringBuf Value() const {
            if (!Valid()) {
                return {};
            }

            rocksdb::Slice value = Iterator_->value();
            return TStringBuf(value.data(), value.size());
        }

        void SetStatus(TStatus status) {
            if (Status_) {
                Status_ = std::move(status);
            }
        }

    private:
        TColumnFamilyPtr ColumnFamily_;
        THolder<rocksdb::Iterator> Iterator_;
        TStatus Status_;
    };

    TIterator::TIterator(TColumnFamilyPtr cfHandle, rocksdb::Iterator* it)
        : Impl_(MakeHolder<TImpl>(std::move(cfHandle), it))
    {
    }

    TIterator::TIterator(TColumnFamilyPtr cfHandle, TStatus status)
        : Impl_(MakeHolder<TImpl>(std::move(cfHandle), std::move(status)))
    {
    }

    TIterator::~TIterator() = default;

    void TIterator::Seek(ESeekType type, const TStringBuf key) {
        Impl_->Seek(type, key);
    }

    void TIterator::SeekToFirst() {
        Impl_->SeekToFirst();
    }

    void TIterator::SeekToLast() {
        Impl_->SeekToLast();
    }

    void TIterator::Seek(const TStringBuf key) {
        Impl_->Seek(key);
    }

    bool TIterator::Valid() const {
        return Impl_->Valid();
    }

    TStatus TIterator::Status() const {
        return Impl_->Status();
    }

    void TIterator::Next() {
        Impl_->Next();
    }

    TStringBuf TIterator::Key() const {
        return Impl_->Key();
    }

    TStringBuf TIterator::RawValue() const {
        return Impl_->Value();
    }

    void TIterator::SetStatus(TStatus status) {
        Impl_->SetStatus(std::move(status));
    }

    class TStorage::TImpl {
    public:
        TImpl(TStorageOptions options)
            : Options_(std::move(options))
            , DBOptions_(Options_.CreateDBOptions())
            , ColumnFamilyManager_(new TColumnFamilyManagerReturningStatus(TStatus::DatabaseNotOpened()))
            , LockPath_(Options_.Paths.StoragePath / "LOCK.1")
        {
        }

        ~TImpl() {
            Close();
            if (Options_.DropOnDestruction) {
                Drop();
            }
        }

        TStatus DoOpen(bool validate, bool readOnly) {
            if (IsOpened()) {
                if (validate) {
                    if (TStatus status = Validate(); !status) {
                        Close();
                        return status;
                    }
                }
                return TStatus::Ok();
            }

            Options_.Paths.StoragePath.MkDirs();
            if (Options_.Paths.LogsPath) {
                Options_.Paths.LogsPath.MkDirs();
            }

            rocksdb::DB* db = nullptr;

            std::vector<rocksdb::ColumnFamilyDescriptor> columnFamilyDescriptors;
            if (auto result = GetColumnFamilyDescriptors(); result.IsError()) {
                Close();
                return result.Error();
            } else {
                columnFamilyDescriptors = std::move(result.Success());
            }

            std::vector<rocksdb::ColumnFamilyHandle*> columnFamilyHandles;
            rocksdb::Status status;
            if (readOnly) {
                status = rocksdb::DB::OpenForReadOnly(DBOptions_, Options_.Paths.StoragePath.GetPath(), columnFamilyDescriptors, &columnFamilyHandles, &db);
            } else {
                status = rocksdb::DB::Open(DBOptions_, Options_.Paths.StoragePath.GetPath(), columnFamilyDescriptors, &columnFamilyHandles, &db);
            }
            if (!status.ok()) {
                Close();
                return TStatus(std::move(status));
            }

            Database_.Reset(db);

            ColumnFamilyManager_.Reset(new TColumnFamilyManager(columnFamilyDescriptors, columnFamilyHandles, Options_.ColumnFamilies.size()));

            if (validate) {
                if (TStatus status = Validate(); !status) {
                    Close();
                    return status;
                }
            }

            if (Options_.EnableCache) {
                auto result = ColumnFamilyManager_->CreateCache([this](rocksdb::ColumnFamilyHandle* cf) {
                    THolder<rocksdb::Iterator> it{Database_->NewIterator(MakeOptions(TReadOptions()), cf)};
                    Y_ENSURE(it);
                    return it;
                });
                Cache_.Reset(std::move(result.Success()));
            }

            if (!readOnly) {
                if (TStatus updateStatus = UpdateVersion(TWriteOptions(), Options_.Meta.Version); !updateStatus) {
                    Close();
                    return updateStatus;
                }

                Lock_.Reset(new TFileLock(LockPath_.GetPath(), EFileLockType::Exclusive));
                if (!Lock_->TryAcquire()) {
                    Close();
                    return TStatus::AcquireExclusiveLockFailed();
                }
            }

            return TStatus::Ok();
        }

        TStatus Open(bool validate) {
            return DoOpen(validate, /* readOnly */ false);
        }

        TStatus OpenForReadOnly(bool validate) {
            return DoOpen(validate, /* readOnly */ true);
        }

        TStatus Reopen(bool validate) {
            if (TStatus status = Close(); !status) {
                return status;
            }
            return Open(validate);
        }

        bool IsOpened() const {
            return static_cast<bool>(Database_);
        }

        TStatus Close() {
            TWriteGuard guard(CloseMutex_);

            ColumnFamilyManager_.Reset(new TColumnFamilyManagerReturningStatus(TStatus::DatabaseNotOpened()));

            if (Database_) {
                rocksdb::Status status = Database_->Close();
                if (!status.ok()) {
                    return TStatus(std::move(status));
                }
                Database_.Reset();
            }
            if (Lock_) {
                Lock_->Release();
                Lock_.Reset();
            }
            if (Options_.EnableCache) {
                Cache_.Reset();
            }
            return TStatus::Ok();
        }

        void Drop() {
            Close();

            rocksdb::DestroyDB(Options_.Paths.StoragePath.GetPath(), DBOptions_);

            try {
                Options_.Paths.StoragePath.ForceDelete();
            } catch (...) {
            }
        }

        THolder<rocksdb::BackupEngineReadOnly> CreateBackupEngine() const {
            rocksdb::BackupEngineReadOnly* backupEngine;
            if (!rocksdb::BackupEngineReadOnly::Open(rocksdb::Env::Default(), rocksdb::BackupEngineOptions(Options_.Paths.BackupPath.GetPath()), &backupEngine).ok()) {
                return nullptr;
            }
            return THolder<rocksdb::BackupEngineReadOnly>(backupEngine);
        }

        TStatus RestoreFromBackup(TMaybe<ui64> backupId) {
            if (IsOpened()) {
                return TStatus::DatabaseIsOpened();
            }

            Options_.Paths.StoragePath.MkDirs();
            TFileLock lock(LockPath_.GetPath(), EFileLockType::Exclusive);
            if (!lock.TryAcquire()) {
                return TStatus::AcquireExclusiveLockFailed();
            }

            THolder<rocksdb::BackupEngineReadOnly> backupEngineHolder = CreateBackupEngine();
            if (!backupEngineHolder) {
                return TStatus::BackupEngineNotInitialized();
            }

            rocksdb::Status status;
            if (backupId.Defined()) {
                status = backupEngineHolder->RestoreDBFromBackup(*backupId, Options_.Paths.StoragePath.GetPath(), Options_.Paths.StoragePath.GetPath());
            } else {
                status = backupEngineHolder->RestoreDBFromLatestBackup(Options_.Paths.StoragePath.GetPath(), Options_.Paths.StoragePath.GetPath());
            }
            return TStatus(std::move(status));
        }

        TStatus RestoreFromLatestBackup() {
            return RestoreFromBackup(Nothing());
        }

        TStatus RestoreFromBackup(ui64 backupId) {
            return RestoreFromBackup(TMaybe<ui64>(backupId));
        }

        TStatus CreateCheckpoint(const TFsPath& path) {
            if (!IsOpened()) {
                return TStatus::DatabaseNotOpened();
            }

            rocksdb::Checkpoint* checkpoint = nullptr;
            rocksdb::Status status = rocksdb::Checkpoint::Create(Database_.Get(), &checkpoint);
            if (!status.ok()) {
                return TStatus(std::move(status));
            }

            if (!checkpoint) {
                return TStatus::CheckpointNotCreated();
            }

            path.Parent().MkDirs();
            path.ForceDelete();

            status = checkpoint->CreateCheckpoint(path.GetPath());
            return TStatus(std::move(status));
        }

        const TStorageOptions& Options() const {
            return Options_;
        }

        bool ValidateVersion() const {
            return GetVersion(TReadOptions()) == Options_.Meta.Version;
        }

        bool ValidateAge(const TValidationOptions& options) const {
            return !options.MaxAge || Age(TReadOptions()) <= options.MaxAge;
        }

        TStatus Validate() const {
            return Validate(Options_.ValidationOptions);
        }

        TStatus Validate(const TValidationOptions& options) const {
            if (!IsOpened()) {
                return TStatus::DatabaseNotOpened();
            }

            if (options.ValidateVersion && !ValidateVersion()) {
                return TStatus::ValidateVersionFail(GetVersion(TReadOptions()).GetOrElse(0), Options_.Meta.Version);
            }

            if (!ValidateAge(options)) {
                return TStatus::ValidateAgeFail(Age(TReadOptions()), options.MaxAge);
            }

            return TStatus::Ok();
        }

        TExpected<std::vector<rocksdb::ColumnFamilyDescriptor>, TStatus> GetColumnFamilyDescriptors() const {
            std::vector<rocksdb::ColumnFamilyDescriptor> descriptors;
            THashSet<TString> dynamicColumnFamilies;

            if (auto result = ListInternalColumnFamilies(); result.IsError()) {
                if (TStatus status = result.Error(); status != TStatus(TError::IOError, TError::PathNotFound)) {
                    return status;
                }
            } else {
                dynamicColumnFamilies = std::move(result.Success());
            }

            descriptors.push_back(CreateColumnFamilyDescriptor(TColumnFamilyOptions(TString(ROCKSDB_INTERNAL_COLUMN_FAMILY)), DBOptions_));
            dynamicColumnFamilies.erase(std::string(ROCKSDB_INTERNAL_COLUMN_FAMILY));
            descriptors.push_back(CreateColumnFamilyDescriptor(TColumnFamilyOptions(TString(ROCKSDB_NAMESPACE::kDefaultColumnFamilyName)), DBOptions_));
            dynamicColumnFamilies.erase(TString(ROCKSDB_NAMESPACE::kDefaultColumnFamilyName));
            for (const TString& columnFamily : Options_.ColumnFamilies) {
                auto columnFamilyOptions = Options_.ColumnFamilyResolver(columnFamily);
                if (Y_UNLIKELY(columnFamilyOptions.Empty())) {
                    return TStatus::ColumnFamilyError(columnFamily, "ColumnFamilyResolver couldn't find stable column family");
                }
                descriptors.push_back(CreateColumnFamilyDescriptor(columnFamilyOptions.GetRef(), DBOptions_));
                dynamicColumnFamilies.erase(columnFamily);
            }

            for (const TString& columnFamily : dynamicColumnFamilies) {
                auto columnFamilyOptions = Options_.ColumnFamilyResolver(columnFamily);
                if (Y_UNLIKELY(columnFamilyOptions.Empty())) {
                    return TStatus::ColumnFamilyError(columnFamily, "ColumnFamilyResolver couldn't find dynamic column family");
                }
                descriptors.push_back(CreateColumnFamilyDescriptor(columnFamilyOptions.GetRef(), DBOptions_));
            }

            return descriptors;
        }

        TExpected<bool, TStatus> ContainsColumnFamily(const TStringBuf name) const {
            return ColumnFamilyManager_->ContainsColumnFamily(name);
        }

        TExpected<TVector<TString>, TStatus> CreateColumnFamilies(const TVector<TColumnFamilyOptions>& columnFamilies, bool ignoreExisting) {
            for (const TColumnFamilyOptions& columnFamily : columnFamilies) {
                if (columnFamily.Name == ROCKSDB_INTERNAL_COLUMN_FAMILY || columnFamily.Name == ROCKSDB_NAMESPACE::kDefaultColumnFamilyName) {
                    return TStatus::ColumnFamilyError(columnFamily.Name, "Column families name cannot be \"default\" or \"internal\"");
                }
            }
            auto [cfHandles, newCF] = std::move(OUTCOME_TRYX(ColumnFamilyManager_->CreateColumnFamilies(columnFamilies, ignoreExisting, DBOptions_, *Database_)));
            if (Options_.EnableCache) {
                Cache_->CreateColumnFamilies(cfHandles);
            }
            return newCF;
        }

        TExpected<TVector<TString>, TStatus> DeleteColumnFamilies(const TVector<TString>& names, bool ignoreNonExisting) {
            auto listInternalColumnFamilies = [this] {
                return ListInternalColumnFamilies();
            };
            auto [cfHandles, deletedCF] = std::move(OUTCOME_TRYX(ColumnFamilyManager_->DeleteColumnFamilies(names, ignoreNonExisting, listInternalColumnFamilies, *Database_)));
            if (Options_.EnableCache) {
                Cache_->DeleteColumnFamilies(cfHandles);
            }
            return deletedCF;
        }

        TExpected<TVector<TString>, TStatus> ListColumnFamilies() const {
            return ColumnFamilyManager_->ListColumnFamilies();
        }

        TExpected<THashSet<TString>, TStatus> ListInternalColumnFamilies() const {
            std::vector<std::string> internalColumnFamilies;
            if (rocksdb::Status status = rocksdb::DB::ListColumnFamilies(DBOptions_, Options_.Paths.StoragePath.GetPath(), &internalColumnFamilies); !status.ok()) {
                return TStatus(std::move(status));
            }
            return THashSet<TString>(
                std::move_iterator(internalColumnFamilies.begin()),
                std::move_iterator(internalColumnFamilies.end())
            );
        }

        TAtomicSharedPtr<TSnapshot> CreateSnapshot() const {
            return new TSnapshot(Database_.Get(), CloseMutex_);
        }

        TIterator NewIterator(const TReadOptions& options, const TStringBuf columnFamily) const {
            if (!IsOpened()) {
                return TIterator(nullptr, TStatus::DatabaseNotOpened());
            }

            rocksdb::ReadOptions readOptions = MakeOptions(options);
            // TODO(azuremint): Возможно, нужен readOptions.auto_prefix_mode = true
            readOptions.total_order_seek = options.TotalOrderSeek;
            if (auto result = ColumnFamilyManager_->GetColumnFamily(columnFamily); result.IsError()) {
                return TIterator(nullptr, result.Error());
            } else {
                rocksdb::ColumnFamilyHandle* ptr = result.Success().Get();
                return TIterator(std::move(result.Success()), Database_->NewIterator(readOptions, ptr));
            }
        }

        TWriteBatch CreateWriteBatch() const {
            return TWriteBatch(*ColumnFamilyManager_);
        }

        TDuration Age(const TReadOptions& options) const {
            return Now() - GetTimestamp(options);
        }

    public:
        TStatus UpdateVersion(const TWriteOptions& options, ui64 version) {
            return Put(options, OUTCOME_TRYX(ColumnFamilyManager_->GetInternalColumnFamily()), ROCKSDB_VERSION_KEY, ToString(version));
        }

        TMaybe<ui64> GetVersion(const TReadOptions& options) const {
            if (TExpected<TString, TStatus> getResult = GetFromInternalCf(options, ROCKSDB_VERSION_KEY); getResult.IsError()) {
                return Nothing();
            } else {
                return FromString<ui64>(getResult.Success());
            }
        }

        TStatus UpdateTimestamp(const TWriteOptions& options, TInstant timestamp) {
            return Put(options, OUTCOME_TRYX(ColumnFamilyManager_->GetInternalColumnFamily()), ROCKSDB_TIMESTAMP_KEY, ToString(timestamp.GetValue()));
        }

        TInstant GetTimestamp(const TReadOptions& options) const {
            if (TExpected<TString, TStatus> getResult = GetFromInternalCf(options, ROCKSDB_TIMESTAMP_KEY); getResult.IsError()) {
                return TInstant::Zero();
            } else {
                return TInstant::FromValue(FromString<ui64>(getResult.Success()));
            }
        }

        TStatus UpdateYpTimestamp(const TWriteOptions& options, ui64 timestamp) {
            return Put(options, OUTCOME_TRYX(ColumnFamilyManager_->GetInternalColumnFamily()), ROCKSDB_YP_TIMESTAMP_KEY, ToString(timestamp));
        }

        ui64 GetYpTimestamp(const TReadOptions& options) const {
            if (options.Snapshot) {
                if (TMaybe<ui64> cached = options.Snapshot->CachedYpTimestamp(); cached.Defined()) {
                    return *cached;
                }
            }

            if (TExpected<TString, TStatus> getResult = GetFromInternalCf(options, ROCKSDB_YP_TIMESTAMP_KEY); getResult.IsError()) {
                return 0;
            } else {
                const ui64 result = FromString<ui64>(getResult.Success());
                if (options.Snapshot) {
                    options.Snapshot->SetCachedYpTimestamp(result);
                }
                return result;
            }
        }

        TStatus Write(const TWriteOptions& options, TWriteBatch& batch) {
            if (!Database_) {
                return TStatus::DatabaseNotOpened();
            }

            rocksdb::WriteOptions writeOptions = MakeOptions(options);
            rocksdb::Status status = Database_->Write(writeOptions, &batch.InternalBatch());

            if (status.ok() && Options_.EnableCache) {
                Cache_->Update(batch.InternalBatch());
            }

            return TStatus(std::move(status));
        }

        TStatus Delete(const TWriteOptions& options, const TStringBuf columnFamily, const TStringBuf key) {
            if (!Database_) {
                return TStatus::DatabaseNotOpened();
            }

            rocksdb::WriteOptions writeOptions = MakeOptions(options);
            TColumnFamilyPtr cfHandle = OUTCOME_TRYX(ColumnFamilyManager_->GetColumnFamily(columnFamily));
            rocksdb::Status status = Database_->Delete(writeOptions, cfHandle.Get(), MakeSlice(key));

            if (status.ok() && Options_.EnableCache) {
                Cache_->Delete(cfHandle->GetID(), ToString(key));
            }

            return TStatus(std::move(status));
        }

        TStatus Put(const TWriteOptions& options, const TStringBuf columnFamily, const TStringBuf key, const TString& value) {
            return Put(options, OUTCOME_TRYX(ColumnFamilyManager_->GetColumnFamily(columnFamily)).Get(), key, value);
        }

        TExpected<TString, TStatus> Get(const TReadOptions& options, const TStringBuf columnFamily, const TStringBuf key) const {
            return Get(options, OUTCOME_TRYX(ColumnFamilyManager_->GetColumnFamily(columnFamily)).Get(), key);
        }

        TStatus Flush(const TFlushOptions& options) {
            return ColumnFamilyManager_->Flush(MakeOptions(options), *Database_);
        }

        rocksdb::DB* InternalDB() {
            return Database_.Get();
        }

    public:
        void UpdateReplicaConfig(const TYPReplicaConfig& replicaConfig) {
            Options_.ReplicaConfig.CopyFrom(replicaConfig);
            Options_.UpdateDBOptions(DBOptions_);
            Resume();
        }

        void UpdateClusterConfig(const TYPClusterConfig& clusterConfig) {
            Options_.ClusterConfig.CopyFrom(clusterConfig);
            Options_.UpdateDBOptions(DBOptions_);
            Resume();
        }

    public:
        ui64 GetSstFilesSizeBytes() const {
            return DBOptions_.sst_file_manager->GetTotalSize();
        }

        bool SstIsMaxAllowedSpaceReached() const {
            return DBOptions_.sst_file_manager->IsMaxAllowedSpaceReached();
        }

        bool SstIsMaxAllowedSpaceReachedIncludingCompactions() const {
            return DBOptions_.sst_file_manager->IsMaxAllowedSpaceReachedIncludingCompactions();
        }

        void UpdateSensors() const {
            const ui64 storageSizeBytes = GetTotalDirectorySizeBytes(Options_.Paths.StoragePath);

            NInfra::TIntGaugeSensor(Options_.SensorGroup, NSensors::SST_FILES_SIZE).Set(GetSstFilesSizeBytes());
            NInfra::TIntGaugeSensor(Options_.SensorGroup, NSensors::STORAGE_SIZE).Set(storageSizeBytes);
            NInfra::TIntGaugeSensor(Options_.SensorGroup, NSensors::IS_MAX_ALLOWED_SPACE_REACHED).Set(SstIsMaxAllowedSpaceReached());
            NInfra::TIntGaugeSensor(Options_.SensorGroup, NSensors::IS_MAX_ALLOWED_SPACE_REACHED_INCLUDING_COMPLACTIONS).Set(SstIsMaxAllowedSpaceReachedIncludingCompactions());

            NSensors::TRocksDBSensorsUpdater updater(Options_.SensorGroup, DBOptions_.statistics);
            updater.UpdateTickerSensors();
            updater.UpdateHistogramSensors();

            Options_.UpdateSensors();
        }

        TStatus Resume() {
            return Database_->Resume();
        }

    private:
        TStatus Put(const TWriteOptions& options, rocksdb::ColumnFamilyHandle* columnFamily, const TStringBuf key, const TStringBuf value) {
            if (!Database_) {
                return TStatus::DatabaseNotOpened();
            }

            rocksdb::WriteOptions writeOptions = MakeOptions(options);
            rocksdb::Status status = Database_->Put(writeOptions, columnFamily, MakeSlice(key), MakeSlice(value));

            if (status.ok() && Options_.EnableCache) {
                Cache_->Update(columnFamily->GetID(), ToString(key), ToString(value));
            }

            return TStatus(std::move(status));
        }

        TExpected<TString, TStatus> Get(const TReadOptions& options, rocksdb::ColumnFamilyHandle* columnFamily, const TStringBuf key) const {
            if (!Database_) {
                return TStatus::DatabaseNotOpened();
            }

            if (options.UseCache && Options_.EnableCache) {
                TExpected<TString, TCache::EGetError> result = Cache_->Get(columnFamily->GetID(), TString{key});
                if (result.IsSuccess()) {
                    return std::move(result.Success());
                } else {
                    switch (result.Error()) {
                        case TCache::EGetError::NOT_FOUND:
                            return TStatus{TError::NotFound};
                        case TCache::EGetError::BUSY:
                            break;
                        case TCache::EGetError::CF_REMOVED:
                            break;
                    }
                }
            }

            rocksdb::ReadOptions readOptions = MakeOptions(options);
            std::string rawValue;
            if (TStatus status = Database_->Get(readOptions, columnFamily, MakeSlice(key), &rawValue); !status) {
                return status;
            }
            return TString(std::move(rawValue));
        }

        TExpected<TString, TStatus> GetFromInternalCf(const TReadOptions& options, const TStringBuf key) const {
            return Get(options, OUTCOME_TRYX(ColumnFamilyManager_->GetInternalColumnFamily()), key);
        }

    private:
        rocksdb::WriteOptions MakeOptions(const TWriteOptions& options) const {
            rocksdb::WriteOptions result;

            result.sync = options.Sync;
            result.disableWAL = options.DisableWAL;
            result.no_slowdown = options.NoSlowdown;

            return result;
        }

        rocksdb::ReadOptions MakeOptions(const TReadOptions& options) const {
            rocksdb::ReadOptions result;

            if (options.Snapshot) {
                result.snapshot = options.Snapshot->InternalSnapshot();
            }
            result.verify_checksums = false;

            return result;
        }

        rocksdb::FlushOptions MakeOptions(const TFlushOptions& options) const {
            rocksdb::FlushOptions result;

            result.wait = options.Wait;

            return result;
        }

    private:
        TStorageOptions Options_;

        rocksdb::Options DBOptions_;
        rocksdb::ColumnFamilyOptions ColumnFamilyOptions_;

        THolder<IColumnFamilyManager> ColumnFamilyManager_;
        THolder<rocksdb::DB> Database_;

        mutable TRWMutex CloseMutex_;

        const TFsPath LockPath_;
        THolder<TFileLock> Lock_;

        TCacheHolder Cache_;
    };

    TStorage::TStorage(TStorageOptions options)
        : Impl_(MakeHolder<TImpl>(std::move(options)))
    {
    }

    TStorage::~TStorage() = default;

    TStatus TStorage::Open(bool validate) {
        return Impl_->Open(validate);
    }

    TStatus TStorage::OpenForReadOnly(bool validate) {
        return Impl_->OpenForReadOnly(validate);
    }

    TStatus TStorage::Reopen(bool validate) {
        return Impl_->Reopen(validate);
    }

    bool TStorage::IsOpened() const {
        return Impl_->IsOpened();
    }

    TStatus TStorage::Close() {
        return Impl_->Close();
    }

    TStatus TStorage::RestoreFromLatestBackup() {
        return Impl_->RestoreFromLatestBackup();
    }

    TStatus TStorage::RestoreFromBackup(ui64 backupId) {
        return Impl_->RestoreFromBackup(backupId);
    }

    TStatus TStorage::CreateCheckpoint(const TFsPath& path) {
        return Impl_->CreateCheckpoint(path);
    }

    void TStorage::Drop() {
        return Impl_->Drop();
    }

    const TStorageOptions& TStorage::Options() const {
        return Impl_->Options();
    }

    TStatus TStorage::Validate() const {
        return Impl_->Validate();
    }

    TExpected<bool, TStatus> TStorage::ContainsColumnFamily(const TStringBuf id) const {
        return Impl_->ContainsColumnFamily(id);
    }

    TExpected<TVector<TString>, TStatus> TStorage::CreateColumnFamilies(const TVector<TColumnFamilyOptions>& columnFamilies, bool ignoreExisting) {
        return Impl_->CreateColumnFamilies(columnFamilies, ignoreExisting);
    }

    TExpected<TVector<TString>, TStatus> TStorage::DeleteColumnFamilies(const TVector<TString>& names, bool ignoreNonExisting) {
        return Impl_->DeleteColumnFamilies(names, ignoreNonExisting);
    }

    TExpected<TVector<TString>, TStatus> TStorage::ListColumnFamilies() const {
        return Impl_->ListColumnFamilies();
    }

    TAtomicSharedPtr<TSnapshot> TStorage::CreateSnapshot() const {
        return Impl_->CreateSnapshot();
    }

    TIterator TStorage::NewIterator(const TReadOptions& options, const TStringBuf columnFamily) const {
        return Impl_->NewIterator(options, columnFamily);
    }

    TWriteBatch TStorage::CreateWriteBatch() const {
        return Impl_->CreateWriteBatch();
    }

    TDuration TStorage::Age(const TReadOptions& options) const {
        return Impl_->Age(options);
    }

    TStatus TStorage::UpdateTimestamp(const TWriteOptions& options, TInstant timestamp) {
        return Impl_->UpdateTimestamp(options, timestamp);
    }

    TInstant TStorage::GetTimestamp(const TReadOptions& options) const {
        return Impl_->GetTimestamp(options);
    }

    TStatus TStorage::UpdateYpTimestamp(const TWriteOptions& options, ui64 timestamp) {
        return Impl_->UpdateYpTimestamp(options, timestamp);
    }

    ui64 TStorage::GetYpTimestamp(const TReadOptions& options) const {
        return Impl_->GetYpTimestamp(options);
    }

    TStatus TStorage::Delete(const TWriteOptions& options, const TStringBuf columnFamily, const TStringBuf key) {
        return Impl_->Delete(options, columnFamily, key);
    }

    TStatus TStorage::Write(const TWriteOptions& options, TWriteBatch& batch) {
        return Impl_->Write(options, batch);
    }

    TStatus TStorage::UpdateVersion(const TWriteOptions& options, ui64 version) {
        return Impl_->UpdateVersion(options, version);
    }

    TStatus TStorage::Put(const TWriteOptions& options, const TStringBuf columnFamily, const TStringBuf key, const TString& value) {
        return Impl_->Put(options, columnFamily, key, value);
    }

    TExpected<TString, TStatus> TStorage::Get(const TReadOptions& options, const TStringBuf columnFamily, const TStringBuf key) const {
        return Impl_->Get(options, columnFamily, key);
    }

    rocksdb::DB* TStorage::InternalDB() {
        return Impl_->InternalDB();
    }

    void TStorage::UpdateReplicaConfig(const TYPReplicaConfig& replicaConfig) {
        Impl_->UpdateReplicaConfig(replicaConfig);
    }

    void TStorage::UpdateClusterConfig(const TYPClusterConfig& clusterConfig) {
        Impl_->UpdateClusterConfig(clusterConfig);
    }

    ui64 TStorage::GetSstFilesSizeBytes() const {
        return Impl_->GetSstFilesSizeBytes();
    }

    bool TStorage::SstIsMaxAllowedSpaceReached() const {
        return Impl_->SstIsMaxAllowedSpaceReached();
    }

    bool TStorage::SstIsMaxAllowedSpaceReachedIncludingCompactions() const {
        return Impl_->SstIsMaxAllowedSpaceReachedIncludingCompactions();
    }

    void TStorage::UpdateSensors() const {
        Impl_->UpdateSensors();
    }

    TStatus TStorage::Resume() {
        return Impl_->Resume();
    }

} // namespace NYP::NYPReplica
