#pragma once

#include "cache.h"
#include "data_model.h"
#include "replica_objects.h"
#include "storage_options.h"
#include "status.h"

#include <infra/libs/sensors/sensor_group.h>
#include <infra/libs/yp_replica/protos/config/config.pb.h>
#include <infra/libs/yp_replica/protos/internal.pb.h>

#include <library/cpp/string_utils/base64/base64.h>

#include <util/datetime/base.h>
#include <util/folder/path.h>
#include <util/generic/ptr.h>
#include <util/generic/vector.h>
#include <util/system/rwlock.h>

namespace rocksdb {
    class DB;
    class WriteBatch;
    struct ColumnFamilyDescriptor;
    class ColumnFamilyHandle;
    struct ColumnFamilyOptions;
    class Iterator;
    class Snapshot;
    class TBackupEngine;
    struct FlushOptions;
    struct Options;
}

namespace NYP::NYPReplica {
    constexpr TStringBuf ROCKSDB_INTERNAL_COLUMN_FAMILY = "internal";
    constexpr TStringBuf ROCKSDB_YP_TIMESTAMP_KEY = "ROCKSDB_YP_TIMESTAMP";
    constexpr TStringBuf ROCKSDB_VERSION_KEY = "ROCKSDB_VERSION";
    constexpr TStringBuf ROCKSDB_TIMESTAMP_KEY = "ROCKSDB_TIMESTAMP";
    constexpr TStringBuf ROCKSDB_OBJECTS_NUMBER_KEY = "ROCKSDB_OBJECTS_NUMBER";

    constexpr TStringBuf ROCKSDB_META_COLUMN_FAMILY_SUFFIX = "&META";

    namespace {
        template <typename TReplicaObject>
        TString SerializeObjectsList(const TVector<TStorageElement<TReplicaObject>>& list) {
            NInternal::TObjectList protoList;
            for (const auto& ro : list) {
                auto& object = *protoList.AddList();
                object.SetValue(ro.ReplicaObject.ToString());
            }

            return protoList.SerializeAsString();
        }

        template <typename TReplicaObject>
        TString SerializeObjectsList(const TVector<TStorageElementRef<TReplicaObject>>& list) {
            NInternal::TObjectList protoList;
            for (const auto& ro : list) {
                auto& object = *protoList.AddList();
                object.SetValue(ro.ReplicaObject.get().ToString());
            }

            return protoList.SerializeAsString();
        }

        template <typename TReplicaObject>
        TExpected<TVector<TStorageElement<TReplicaObject>>, TStatus> DeserializeObjectsList(const TString& raw) {
            NInternal::TObjectList protoList;
            if (Y_UNLIKELY(!protoList.ParseFromString(raw))) {
                return TStatus::DeserializationFailed(TReplicaObject::NAME, Base64Encode(raw));
            }

            TVector<TStorageElement<TReplicaObject>> result;
            for (const auto& value : protoList.GetList()) {
                TStorageElement<TReplicaObject> object;
                object.ReplicaObject.FromString(value.GetValue());
                result.emplace_back(std::move(object));
            }

            return result;
        }
    } // anonymous namespace

    rocksdb::StatsLevel GetStatsLevel(TStorageConfig_EStatsLevel level);

    template <typename TReplicaObject>
    inline TString GetMetaColumnFamilyName() {
        return TString(TReplicaObject::COLUMN_FAMILY_NAME) + ROCKSDB_META_COLUMN_FAMILY_SUFFIX;
    }

    inline TString GetObjectsNumberKey(const TStringBuf key) {
        return TString::Join(key, '_', ROCKSDB_OBJECTS_NUMBER_KEY);
    }

    class TStorage;
    class TSnapshot;

    struct TReadOptions {
        bool UseCache = false;
        TAtomicSharedPtr<TSnapshot> Snapshot;
        // Used only in TIterator
        bool TotalOrderSeek = true;
    };

    struct TWriteOptions {
        bool Sync = false;
        bool DisableWAL = false;
        bool NoSlowdown = false;
    };

    struct TFlushOptions {
        bool Wait = true;
    };

    using TColumnFamilyPtr = TAtomicSharedPtr<rocksdb::ColumnFamilyHandle>;
    using TColumnFamilyMap = THashMap<TString, TColumnFamilyPtr>;

    class IColumnFamilyManager {
        friend class TStorage;
    public:
        virtual ~IColumnFamilyManager();

        virtual TExpected<bool, TStatus> ContainsColumnFamily(const TStringBuf name) const = 0;

        virtual TExpected<TColumnFamilyPtr, TStatus> GetColumnFamily(const TStringBuf name) const = 0;

        // Do not use this function. Need for TWriteBatch
        virtual TExpected<TColumnFamilyPtr, TStatus> GetColumnFamilyNoLock(const TStringBuf name) const = 0;

        virtual TExpected<TVector<TString>, TStatus> ListColumnFamilies() const = 0;

        virtual TExpected<rocksdb::ColumnFamilyHandle*, TStatus> GetInternalColumnFamily() const = 0;

    protected:
        virtual TExpected<std::pair<TVector<TColumnFamilyPtr>, TVector<TString>>, TStatus> CreateColumnFamilies(
            const TVector<TColumnFamilyOptions>& columnFamilies,
            bool ignoreExisting,
            const rocksdb::Options& dbOptions,
            rocksdb::DB& database
        ) = 0;

        virtual TExpected<std::pair<TVector<TColumnFamilyPtr>, TVector<TString>>, TStatus> DeleteColumnFamilies(
            const TVector<TString>& names,
            bool ignoreNonExisting,
            std::function<TExpected<THashSet<TString>, TStatus>()> listInternalColumnFamilies,
            rocksdb::DB& database
        ) = 0;

    protected:
        virtual TStatus Flush(const rocksdb::FlushOptions& options, rocksdb::DB& database) const = 0;

    protected:
        virtual TExpected<THolder<TCache>, TStatus> CreateCache(std::function<THolder<rocksdb::Iterator>(rocksdb::ColumnFamilyHandle*)> /* getIterator */) const = 0;
    };

    class TWriteBatch {
        friend class TStorage;
    public:
        ~TWriteBatch();

        template <template <class TReplicaObjectType> typename TStorageElementType, typename TReplicaObject>
        TStatus PutReplicaObjects(const TStringBuf columnFamily, const TStringBuf key, const TVector<TStorageElementType<TReplicaObject>>& value) {
            return Put(columnFamily, key, SerializeObjectsList(value));
        }

        TStatus Delete(const TStringBuf columnFamily, const TStringBuf key);

        TStatus UpdateTimestamp(TInstant timestamp);

        TStatus UpdateYpTimestamp(ui64 timestamp);

        template <typename TReplicaObject>
        TStatus UpdateObjectsNumber(const TStringBuf key, ui64 objectsNumber) {
            return PutInMetaCf<TReplicaObject>(GetObjectsNumberKey(key), ToString(objectsNumber));
        }

    private:
        TWriteBatch(const IColumnFamilyManager& columnFamilyManager);

        TStatus Put(const TStringBuf columnFamily, const TStringBuf key, const TString& value);

        template <typename TReplicaObject>
        TStatus PutInMetaCf(const TStringBuf key, const TString& value) {
            return Put(GetMetaColumnFamilyName<TReplicaObject>(), key, value);
        }

        rocksdb::WriteBatch& InternalBatch();

    private:
        class TImpl;
        THolder<TImpl> Impl_;
    };

    class TSnapshot {
        friend class TStorage;
    public:
        ~TSnapshot();

    private:
        TSnapshot(rocksdb::DB* db, TRWMutex& closeStorageMutex);

        const rocksdb::Snapshot* InternalSnapshot();

        void SetCachedYpTimestamp(ui64 timestamp);
        TMaybe<ui64> CachedYpTimestamp() const;

    private:
        class TImpl;
        THolder<TImpl> Impl_;
    };

    enum class ESeekType {
        ToFirst  /* "ToFirst" */,
        ToLast   /* "ToLast" */,
        AtOrNext /* "AtOrNext" */,
        Next     /* "Next" */,
    };

    class TIterator {
        friend class TStorage;
    private:
        explicit TIterator(TColumnFamilyPtr cfHandle, rocksdb::Iterator* it);

        explicit TIterator(TColumnFamilyPtr cfHandle, TStatus status);

    public:
        ~TIterator();

        void Seek(ESeekType type, const TStringBuf key = {});

        void SeekToFirst();

        void SeekToLast();

        void Seek(const TStringBuf key);

        bool Valid() const;

        TStatus Status() const;

        void Next();

        TStringBuf Key() const;

        template <typename TReplicaObject>
        TVector<TStorageElement<TReplicaObject>> Value() {
            auto objectsList = DeserializeObjectsList<TReplicaObject>(TString{RawValue()});
            if (Y_UNLIKELY(objectsList.IsError())) {
                SetStatus(objectsList.Error());
                return {};
            } else {
                return std::move(objectsList.Success());
            }
        }

    private:
        TStringBuf RawValue() const;

        void SetStatus(TStatus status);

    private:
        class TImpl;
        THolder<TImpl> Impl_;
    };

    class TStorage {
        friend class TBackupEngine;
        friend class TCheckpoint;
    public:
        TStorage(TStorageOptions options);
        virtual ~TStorage();

        TStatus Open(bool validate);

        TStatus OpenForReadOnly(bool validate);

        TStatus Reopen(bool validate);

        bool IsOpened() const;

        TStatus Close();

        TStatus RestoreFromLatestBackup();

        TStatus RestoreFromBackup(ui64 backupId);

        TStatus CreateCheckpoint(const TFsPath& path);

        void Drop();

        const TStorageOptions& Options() const;

        TStatus Validate() const;

        TExpected<bool, TStatus> ContainsColumnFamily(const TStringBuf name) const;

        // Non-thread safe. Cannot be used in parallel with TWriteBatch
        TExpected<TVector<TString>, TStatus> CreateColumnFamilies(const TVector<TColumnFamilyOptions>& columnFamilies, bool ignoreExisting = false);

        // Non-thread safe. Cannot be used in parallel with TWriteBatch
        TExpected<TVector<TString>, TStatus> DeleteColumnFamilies(const TVector<TString>& names, bool ignoreNonExisting = false);

        TExpected<TVector<TString>, TStatus> ListColumnFamilies() const;

        TAtomicSharedPtr<TSnapshot> CreateSnapshot() const;

        TIterator NewIterator(const TReadOptions& options, const TStringBuf columnFamily) const;

        // Non-thread safe. Cannot be used TWriteBatch in parallel with functions that 
        // modify the column family list (CreateColumnFamilies, DeleteColumnFamilies)
        TWriteBatch CreateWriteBatch() const;

        TDuration Age(const TReadOptions& options) const;

        TStatus UpdateTimestamp(const TWriteOptions& options, TInstant timestamp);
        TInstant GetTimestamp(const TReadOptions& options) const;

        TStatus UpdateYpTimestamp(const TWriteOptions& options, ui64 timestamp);
        ui64 GetYpTimestamp(const TReadOptions& options) const;

        template <typename TReplicaObject>
        TStatus UpdateObjectsNumber(const TWriteOptions& options, const TStringBuf key, ui64 objectsNumber) {
            return PutInMetaCf<TReplicaObject>(options, GetObjectsNumberKey(key), ToString(objectsNumber));
        }

        template <typename TReplicaObject>
        ui64 GetObjectsNumber(const TReadOptions& options, const TStringBuf key) const {
            if (TExpected<TString, TStatus> getResult = GetFromMetaCf<TReplicaObject>(options, GetObjectsNumberKey(key)); getResult.IsError()) {
                return 0;
            } else {
                return FromString<ui64>(getResult.Success());
            }
        }

        TStatus Write(const TWriteOptions& options, TWriteBatch& batch);

        TStatus Delete(const TWriteOptions& options, const TStringBuf columnFamily, const TStringBuf key);

        template <template <class TReplicaObjectType> typename TStorageElementType, typename TReplicaObject>
        TStatus PutReplicaObjects(const TWriteOptions& options, const TStringBuf columnFamily, const TStringBuf key, const TVector<TStorageElementType<TReplicaObject>>& value) {
            return Put(options, columnFamily, key, SerializeObjectsList(value));
        }

        template <typename TReplicaObject>
        TExpected<TVector<TStorageElement<TReplicaObject>>, TStatus> GetReplicaObjects(const TReadOptions& options, const TStringBuf columnFamily, const TStringBuf key) const {
            return DeserializeObjectsList<TReplicaObject>(std::move(OUTCOME_TRYX(Get(options, columnFamily, key))));
        }

    public:
        void UpdateReplicaConfig(const TYPReplicaConfig& replicaConfig);

        void UpdateClusterConfig(const TYPClusterConfig& clusterConfig);

    public:
        ui64 GetSstFilesSizeBytes() const;

        bool SstIsMaxAllowedSpaceReached() const;

        bool SstIsMaxAllowedSpaceReachedIncludingCompactions() const;

        void UpdateSensors() const;

        TStatus Resume();

    protected:
        TStatus UpdateVersion(const TWriteOptions& options, ui64 version);

        TStatus Put(const TWriteOptions& options, const TStringBuf columnFamily, const TStringBuf key, const TString& value);
        TExpected<TString, TStatus> Get(const TReadOptions& options, const TStringBuf columnFamily, const TStringBuf key) const;

        template <typename TReplicaObject>
        TStatus PutInMetaCf(const TWriteOptions& options, const TStringBuf key, const TString& value) {
            return Put(options, GetMetaColumnFamilyName<TReplicaObject>(), key, value);
        }

        template <typename TReplicaObject>
        TExpected<TString, TStatus> GetFromMetaCf(const TReadOptions& options, const TStringBuf key) const {
            return Get(options, GetMetaColumnFamilyName<TReplicaObject>(), key);
        }

    private:
        rocksdb::DB* InternalDB();

    private:
        class TImpl;
        THolder<TImpl> Impl_;
    };

} // namespace NYP::NYPReplica
