#include "storage_options.h"

#include "event_listener.h"
#include "sensors.h"

#include <contrib/libs/rocksdb/include/rocksdb/db.h>
#include <contrib/libs/rocksdb/include/rocksdb/filter_policy.h>
#include <contrib/libs/rocksdb/include/rocksdb/options.h>
#include <contrib/libs/rocksdb/include/rocksdb/slice_transform.h>
#include <contrib/libs/rocksdb/include/rocksdb/sst_file_manager.h>
#include <contrib/libs/rocksdb/include/rocksdb/statistics.h>
#include <contrib/libs/rocksdb/include/rocksdb/table.h>

namespace NYP::NYPReplica {

rocksdb::StatsLevel GetStatsLevel(TRocksDBConfig_EStatsLevel level) {
    switch (level) {
    case TRocksDBConfig_EStatsLevel_DISABLE_ALL:
        return rocksdb::StatsLevel::kDisableAll;
    case TRocksDBConfig_EStatsLevel_EXCEPT_TICKERS:
        return rocksdb::StatsLevel::kExceptTickers;
    case TRocksDBConfig_EStatsLevel_EXCEPT_HISTOGRAM_OR_TIMERS:
        return rocksdb::StatsLevel::kExceptHistogramOrTimers;
    case TRocksDBConfig_EStatsLevel_EXCEPT_TIMERS:
        return rocksdb::StatsLevel::kExceptTimers;
    case TRocksDBConfig_EStatsLevel_EXCEPT_DETAILED_TIMERS:
        return rocksdb::StatsLevel::kExceptDetailedTimers;
    case TRocksDBConfig_EStatsLevel_EXCEPT_TIME_FOR_MUTEX:
        return rocksdb::StatsLevel::kExceptTimeForMutex;
    case TRocksDBConfig_EStatsLevel_ALL:
        return rocksdb::StatsLevel::kAll;
    }
}

auto GetIndexType(TBlockTableConfig_EIndexType indexType) {
    switch (indexType) {
    case TBlockTableConfig_EIndexType_BINARY_SEARCH:
        return rocksdb::BlockBasedTableOptions::IndexType::kBinarySearch;
    case TBlockTableConfig_EIndexType_HASH_SEARCH:
        return rocksdb::BlockBasedTableOptions::IndexType::kHashSearch;
    case TBlockTableConfig_EIndexType_TWO_LEVEL_INDEX_SEARCH:
        return rocksdb::BlockBasedTableOptions::IndexType::kTwoLevelIndexSearch;
    case TBlockTableConfig_EIndexType_BINARY_SEARCH_WITH_FIRST_KEY:
        return rocksdb::BlockBasedTableOptions::IndexType::kBinarySearchWithFirstKey;
    }
}

rocksdb::BlockBasedTableOptions GetBlockTableOptions(const TBlockTableConfig& blockTableConfig) {
    rocksdb::BlockBasedTableOptions tableOptions;
    tableOptions.block_restart_interval = blockTableConfig.GetBlockRestartInterval();
    tableOptions.no_block_cache = blockTableConfig.GetNoBlockCache();
    tableOptions.index_type = GetIndexType(blockTableConfig.GetIndexType());
    // TODO: move to protobuf
    tableOptions.filter_policy.reset(rocksdb::NewBloomFilterPolicy(10, true));
    return tableOptions;
}

TStoragePaths::TStoragePaths(const TStorageMeta& meta, const TYPReplicaConfig& replicaConfig)
    : StoragePath(TFsPath(replicaConfig.GetStorageConfig().GetPath()) / meta.Name)
    , BackupPath(TFsPath(replicaConfig.GetBackupConfig().GetPath()) / meta.Name)
    , LogsPath(replicaConfig.GetStorageConfig().HasLogsPath() ? TFsPath(replicaConfig.GetStorageConfig().GetLogsPath()) / meta.Name : "")
{
}

TValidationOptions::TValidationOptions(const TStorageValidationConfig& config)
    : MaxAge(TDuration::Parse(config.GetMaxAge()))
{
}

TColumnFamilyOptions GetColumnFamilyOptions(const TYPReplicaConfig& ReplicaConfig, const TYPClusterConfig& ClusterConfig) {
    const auto& replicaRocksDBConfig = ReplicaConfig.GetStorageConfig().GetRocksDBConfig();
    const auto& clusterRocksDBConfig = ClusterConfig.GetRocksDBConfig();

    return TColumnFamilyOptions("", clusterRocksDBConfig.HasBlockTableConfig()
        ? clusterRocksDBConfig.GetBlockTableConfig()
        : replicaRocksDBConfig.GetBlockTableConfig()
    );
}

rocksdb::ColumnFamilyDescriptor CreateColumnFamilyDescriptor(const TColumnFamilyOptions& columnFamilyOptions, const rocksdb::Options& dbOptions) {
    rocksdb::ColumnFamilyOptions options = rocksdb::ColumnFamilyOptions(dbOptions);
    options.table_factory.reset(rocksdb::NewBlockBasedTableFactory(GetBlockTableOptions(columnFamilyOptions.BlockTableConfig)));
    return rocksdb::ColumnFamilyDescriptor(std::string(columnFamilyOptions.Name), options);
}

TColumnFamilyResolveFunc DefaultColumnFamilyResolveFunc() {
    return [](const TString& columnFamilyName) {
        return TColumnFamilyOptions(columnFamilyName);
    };
}

TColumnFamilyResolveFunc MergeColumnFamilyResolverFuncs(TVector<TColumnFamilyResolveFunc> resolveFuncs) {
    return [resolveFuncs = std::move(resolveFuncs)](const TString& columnFamilyName) -> TMaybe<TColumnFamilyOptions> {
        for (const auto& resolveFunc : resolveFuncs) {
            const auto& options = resolveFunc(columnFamilyName);
            if (options.Defined()) {
                return options;
            }
        }
        return Nothing();
    };
}

TStorageOptions::TStorageOptions(
    TStorageMeta meta,
    TVector<TString> columnFamilies,
    TColumnFamilyResolveFunc columnFamilyResolver,
    const TYPClusterConfig& clusterConfig,
    const TYPReplicaConfig& replicaConfig,
    const NInfra::TSensorGroup sensorGroup
)
    : Meta(std::move(meta))
    , ColumnFamilies(std::move(columnFamilies))
    , ColumnFamilyResolver(std::move(columnFamilyResolver))
    , ClusterConfig(clusterConfig)
    , ReplicaConfig(replicaConfig)
    , Paths(Meta, ReplicaConfig)
    , EnableCache(ReplicaConfig.GetStorageConfig().GetEnableCache())
    , ValidationOptions(ClusterConfig.HasStorageValidationConfig()
        ? ClusterConfig.GetStorageValidationConfig()
        : ReplicaConfig.GetStorageConfig().GetValidationConfig())
    , SensorGroup(sensorGroup, NSensors::STORAGE)
{
}

rocksdb::Options TStorageOptions::CreateDBOptions() const {
    rocksdb::Options dbOptions;

    const auto& replicaRocksDBConfig = ReplicaConfig.GetStorageConfig().GetRocksDBConfig();

    dbOptions.allow_concurrent_memtable_write = replicaRocksDBConfig.GetAllowConcurrentMemtableWrite();
    dbOptions.allow_mmap_reads = replicaRocksDBConfig.GetAllowMmapReads();
    dbOptions.allow_mmap_writes = replicaRocksDBConfig.GetAllowMmapWrites();
    dbOptions.atomic_flush = replicaRocksDBConfig.GetAtomicFlush();
    dbOptions.bloom_locality = replicaRocksDBConfig.GetBloomLocality();
    // TODO: move to protobuf
    dbOptions.compression = rocksdb::CompressionType::kNoCompression;
    dbOptions.create_if_missing = replicaRocksDBConfig.GetCreateIfMissing();
    dbOptions.create_missing_column_families = replicaRocksDBConfig.GetCreateMissingColumnFamilies();
    dbOptions.db_log_dir = Paths.LogsPath.GetPath();
    dbOptions.db_write_buffer_size = replicaRocksDBConfig.GetDBWriteBufferSize();
    dbOptions.keep_log_file_num = replicaRocksDBConfig.GetKeepLogFileNum();
    dbOptions.max_log_file_size = replicaRocksDBConfig.GetMaxLogFileSize();
    dbOptions.max_open_files = replicaRocksDBConfig.GetMaxOpenFiles();
    dbOptions.max_total_wal_size = replicaRocksDBConfig.GetMaxTotalWalSize();
    dbOptions.paranoid_checks = replicaRocksDBConfig.GetParanoidChecks();
    dbOptions.skip_checking_sst_file_sizes_on_db_open = replicaRocksDBConfig.GetSkipCheckingSstFileSizesOnDbOpen();

    dbOptions.listeners.push_back(CreateSpaceLimitEventListener());

    // TODO: move to protobuf
    dbOptions.memtable_factory.reset(rocksdb::NewHashLinkListRepFactory(200000));

    dbOptions.prefix_extractor.reset(rocksdb::NewCappedPrefixTransform(replicaRocksDBConfig.GetCappedPrefixLen()));

    dbOptions.sst_file_manager.reset(rocksdb::NewSstFileManager(rocksdb::Env::Default(), rocksdb::FileSystem::Default()));
    dbOptions.sst_file_manager->SetMaxAllowedSpaceUsage(GetMaxAllowedSpaceUsageBytes());
    dbOptions.sst_file_manager->SetCompactionBufferSize(GetCompactionBufferSizeBytes());

    dbOptions.statistics = rocksdb::CreateDBStatistics();

    dbOptions.table_factory.reset(rocksdb::NewBlockBasedTableFactory(GetBlockTableOptions(replicaRocksDBConfig.GetBlockTableConfig())));

    UpdateDBOptions(dbOptions);

    return dbOptions;
}

void TStorageOptions::UpdateDBOptions(rocksdb::Options& dbOptions) const {
    const auto& replicaRocksDBConfig = ReplicaConfig.GetStorageConfig().GetRocksDBConfig();

    dbOptions.sst_file_manager->SetMaxAllowedSpaceUsage(GetMaxAllowedSpaceUsageBytes());
    dbOptions.sst_file_manager->SetCompactionBufferSize(GetCompactionBufferSizeBytes());

    dbOptions.statistics->set_stats_level(GetStatsLevel(replicaRocksDBConfig.GetStatsLevel()));
}

ui64 TStorageOptions::GetMaxAllowedSpaceUsageBytes() const {
    const auto& replicaRocksDBConfig = ReplicaConfig.GetStorageConfig().GetRocksDBConfig();
    const auto& clusterRocksDBConfig = ClusterConfig.GetRocksDBConfig();

    return clusterRocksDBConfig.GetMaxAllowedSpaceUsageBytes() > 0
        ? clusterRocksDBConfig.GetMaxAllowedSpaceUsageBytes()
        : replicaRocksDBConfig.GetMaxAllowedSpaceUsageBytes();
}

ui64 TStorageOptions::GetCompactionBufferSizeBytes() const {
    const auto& replicaRocksDBConfig = ReplicaConfig.GetStorageConfig().GetRocksDBConfig();
    const auto& clusterRocksDBConfig = ClusterConfig.GetRocksDBConfig();

    return clusterRocksDBConfig.GetCompactionBufferSizeBytes() > 0
        ? clusterRocksDBConfig.GetCompactionBufferSizeBytes()
        : replicaRocksDBConfig.GetCompactionBufferSizeBytes();
}

TColumnFamilyOptions TStorageOptions::GetColumnFamilyOptions() const {
    return ::NYP::NYPReplica::GetColumnFamilyOptions(ReplicaConfig, ClusterConfig);
}

TDuration TStorageOptions::GetAgeAlertThreshold() const {
    return TDuration::Parse(ReplicaConfig.GetStorageConfig().GetAgeAlertThreshold());
}

void TStorageOptions::UpdateSensors() const {
    NInfra::TIntGaugeSensor(SensorGroup, NSensors::MAX_ALLOWED_SPACE_USAGE).Set(GetMaxAllowedSpaceUsageBytes());
    NInfra::TIntGaugeSensor(SensorGroup, NSensors::COMPACTION_BUFFER_SIZE).Set(GetCompactionBufferSizeBytes());
}

} // namespace NYP::NYPReplica
