#pragma once

#include <infra/libs/sensors/sensor_group.h>
#include <infra/libs/yp_replica/protos/config/config.pb.h>

#include <util/datetime/base.h>
#include <util/folder/path.h>

namespace rocksdb {
    struct BlockBasedTableOptions;
    struct ColumnFamilyDescriptor;
    struct Options;
    enum StatsLevel : ui8;
}

namespace NYP::NYPReplica {

rocksdb::StatsLevel GetStatsLevel(TRocksDBConfig_EStatsLevel level);

auto GetIndexType(TBlockTableConfig_EIndexType indexType);

rocksdb::BlockBasedTableOptions GetBlockTableOptions(const TBlockTableConfig& blockTableConfig = TBlockTableConfig());

struct TStorageMeta {
    TString Name;
    ui64 Version = 0;
};

struct TStoragePaths {
    TStoragePaths() = default;
    TStoragePaths(const TStorageMeta& meta, const TYPReplicaConfig& replicaConfig);

    TFsPath StoragePath;
    TFsPath BackupPath;
    TFsPath LogsPath;
};

struct TValidationOptions {
    TValidationOptions() = default;
    TValidationOptions(const TStorageValidationConfig& config);

    bool ValidateVersion = true;

    TDuration MaxAge;
};

struct TColumnFamilyOptions {
    TColumnFamilyOptions(TString name, TBlockTableConfig blockTableConfig = TBlockTableConfig())
        : Name(std::move(name))
        , BlockTableConfig(std::move(blockTableConfig))
    {
    }

    TString Name;
    TBlockTableConfig BlockTableConfig;
};

TColumnFamilyOptions GetColumnFamilyOptions(const TYPReplicaConfig& ReplicaConfig, const TYPClusterConfig& ClusterConfig);

rocksdb::ColumnFamilyDescriptor CreateColumnFamilyDescriptor(const TColumnFamilyOptions& columnFamilyOptions, const rocksdb::Options& dbOptions);

using TColumnFamilyResolveFunc = std::function<TMaybe<TColumnFamilyOptions>(const TString&)>;

TColumnFamilyResolveFunc DefaultColumnFamilyResolveFunc();

TColumnFamilyResolveFunc MergeColumnFamilyResolverFuncs(TVector<TColumnFamilyResolveFunc> resolveFuncs);

struct TStorageOptions {
public:
    TStorageOptions() = default;
    TStorageOptions(
        TStorageMeta meta,
        TVector<TString> columnFamilies,
        TColumnFamilyResolveFunc columnFamilyResolver,
        const TYPClusterConfig& clusterConfig,
        const TYPReplicaConfig& replicaConfig,
        const NInfra::TSensorGroup sensorGroup
    );

    rocksdb::Options CreateDBOptions() const;

    void UpdateDBOptions(rocksdb::Options& dbOptions) const;

    ui64 GetMaxAllowedSpaceUsageBytes() const;

    ui64 GetCompactionBufferSizeBytes() const;

    TColumnFamilyOptions GetColumnFamilyOptions() const;

    TDuration GetAgeAlertThreshold() const;

    void UpdateSensors() const;

public:
    TStorageMeta Meta;

    TVector<TString> ColumnFamilies;
    TColumnFamilyResolveFunc ColumnFamilyResolver = DefaultColumnFamilyResolveFunc();

    TYPClusterConfig ClusterConfig;
    TYPReplicaConfig ReplicaConfig;

    TStoragePaths Paths;

    bool EnableCache = false;
    TValidationOptions ValidationOptions;

    bool DropOnDestruction = false;

    NInfra::TSensorGroup SensorGroup = NInfra::TSensorGroup(TString());
};

} // namespace NYP::NYPReplica
