#include "backup.h"
#include "replica_objects.h"
#include "status.h"
#include "storage.h"

#include <yp/cpp/yp/data_model.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/map.h>
#include <util/generic/maybe.h>
#include <util/generic/xrange.h>
#include <util/random/random.h>
#include <util/random/shuffle.h>
#include <util/string/builder.h>

using namespace NYP::NYPReplica;
using namespace NYP::NClient;

namespace {
    constexpr TStringBuf STORAGE_NAME = "test-storage";
    const TFsPath STORAGE_PATH = "storage";
    const TFsPath BACKUP_PATH = "backup";
    constexpr ui64 STORAGE_VERSION = 1;

    TVector<TString> GetDefaultColumnFamilies() {
        return {
            TString{TEndpointReplicaObject::COLUMN_FAMILY_NAME},
            TString{GetMetaColumnFamilyName<TEndpointReplicaObject>()},
            TString{TDnsRecordSetReplicaObject::COLUMN_FAMILY_NAME},
            TString{GetMetaColumnFamilyName<TDnsRecordSetReplicaObject>()},
        };
    }

    TStorageOptions GetDefaultStorageOptions(TStringBuf storageName = STORAGE_NAME, const TVector<TString>& columnFamilies = GetDefaultColumnFamilies()) {
        TStorageOptions options;
        options.ColumnFamilies = columnFamilies;
        options.Meta = TStorageMeta({TString(storageName), STORAGE_VERSION});
        options.Paths.StoragePath = STORAGE_PATH / storageName;
        options.Paths.BackupPath = BACKUP_PATH / storageName;
        return options;
    }

    TBackupEngineOptions GetDefaultBackupEngineOptions(TStringBuf storageName = STORAGE_NAME) {
        TBackupEngineOptions options;
        options.Path = BACKUP_PATH / storageName;
        return options;
    }

    void Cleanup(const TFsPath& path) {
        try {
            path.ForceDelete();
        } catch (...) {
        }
    }

    void Cleanup(const TFsPath& path, const TFsPath& paths...) {
        Cleanup(path);
        Cleanup(paths);
    }

    void Cleanup(const TStorageOptions& options) {
        Cleanup(options.Paths.StoragePath, options.Paths.BackupPath);
    }

    void Cleanup(TStorage& storage) {
        storage.Drop();
    }

    class TDropGuard {
    public:
        TDropGuard(TStorage& storage)
            : Storage_(&storage)
        {
        }

        TDropGuard(TStorageOptions options)
            : StorageOptions_(options)
        {
        }

        ~TDropGuard() {
            if (Storage_) {
                Cleanup(*Storage_);
            }
            Cleanup(StorageOptions_);
        }

    private:
        TStorage* Storage_ = nullptr;
        const TStorageOptions StorageOptions_;

    };

    template <typename TReplicaObject>
    using TStorageValue = TVector<TStorageElement<TReplicaObject>>;

    template <typename TReplicaObject>
    struct IStorageElementsGenerator {
        virtual std::pair<TString, TStorageValue<TReplicaObject>> operator() (TMaybe<TString> key = Nothing()) const = 0;
    };

    struct TRandomDnsRecordSetGenerator: IStorageElementsGenerator<TDnsRecordSetReplicaObject> {
        std::pair<TString, TStorageValue<TDnsRecordSetReplicaObject>> operator() (TMaybe<TString> key = Nothing()) const override {
            TDnsRecordSet recordSet;
            if (!key.Defined()) {
                recordSet.MutableMeta()->set_id(NUnitTest::RandomString(1 + RandomNumber<size_t>(63), RandomNumber<ui32>()));
                key = recordSet.Meta().id();
            } else {
                recordSet.MutableMeta()->set_id(*key);
            }

            TVector<NApi::NProto::TDnsRecordSetSpec::TResourceRecord::EType> types = {
                NApi::NProto::TDnsRecordSetSpec::TResourceRecord::A,
                NApi::NProto::TDnsRecordSetSpec::TResourceRecord::AAAA,
                NApi::NProto::TDnsRecordSetSpec::TResourceRecord::PTR,
                NApi::NProto::TDnsRecordSetSpec::TResourceRecord::SRV,
            };

            size_t recordsNumber = RandomNumber<size_t>(64);
            for (size_t i = 0; i < recordsNumber; ++i) {
                auto& record = *recordSet.MutableSpec()->add_records();
                record.set_data(NUnitTest::RandomString(30 + RandomNumber<size_t>(10)));
                record.set_type(types[RandomNumber<size_t>(types.size())]);
                record.set_class_("IN");
                record.set_ttl(RandomNumber<ui64>());
            }

            return {*key, TVector<TStorageElement<TDnsRecordSetReplicaObject>>({TStorageElement<TDnsRecordSetReplicaObject>(recordSet)})};
        }
    };

    template <typename TReplicaObject>
    using TInMemoryColumnFamily = TMap<TString, TStorageValue<TReplicaObject>>;

    template <typename TReplicaObject>
    using TInMemoryStorage = TMap<TString, TInMemoryColumnFamily<TReplicaObject>>;

    template <typename TReplicaObject>
    using TCheckFunction = std::function<void(const TStorage&, const TInMemoryStorage<TReplicaObject>&, TStatus expected)>;

    template <typename TReplicaObject>
    TVector<TString> ListKeys(const TInMemoryStorage<TReplicaObject>& storage, const TString& columnFamilyName = TString(TReplicaObject::COLUMN_FAMILY_NAME)) {
        TVector<TString> keys;
        keys.reserve(storage.at(columnFamilyName).size());
        for (const auto& [key, value] : storage.at(columnFamilyName)) {
            keys.emplace_back(key);
        }
        return keys;
    }

    template <typename TReplicaObject>
    TVector<TString> ListColumnFamilies(const TInMemoryStorage<TReplicaObject>& storage) {
        TVector<TString> columnFamilies;
        columnFamilies.reserve(storage.size());
        for (const auto& [columnFamily, value] : storage) {
            columnFamilies.emplace_back(columnFamily);
        }
        return columnFamilies;
    }

    template <typename TReplicaObject>
    void GenerateAndPut(TStorage& storage, TInMemoryStorage<TReplicaObject>& inMemStorage, const IStorageElementsGenerator<TReplicaObject>& generator, ui64 n, TStatus expected = TStatus(), const TString& columnFamilyName = TString(TReplicaObject::COLUMN_FAMILY_NAME)) {
        for (ui64 i = 0; i < n; ++i) {
            auto [key, value] = generator();
            if (expected) {
                inMemStorage[columnFamilyName][key] = value;
            }
            UNIT_ASSERT_VALUES_EQUAL(storage.PutReplicaObjects(TWriteOptions(), columnFamilyName, key, value), expected);
        }
    }

    template <typename TReplicaObject>
    TVector<TString> DeleteRandomKey(TStorage& storage, TInMemoryStorage<TReplicaObject>& inMemStorage, const IStorageElementsGenerator<TReplicaObject>& generator, ui64 n, TStatus expected = TStatus(), bool restore = false, const TString& columnFamilyName = TString(TReplicaObject::COLUMN_FAMILY_NAME)) {
        TVector<TString> keys = ListKeys(inMemStorage, columnFamilyName);
        ShuffleRange(keys);
        keys.resize(Min(n, keys.size()));
        TVector<TString> deletedKeys;
        for (const TString& key : keys) {
            if (expected) {
                inMemStorage.at(columnFamilyName).erase(key);
            }
            UNIT_ASSERT_VALUES_EQUAL(storage.Delete(TWriteOptions(), columnFamilyName, key), expected);
            if (expected && restore) {
                if (RandomNumber<bool>()) {
                    auto [genKey, value] = generator(key);
                    UNIT_ASSERT_VALUES_EQUAL(genKey, key);
                    inMemStorage[columnFamilyName][key] = value;
                    UNIT_ASSERT(storage.PutReplicaObjects(TWriteOptions(), columnFamilyName, key, value));
                } else {
                    deletedKeys.push_back(key);
                }
            }
        }
        return deletedKeys;
    }

    template <typename TReplicaObject>
    void DeleteAndRestoreRandomKey(TStorage& storage, TInMemoryStorage<TReplicaObject>& inMemStorage, ui64 n, TStatus expected = TStatus(), const TString& columnFamilyName = TString(TReplicaObject::COLUMN_FAMILY_NAME)) {
        TVector<TString> existentKeys = ListKeys(inMemStorage, columnFamilyName);
        ShuffleRange(existentKeys);
        existentKeys.resize(Min(n, existentKeys.size()));
        for (ui64 i = 0; i < Min(n, existentKeys.size()); ++i) {
            if (expected) {
                inMemStorage.at(columnFamilyName).erase(existentKeys[i]);
            }
            UNIT_ASSERT_VALUES_EQUAL(storage.Delete(TWriteOptions(), columnFamilyName, existentKeys[i]), expected);
        }
    }

    template <typename TReplicaObject>
    void GenerateAndPutWithoutLocalRecording(TStorage& storage, const IStorageElementsGenerator<TReplicaObject>& generator, ui64 n, TStatus expected = TStatus(), const TString& columnFamilyName = TString(TReplicaObject::COLUMN_FAMILY_NAME)) {
        for (ui64 i = 0; i < n; ++i) {
            auto [key, value] = generator();
            UNIT_ASSERT_VALUES_EQUAL(storage.PutReplicaObjects(TWriteOptions(), columnFamilyName, key, value), expected);
        }
    }

    template <typename TReplicaObject>
    void RegenerateAndPut(TStorage& storage, TInMemoryStorage<TReplicaObject>& inMemStorage, const IStorageElementsGenerator<TReplicaObject>& generator, const TVector<TString>& keys, const TString& columnFamilyName = TString(TReplicaObject::COLUMN_FAMILY_NAME)) {
        for (const TString& key : keys) {
            auto [genKey, value] = generator(key);
            UNIT_ASSERT_VALUES_EQUAL(genKey, key);
            UNIT_ASSERT(inMemStorage[columnFamilyName].contains(key));
            inMemStorage[columnFamilyName][key] = value;
            UNIT_ASSERT(storage.PutReplicaObjects(TWriteOptions(), columnFamilyName, key, value));
        }
    }

    template <typename TReplicaObject>
    void CheckKeyNotFound(const TStorage& storage, const TVector<TString>& keys, const TString& columnFamily, bool UseCache = false) {
        for (const TString& key : keys) {
            TReadOptions options = TReadOptions();
            options.UseCache = UseCache;
            auto getResult = storage.template GetReplicaObjects<TReplicaObject>(options, columnFamily, key);
            UNIT_ASSERT(getResult.IsError());
            UNIT_ASSERT_EQUAL(getResult.Error(), TStatus::NotFound());
        }
    }

    template <typename TReplicaObject>
    void PutUpdateDeletePipeline(const IStorageElementsGenerator<TReplicaObject>& generator, const TCheckFunction<TReplicaObject>& check, bool EnableCache = false) {
        TStorageOptions options = GetDefaultStorageOptions();
        options.EnableCache = EnableCache;
        TStorage storage(options);
        TDropGuard g(storage);

        TInMemoryStorage<TReplicaObject> values;
        for (size_t n : {1, 100}) {
            GenerateAndPut(storage, values, generator, n, /* expectedResult */ TStatus::DatabaseNotOpened());
            check(storage, values, TStatus::DatabaseNotOpened());
        }

        UNIT_ASSERT(storage.Open(/* validate */ false));

        for (size_t n : {1, 100, 1000}) {
            GenerateAndPut(storage, values, generator, n);
            check(storage, values, TStatus::Ok());
        }

        UNIT_ASSERT(storage.Close());

        for (size_t n : {1, 100}) {
            TInMemoryStorage<TReplicaObject> tmp;
            GenerateAndPut(storage, tmp, generator, n, /* expectedResult */ TStatus::DatabaseNotOpened());
            check(storage, tmp, TStatus::DatabaseNotOpened());
        }

        UNIT_ASSERT(storage.Open(/* validate */ true));

        check(storage, values, TStatus::Ok());

        for (size_t n : {1000, 100, 1}) {
            GenerateAndPut(storage, values, generator, n);
            check(storage, values, TStatus::Ok());
        }

        TVector<TString> existentKeys = ListKeys(values);
        for (size_t n : {1, 10, 100, 1000}) {
            TVector<TString> keys(n);
            for (size_t i = 0; i < n; ++i) {
                keys[i] = existentKeys[(i * n) % existentKeys.size()];
            }
            RegenerateAndPut(storage, values, generator, keys);
            check(storage, values, TStatus::Ok());
        }

        TString columnFamily(TReplicaObject::COLUMN_FAMILY_NAME);
        for (size_t n : {1, 10, 100, 1000}) {
            TVector<TString> keys(n + n);
            for (size_t i = 0; i < n; ++i) {
                keys[i] = existentKeys[(i * n) % existentKeys.size()];
            }
            for (size_t i = 0; i < n; ++i) {
                keys[i + n] = generator().first;
            }
            ShuffleRange(keys);

            for (const TString& key : keys) {
                UNIT_ASSERT(storage.Delete(TWriteOptions(), columnFamily, key));
                values.at(columnFamily).erase(key);
            }
            CheckKeyNotFound<TReplicaObject>(storage, keys, columnFamily, EnableCache);
            check(storage, values, TStatus::Ok());
        }
    }

    template <typename TReplicaObject>
    void CheckCFEqual(const TInMemoryStorage<TReplicaObject>& actual, TIterator& it, const TString& columnFamily, TStatus expected = TStatus::Ok()) {
        it.SeekToFirst();
        size_t sizeCF = 0;
        for (;  it.Valid(); it.Next(), ++sizeCF) {
            UNIT_ASSERT(actual.at(columnFamily).at(TString{it.Key()}) == it.Value<TReplicaObject>());
        }
        if (!expected) {
            UNIT_ASSERT_EQUAL(it.Status(), expected);
        } else {
            UNIT_ASSERT(sizeCF == actual.at(columnFamily).size());
            UNIT_ASSERT(it.Status());
        }
    }

    template <typename TReplicaObject>
    void CheckStoragesEqual(const TStorage& storage, const TInMemoryStorage<TReplicaObject>& actual, bool UseCache = false, TStatus expected = TStatus::Ok()) {
        TReadOptions readOptions = TReadOptions();
        readOptions.UseCache = UseCache;
        TVector<TString> columnFamilies = ListColumnFamilies(actual);
        for (const TString& columnFamily : columnFamilies) {
            TVector<TString> keys = ListKeys(actual, columnFamily);

            ShuffleRange(keys);
            for (const TString& key : keys) {
                auto getResult = storage.template GetReplicaObjects<TReplicaObject>(readOptions, columnFamily, key);
                UNIT_ASSERT(getResult.IsSuccess());
                UNIT_ASSERT_EQUAL(getResult.Success(), actual.at(columnFamily).at(key));
            }

            TIterator it = storage.NewIterator(readOptions, columnFamily);
            CheckCFEqual(actual, it, columnFamily, expected);
        }
    }

    template <typename TReplicaObject>
    void CreateColumnFamilies(TStorage& storage, TInMemoryStorage<TReplicaObject>& inMemoryStorage, const TVector<TColumnFamilyOptions>& columnFamilies, TStatus expected = TStatus()) {
        if (expected) {
            UNIT_ASSERT(storage.CreateColumnFamilies(columnFamilies).IsSuccess());
        } else {
            auto status = storage.CreateColumnFamilies(columnFamilies);
            UNIT_ASSERT(status.IsError());
            UNIT_ASSERT_VALUES_EQUAL(status.Error(), expected);
        }
        if (expected) {
            for (const TColumnFamilyOptions& cf : columnFamilies) {
                inMemoryStorage[cf.Name] = TInMemoryColumnFamily<TReplicaObject>();
                UNIT_ASSERT(storage.ContainsColumnFamily(cf.Name).Success());
            }
        }
    }

    template <typename TReplicaObject>
    void DeleteColumnFamilies(TStorage& storage, TInMemoryStorage<TReplicaObject>& inMemoryStorage, TInMemoryStorage<TReplicaObject>& deletedCF, const TVector<TString>& columnFamilies, TStatus expected = TStatus()) {
        if (expected) {
            UNIT_ASSERT(storage.DeleteColumnFamilies(columnFamilies).IsSuccess());
        } else {
            auto status = storage.DeleteColumnFamilies(columnFamilies);
            UNIT_ASSERT(status.IsError());
            UNIT_ASSERT_VALUES_EQUAL(status.Error(), expected);
        }
        if (expected) {
            for (const TString& columnFamily : columnFamilies) {
                deletedCF[columnFamily] = inMemoryStorage.at(columnFamily);
                inMemoryStorage.erase(columnFamily);
                UNIT_ASSERT(!storage.ContainsColumnFamily(columnFamily).Success());
            }
        }
    }

    void CheckListColumnFamilies(const TStorage& storage, TVector<TString> expectedColumnFamilies) {
        TVector<TString> columnFamilies;
        auto listResult = storage.ListColumnFamilies();
        if (listResult.IsSuccess()) {
            columnFamilies = listResult.Success();
        }
        Sort(columnFamilies);
        Sort(expectedColumnFamilies);
        UNIT_ASSERT_EQUAL(columnFamilies, expectedColumnFamilies);
    }

    template <typename TReplicaObject>
    void CheckListColumnFamilies(const TStorage& storage, const TInMemoryStorage<TReplicaObject>& inMemoryStorage) {
        CheckListColumnFamilies(storage, ListColumnFamilies(inMemoryStorage));
    }

    template <typename TReplicaObject>
    void FullCheckStoragesEqual(const TStorage& storage, const TInMemoryStorage<TReplicaObject>& inMemoryStorage, bool UseCache = false) {
        CheckStoragesEqual(storage, inMemoryStorage, UseCache);
        CheckListColumnFamilies(storage, inMemoryStorage);
    }
} // anonymous namespace

Y_UNIT_TEST_SUITE(ReplicaStorageStartupTests) {
    /**
     * Check various combinations of Open, Close, Drop. Test IsOpened().
     */
    Y_UNIT_TEST(OpenAndClose) {
        SetRandomSeed(42);
        const TStorageOptions options = GetDefaultStorageOptions();

        UNIT_ASSERT(!options.Paths.StoragePath.Exists());
        UNIT_ASSERT(!options.Paths.BackupPath.Exists());

        TStorage storage(options);
        TDropGuard g(storage);
        for (size_t i = 0; i < 10; ++i) {
            UNIT_ASSERT(!storage.IsOpened());
            UNIT_ASSERT(!storage.Open(/* validate */ true)); // Incorrect version
            UNIT_ASSERT(options.Paths.StoragePath.Exists());
            UNIT_ASSERT(!options.Paths.BackupPath.Exists());
        }
        UNIT_ASSERT(!storage.IsOpened());

        for (bool validate : {false, true, false, true}) {
            for (size_t i = 0; i < 2; ++i) {
                for (size_t j = 0; j < 10; ++j) {
                    UNIT_ASSERT(storage.Open(validate));
                    UNIT_ASSERT(storage.IsOpened());
                    UNIT_ASSERT(options.Paths.StoragePath.Exists());
                    UNIT_ASSERT(!options.Paths.BackupPath.Exists());
                }
                UNIT_ASSERT(storage.Close());
                UNIT_ASSERT(!storage.IsOpened());
                UNIT_ASSERT(options.Paths.StoragePath.Exists());
                UNIT_ASSERT(!options.Paths.BackupPath.Exists());
            }
        }

        storage.Drop();
        UNIT_ASSERT(!options.Paths.StoragePath.Exists());
        UNIT_ASSERT(!options.Paths.BackupPath.Exists());

        UNIT_ASSERT(!storage.Open(/* validate */ true));
        UNIT_ASSERT(!storage.IsOpened());
        UNIT_ASSERT(options.Paths.StoragePath.Exists());
        UNIT_ASSERT(!options.Paths.BackupPath.Exists());

        UNIT_ASSERT(storage.Open(/* validate */ false));
        UNIT_ASSERT(storage.IsOpened());
        UNIT_ASSERT(options.Paths.StoragePath.Exists());
        UNIT_ASSERT(!options.Paths.BackupPath.Exists());

        storage.Drop();
        UNIT_ASSERT(!storage.IsOpened());
        UNIT_ASSERT(!options.Paths.StoragePath.Exists());
        UNIT_ASSERT(!options.Paths.BackupPath.Exists());
    }

    /**
     * 1. Restore from empty backup fails.
     * 2. Restore from backup fails for opened storage.
     * 3. Successful restore from valid backup.
     */
    Y_UNIT_TEST(RestoreFromLatestBackup) {
        SetRandomSeed(42);
        const TStorageOptions storageOptions = GetDefaultStorageOptions();
        TStorage storage(storageOptions);
        TDropGuard g(storage);

        UNIT_ASSERT(!storage.RestoreFromLatestBackup());
        UNIT_ASSERT(!storage.IsOpened());

        UNIT_ASSERT(storage.Open(/* validate */ false));
        UNIT_ASSERT(!storage.RestoreFromLatestBackup());

        TEndpoint endpoint;
        endpoint.MutableMeta()->set_id("eid-1");
        endpoint.MutableMeta()->set_endpoint_set_id("esid-1");
        TEndpointReplicaObject putEndpointObject(endpoint);
        TVector<TStorageElement<TEndpointReplicaObject>> putEndpointObjects;
        putEndpointObjects.emplace_back(putEndpointObject);

        const TStringBuf key = putEndpointObject.GetKey();
        storage.PutReplicaObjects(TWriteOptions(), TEndpointReplicaObject::COLUMN_FAMILY_NAME, key, putEndpointObjects);

        {
            const TBackupEngineOptions backupEngineOptions = GetDefaultBackupEngineOptions();
            TBackupEngine backupEngine(backupEngineOptions);
            UNIT_ASSERT(backupEngine.Open());
            UNIT_ASSERT(backupEngine.CreateNewBackup(TCreateBackupOptions(), &storage));
        }

        // storage already successfully opened
        UNIT_ASSERT(!storage.RestoreFromLatestBackup());

        UNIT_ASSERT(storage.Close());
        UNIT_ASSERT(storage.RestoreFromLatestBackup());
        // RestoreFromLatestBackup just prepares files, but doesn't "Open" storage
        UNIT_ASSERT(!storage.IsOpened());

        UNIT_ASSERT(storage.Open(/* validate */ true));
        UNIT_ASSERT(storage.IsOpened());

        auto getResult = storage.template GetReplicaObjects<TEndpointReplicaObject>(TReadOptions(), TEndpointReplicaObject::COLUMN_FAMILY_NAME, key);
        UNIT_ASSERT(getResult.IsSuccess());
        TVector<TStorageElement<TEndpointReplicaObject>> gotEndpointObjects = std::move(getResult.Success());
        UNIT_ASSERT(putEndpointObjects == gotEndpointObjects);
    }

    /**
     * Only one storage can be opened on a path.
     */
     Y_UNIT_TEST(UniqueStorageOnPath) {
        SetRandomSeed(42);
        const TStorageOptions storageOptions = GetDefaultStorageOptions();
        TStorage firstStorage(storageOptions);
        TDropGuard gFirst(firstStorage);
        TStorage secondStorage(storageOptions);
        TDropGuard gSecond(secondStorage);

        UNIT_ASSERT(!firstStorage.IsOpened());
        UNIT_ASSERT(!secondStorage.IsOpened());

        UNIT_ASSERT(firstStorage.Open(/* validate */ false));
        UNIT_ASSERT(!secondStorage.Open(/* validate */ false));
        UNIT_ASSERT(firstStorage.IsOpened());
        UNIT_ASSERT(!secondStorage.IsOpened());

        UNIT_ASSERT(firstStorage.Close());
        UNIT_ASSERT(!firstStorage.IsOpened());
        UNIT_ASSERT(secondStorage.Open(/* validate */ true));
        UNIT_ASSERT(secondStorage.IsOpened());

        UNIT_ASSERT(!firstStorage.Open(/* validate */ false));
        UNIT_ASSERT(!firstStorage.IsOpened());

        // Create backup for opened secondStorage, try to restore from backup for firstStorage - must fail
        {
            const TBackupEngineOptions backupEngineOptions = GetDefaultBackupEngineOptions();
            TBackupEngine backupEngine(backupEngineOptions);
            UNIT_ASSERT(backupEngine.Open());
            UNIT_ASSERT(backupEngine.CreateNewBackup(TCreateBackupOptions(), &secondStorage));
        }
        UNIT_ASSERT(secondStorage.IsOpened());
        UNIT_ASSERT(!firstStorage.IsOpened());
        UNIT_ASSERT(!firstStorage.RestoreFromLatestBackup());
     }
}

Y_UNIT_TEST_SUITE(ReplicaStorageValidationTests) {
    Y_UNIT_TEST(SkipValidation) {
        SetRandomSeed(42);
        TStorageOptions options = GetDefaultStorageOptions();
        options.ValidationOptions.ValidateVersion = false;
        options.ValidationOptions.MaxAge = TDuration::Zero();

        TStorage storage(options);
        TDropGuard g(storage);
        UNIT_ASSERT(storage.Open(/* validate */ true));
        UNIT_ASSERT(storage.IsOpened());
        storage.Drop();

        UNIT_ASSERT(storage.Open(/* validate */ false));
        UNIT_ASSERT(storage.IsOpened());
    }

    Y_UNIT_TEST(ValidateVersion) {
        SetRandomSeed(42);
        TStorageOptions options = GetDefaultStorageOptions();
        options.ValidationOptions.ValidateVersion = true;
        options.ValidationOptions.MaxAge = TDuration::Zero();
        TDropGuard g(options);

        {
            options.Meta.Version = 1;

            TStorage storage(options);
            UNIT_ASSERT(!storage.Validate());
            UNIT_ASSERT(!storage.Open(/* validate */ true));
            UNIT_ASSERT(!storage.Validate());
            UNIT_ASSERT(storage.Open(/* validate */ false));
            UNIT_ASSERT(storage.Validate());
        }

        {
            options.Meta.Version = 2;

            TStorage storage(options);
            UNIT_ASSERT(!storage.Open(/* validate */ true));
            UNIT_ASSERT(storage.Open(/* validate */ false));
            UNIT_ASSERT(storage.Validate());
        }

        {
            options.Meta.Version = 2;

            TStorage storage(options);
            UNIT_ASSERT(storage.Open(/* validate */ true));
            UNIT_ASSERT(storage.Validate());
        }
    }

    Y_UNIT_TEST(ValidateAge) {
        SetRandomSeed(42);
        TStorageOptions options = GetDefaultStorageOptions();
        options.ValidationOptions.ValidateVersion = false;
        TDropGuard g(options);

        const TDuration maxAge = TDuration::Seconds(3);
        {
            options.ValidationOptions.MaxAge = maxAge;

            TStorage storage(options);
            UNIT_ASSERT(!storage.Validate());
            UNIT_ASSERT(!storage.Open(/* validate */ true));
            UNIT_ASSERT(storage.Open(/* validate */ false));
            UNIT_ASSERT(!storage.Validate());
            UNIT_ASSERT(storage.UpdateTimestamp(TWriteOptions(), TInstant::Now()));
            UNIT_ASSERT(storage.Validate());
        }

        {
            TStorage storage(options);
            Sleep(maxAge + TDuration::Seconds(1));
            UNIT_ASSERT(!storage.Open(/* validate */ true));
        }

        {
            options.ValidationOptions.MaxAge = 10 * maxAge;

            TStorage storage(options);
            UNIT_ASSERT(storage.Open(/* validate */ true));
            UNIT_ASSERT(storage.Validate());
        }

        {
            options.ValidationOptions.MaxAge = TDuration::Zero();

            TStorage storage(options);
            UNIT_ASSERT(storage.Open(/* validate */ true));
            UNIT_ASSERT(storage.Validate());
        }
    }
}

Y_UNIT_TEST_SUITE(TestModificationMethods) {
    Y_UNIT_TEST(TestTimestamp) {
        SetRandomSeed(42);
        TStorageOptions options = GetDefaultStorageOptions();
        TStorage storage(options);
        TDropGuard g(storage);
        UNIT_ASSERT(storage.Open(/* validate */ false));

        for (size_t i = 0; i < 10; ++i) {
            UNIT_ASSERT_VALUES_EQUAL(TInstant::Zero(), storage.GetTimestamp(TReadOptions()));
        }

        auto checkAgeEqualsNow = [](TStorage& storage) {
            TDuration now = Now() - TInstant::Zero();
            TDuration age = storage.Age(TReadOptions());
            UNIT_ASSERT_LE(now, age);
            UNIT_ASSERT_LE(age, Now() - TInstant::Zero());
        };

        for (size_t i = 0; i < 10; ++i) {
            checkAgeEqualsNow(storage);
        }

        for (size_t i = 0; i < 10; ++i) {
            TInstant timestamp = Now();
            UNIT_ASSERT(storage.UpdateTimestamp(TWriteOptions(), timestamp));

            for (size_t j = 0; j < 10; ++j) {
                UNIT_ASSERT_VALUES_EQUAL(timestamp, storage.GetTimestamp(TReadOptions()));
            }
        }

        TInstant timestamp = Now();
        UNIT_ASSERT(storage.UpdateTimestamp(TWriteOptions(), timestamp));
        UNIT_ASSERT(storage.Close());

        for (size_t i = 0; i < 10; ++i) {
            UNIT_ASSERT(!storage.UpdateTimestamp(TWriteOptions(), Now()));
            UNIT_ASSERT_VALUES_EQUAL(TInstant::Zero(), storage.GetTimestamp(TReadOptions()));
            checkAgeEqualsNow(storage);
        }

        UNIT_ASSERT(storage.Open(/* validate */ false));
        UNIT_ASSERT_VALUES_EQUAL(timestamp, storage.GetTimestamp(TReadOptions()));
    }

    void TestUint64Methods(const std::function<ui64(const TStorage&)>& get, const std::function<TStatus(TStorage&, ui64)>& update) {
        TStorageOptions options = GetDefaultStorageOptions();
        TStorage storage(options);
        TDropGuard g(storage);
        UNIT_ASSERT(storage.Open(/* validate */ false));

        for (size_t i = 0; i < 10; ++i) {
            UNIT_ASSERT_VALUES_EQUAL(0, get(storage));
        }

        for (size_t i = 0; i < 10; ++i) {
            ui64 value = RandomNumber<ui64>();
            UNIT_ASSERT(update(storage, value));

            for (size_t j = 0; j < 10; ++j) {
                UNIT_ASSERT_VALUES_EQUAL(value, get(storage));
            }
        }

        ui64 value = RandomNumber<ui64>();
        UNIT_ASSERT(update(storage, value));
        UNIT_ASSERT(storage.Close());

        for (size_t i = 0; i < 10; ++i) {
            UNIT_ASSERT(!update(storage, value));
            UNIT_ASSERT_VALUES_EQUAL(0, get(storage));
        }

        UNIT_ASSERT(storage.Open(/* validate */ false));
        UNIT_ASSERT_VALUES_EQUAL(value, get(storage));
    }

    Y_UNIT_TEST(TestYpTimestamp) {
        SetRandomSeed(42);
        auto get = [](const TStorage& storage) {
            return storage.GetYpTimestamp(TReadOptions());
        };

        auto update = [](TStorage& storage, ui64 ypTimestamp) {
            return storage.UpdateYpTimestamp(TWriteOptions(), ypTimestamp);
        };

        TestUint64Methods(get, update);
    }

    Y_UNIT_TEST(TestObjectsNumber) {
        SetRandomSeed(42);
        auto get = [](const TStorage& storage) {
            return storage.GetObjectsNumber<TEndpointReplicaObject>(TReadOptions(), "abacaba");
        };

        auto update = [](TStorage& storage, ui64 objectsNumber) {
            return storage.UpdateObjectsNumber<TEndpointReplicaObject>(TWriteOptions(), "abacaba", objectsNumber);
        };

        TestUint64Methods(get, update);
    }
}

Y_UNIT_TEST_SUITE(TestPutAndGetReplicaObjects) {
    template <typename TReplicaObject>
    void TestPutAndGetRandomReplicaObjects(const IStorageElementsGenerator<TReplicaObject>& generator, bool EnableCache = false) {
        PutUpdateDeletePipeline<TReplicaObject>(generator, [&EnableCache](const TStorage& storage, const TInMemoryStorage<TReplicaObject>& actual, TStatus expected = TStatus::Ok()) {
            CheckStoragesEqual(storage, actual, EnableCache, expected);
        }, EnableCache);
    }

    Y_UNIT_TEST(TestDnsRecordSet) {
        SetRandomSeed(42);
        TestPutAndGetRandomReplicaObjects(TRandomDnsRecordSetGenerator(), /* EnableCache */ false);
        TestPutAndGetRandomReplicaObjects(TRandomDnsRecordSetGenerator(), /* EnableCache */ true);
    }

    // TODO Y_UNIT_TEST(TestEndpoint)
}

Y_UNIT_TEST_SUITE(TestIterator) {
    Y_UNIT_TEST(EmptyStorage) {
        SetRandomSeed(42);
        TStorageOptions options = GetDefaultStorageOptions();
        TStorage storage(options);
        TDropGuard g(storage);

        TIterator it = storage.NewIterator(TReadOptions(), TDnsRecordSetReplicaObject::COLUMN_FAMILY_NAME);
        it.SeekToFirst();
        for (size_t i = 0; i < 10; ++i) {
            UNIT_ASSERT(!it.Valid());
            UNIT_ASSERT_EQUAL(it.Status(), TStatus::DatabaseNotOpened());
            UNIT_ASSERT(it.Key().empty());
            UNIT_ASSERT(it.Value<TDnsRecordSetReplicaObject>().empty());
            UNIT_ASSERT(it.Value<TEndpointReplicaObject>().empty());
            it.Next();
        }

        UNIT_ASSERT(storage.Open(/* validate */ false));

        TIterator jt = storage.NewIterator(TReadOptions(), TDnsRecordSetReplicaObject::COLUMN_FAMILY_NAME);
        jt.SeekToFirst();

        for (TIterator* iterator : {&jt, &it}) {
            for (size_t i = 0; i < 10; ++i) {
                UNIT_ASSERT(!iterator->Valid());
                UNIT_ASSERT_EQUAL(it.Status(), TStatus::DatabaseNotOpened());
                UNIT_ASSERT(iterator->Key().empty());
                UNIT_ASSERT(iterator->Value<TDnsRecordSetReplicaObject>().empty());
                UNIT_ASSERT(iterator->Value<TEndpointReplicaObject>().empty());
                iterator->Next();
            }
        }
    }

    template <typename TReplicaObject>
    void CheckIterationOrder(const TStorage& storage, size_t expectedIterations, const TString& columnFamily, TStatus expectedStatus = TStatus::Ok()) {
        TString lastKey;
        size_t iterations = 0;

        TIterator it = storage.NewIterator(TReadOptions(), columnFamily);
        it.SeekToFirst();
        for (; it.Valid(); it.Next()) {
            UNIT_ASSERT_LT(lastKey, it.Key());
            lastKey = TString{it.Key()};
            UNIT_ASSERT_NO_EXCEPTION(it.Value<TReplicaObject>());
            ++iterations;
        }
        UNIT_ASSERT_EQUAL(it.Status(), expectedStatus);
        UNIT_ASSERT_VALUES_EQUAL(iterations, expectedIterations);
    }

    template <typename TReplicaObject>
    void TestIterationOrder(const IStorageElementsGenerator<TReplicaObject>& generator) {
        PutUpdateDeletePipeline<TReplicaObject>(generator, [](const TStorage& storage, const TInMemoryStorage<TReplicaObject>& actual, TStatus expected = TStatus::Ok()) {
            for (const TString& columnFamily : ListColumnFamilies(actual)) {
                CheckIterationOrder<TReplicaObject>(storage, actual.at(columnFamily).size(), columnFamily, expected);
            }
            if (expected) {
                TStringBuf nonexpCF = "nonexpCF";
                CheckIterationOrder<TReplicaObject>(storage, 0, TString(nonexpCF), TStatus::ColumnFamilyError(nonexpCF, "Trying to get a handle on a non-existent column family"));
            }
        });
    }

    Y_UNIT_TEST(IterationFullOrderForDnsRecordSets) {
        SetRandomSeed(42);
        TestIterationOrder(TRandomDnsRecordSetGenerator());
    }

    Y_UNIT_TEST(IterationAfterModification) {
        SetRandomSeed(42);
        TStorageOptions options = GetDefaultStorageOptions();
        TStorage storage(options);
        TDropGuard g(storage);

        TString columnFamily = TString(TDnsRecordSetReplicaObject::COLUMN_FAMILY_NAME);

        UNIT_ASSERT(storage.Open(/* validate */ false));

        TIterator it = storage.NewIterator(TReadOptions(), columnFamily);
        it.SeekToFirst();

        TInMemoryStorage<TDnsRecordSetReplicaObject> values;
        GenerateAndPut(storage, values, TRandomDnsRecordSetGenerator(), 10, TStatus(), columnFamily);

        UNIT_ASSERT(!it.Valid());

        {
            TIterator jt = storage.NewIterator(TReadOptions(), columnFamily);
            jt.SeekToFirst();
            for (const auto& [key, value] : values.at(columnFamily)) {
                UNIT_ASSERT(jt.Valid());
                UNIT_ASSERT_VALUES_EQUAL(jt.Key(), key);
                UNIT_ASSERT(value == jt.Value<TDnsRecordSetReplicaObject>());
                jt.Next();
            }
        }

        TIterator jt = storage.NewIterator(TReadOptions(), columnFamily);
        jt.SeekToFirst();

        TInMemoryStorage<TDnsRecordSetReplicaObject> newValues = values;
        GenerateAndPut(storage, newValues, TRandomDnsRecordSetGenerator(), 10, TStatus(), columnFamily);
        RegenerateAndPut(storage, newValues, TRandomDnsRecordSetGenerator(), ListKeys(values), columnFamily);

        for (auto actualIt = values.at(columnFamily).begin(); 
                  actualIt != values.at(columnFamily).end(); jt.Next(), ++actualIt) {
            UNIT_ASSERT(jt.Valid());
            UNIT_ASSERT_VALUES_EQUAL(actualIt->first, jt.Key());
            UNIT_ASSERT(actualIt->second == jt.Value<TDnsRecordSetReplicaObject>());
        }
        UNIT_ASSERT(!jt.Valid());
        UNIT_ASSERT(it.Status());
    }
}

Y_UNIT_TEST_SUITE(TestColumnFamily) {
    template <typename TReplicaObject>
    void TestPutAndGetReplicaObjects(const IStorageElementsGenerator<TReplicaObject>& generator) {
        TVector<TString> columnFamilies = {"cf1", "cf2", "cf3"};
        TStorageOptions options = GetDefaultStorageOptions(STORAGE_NAME, columnFamilies);
        TStorage storage(options);
        TDropGuard g(storage);

        TInMemoryStorage<TReplicaObject> inMemoryStorage;

        UNIT_ASSERT(storage.Open(/* validate */ false));

        for (const TString& columnFamily : columnFamilies) {
            for (size_t n : {1, 100, 1000}) {
                GenerateAndPut(storage, inMemoryStorage, generator, n, TStatus(), columnFamily);
                CheckStoragesEqual(storage, inMemoryStorage);
            }
        }
    }

    Y_UNIT_TEST(PutAndGetReplicaObjects) {
        SetRandomSeed(42);
        TestPutAndGetReplicaObjects(TRandomDnsRecordSetGenerator());
    }

    template <typename TReplicaObject>
    void TestCreateAndDelete(const IStorageElementsGenerator<TReplicaObject>& generator, bool EnableCache = false) {
        TVector<TString> stableColumnFamilies = {"cf1", "cf2", "cf3"};
        TStorageOptions options = GetDefaultStorageOptions(STORAGE_NAME, stableColumnFamilies);
        options.EnableCache = EnableCache;
        TStorage storage(options);
        TDropGuard g(storage);

        TInMemoryStorage<TReplicaObject> inMemoryStorage;
        TInMemoryStorage<TReplicaObject> deletedCF;
        auto pipeline = [&storage, &inMemoryStorage, &generator, &EnableCache](ui64 n, const TString& columnFamily, TStatus expected = TStatus()) {
            GenerateAndPut(storage, inMemoryStorage, generator, n, expected, columnFamily);
            FullCheckStoragesEqual(storage, inMemoryStorage, EnableCache);
            if (auto contains = storage.ContainsColumnFamily(columnFamily); contains.IsSuccess() && contains.Success()) {
                TVector<TString> keys = DeleteRandomKey(storage, inMemoryStorage, generator, n / 2, expected, true, columnFamily);
                FullCheckStoragesEqual(storage, inMemoryStorage, EnableCache);
                CheckKeyNotFound<TReplicaObject>(storage, keys, columnFamily, EnableCache);
            } else {
                Y_ENSURE(!expected || contains.IsError(), "if there is no column family, status cannot be ok");
            }
        };

        CheckListColumnFamilies(storage, {});

        UNIT_ASSERT(storage.Open(/* validate */ false));

        CheckListColumnFamilies(storage, stableColumnFamilies);

        for (const TString& columnFamily : stableColumnFamilies) {
            GenerateAndPut(storage, inMemoryStorage, generator, 1, TStatus(), columnFamily);
        }

        TVector<TString> allColumnFamilies = {"cf1", "cf2", "cf3", "cf4", "cf5", "cf6"};

        for (const TString& columnFamily : allColumnFamilies) {
            CreateColumnFamilies(storage, inMemoryStorage, {columnFamily}, !storage.ContainsColumnFamily(columnFamily).Success() ? TStatus() : TStatus::ColumnFamilyError(columnFamily, ""));
            pipeline(100, columnFamily);
        }

        TIterator it = storage.NewIterator(TReadOptions(), "cf5");

        DeleteColumnFamilies(storage, inMemoryStorage, deletedCF, {"cf5"});
        FullCheckStoragesEqual(storage, inMemoryStorage, EnableCache);

        CheckCFEqual(deletedCF, it, "cf5");

        for (const TString& columnFamily : allColumnFamilies) {
            if (!storage.ContainsColumnFamily(columnFamily).Success()) {
                pipeline(100, columnFamily, TStatus::ColumnFamilyError(columnFamily, ""));
            } else {
                pipeline(100, columnFamily);
            }
        }

        DeleteColumnFamilies(storage, inMemoryStorage, deletedCF, {"cf5"}, TStatus::ColumnFamilyError("cf5", ""));
        FullCheckStoragesEqual(storage, inMemoryStorage, EnableCache);

        UNIT_ASSERT(storage.DeleteColumnFamilies({"cf5"}, /* ignoreNonExisting */ true).IsSuccess());

        DeleteColumnFamilies(storage, inMemoryStorage, deletedCF, {"cf1"}, TStatus::ColumnFamilyError("cf1", ""));
        FullCheckStoragesEqual(storage, inMemoryStorage, EnableCache);

        for (const TString& columnFamily : allColumnFamilies) {
            if (!storage.ContainsColumnFamily(columnFamily).Success()) {
                pipeline(100, columnFamily, TStatus::ColumnFamilyError(columnFamily, ""));
            } else {
                pipeline(100, columnFamily);
            }
        }

        for (const TString& columnFamily : allColumnFamilies) {
            bool containsCF = storage.ContainsColumnFamily(columnFamily).Success();
            CreateColumnFamilies(storage, inMemoryStorage, {columnFamily}, !containsCF ? TStatus() : TStatus::ColumnFamilyError(columnFamily, ""));
            if (!containsCF) {
                CheckKeyNotFound<TReplicaObject>(storage, ListKeys(deletedCF, columnFamily), columnFamily, EnableCache);
            }
            pipeline(100, columnFamily);
        }
    }

    Y_UNIT_TEST(CreateAndDelete) {
        SetRandomSeed(42);
        TestCreateAndDelete(TRandomDnsRecordSetGenerator());
    }

    template <typename TReplicaObject>
    void TestReopenAndDrop(const IStorageElementsGenerator<TReplicaObject>& generator, bool EnableCache = false) {
        TVector<TString> stableColumnFamilies = {"cf1", "cf2", "cf3"};
        TStorageOptions storageOptions = GetDefaultStorageOptions(STORAGE_NAME, stableColumnFamilies);
        storageOptions.EnableCache = EnableCache;
        TStorage firstStorage(storageOptions);
        TDropGuard gFirst(firstStorage);
        TStorage secondStorage(storageOptions);
        TDropGuard gSecond(secondStorage);

        UNIT_ASSERT(firstStorage.Open(/* validate */ false));

        TInMemoryStorage<TReplicaObject> inMemoryStorage;
        TInMemoryStorage<TReplicaObject> deletedCF;
        TVector<TString> allColumnFamilies = {"cf1", "cf2", "cf3", "cf4", "cf5", "cf6"};

        for (const TString& columnFamily : allColumnFamilies) {
            CreateColumnFamilies(firstStorage, inMemoryStorage, {columnFamily}, !firstStorage.ContainsColumnFamily(columnFamily).Success() ? TStatus() : TStatus::ColumnFamilyError(columnFamily, ""));
            GenerateAndPut(firstStorage, inMemoryStorage, generator, 100, TStatus(), columnFamily);
            TVector<TString> keys = DeleteRandomKey(firstStorage, inMemoryStorage, generator, 25, TStatus(), true, columnFamily);
            CheckKeyNotFound<TReplicaObject>(firstStorage, keys, columnFamily, EnableCache);
        }
        FullCheckStoragesEqual(firstStorage, inMemoryStorage, EnableCache);

        DeleteColumnFamilies(firstStorage, inMemoryStorage, deletedCF, {"cf5"});
        FullCheckStoragesEqual(firstStorage, inMemoryStorage, EnableCache);

        UNIT_ASSERT(firstStorage.Reopen(/* validate */ false));
        FullCheckStoragesEqual(firstStorage, inMemoryStorage, EnableCache);

        UNIT_ASSERT(firstStorage.Close());
        UNIT_ASSERT(!firstStorage.IsOpened());
        UNIT_ASSERT(secondStorage.Open(/* validate */ false));
        FullCheckStoragesEqual(secondStorage, inMemoryStorage, EnableCache);
        
        secondStorage.Drop();

        UNIT_ASSERT(firstStorage.Open(/* validate */ false));
        inMemoryStorage.clear();
        for (TString& columnFamily : stableColumnFamilies) {
            inMemoryStorage[columnFamily] = TInMemoryColumnFamily<TReplicaObject>();
        }
        FullCheckStoragesEqual(firstStorage, inMemoryStorage, EnableCache);
    }

    Y_UNIT_TEST(ReopenAndDrop) {
        SetRandomSeed(42);
        TestReopenAndDrop(TRandomDnsRecordSetGenerator());
    }

    Y_UNIT_TEST(Cache) {
        SetRandomSeed(42);
        TestCreateAndDelete(TRandomDnsRecordSetGenerator(), /* EnableCache */ true);
        TestReopenAndDrop(TRandomDnsRecordSetGenerator(), /* EnableCache */ true);
    }
}
