#include "status.h"
#include "storage.h"
#include "util.h"

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/threading/future/async.h>
#include <library/cpp/threading/future/core/future.h>

#include <util/datetime/base.h>
#include <util/datetime/cputimer.h>
#include <util/folder/path.h>
#include <util/thread/pool.h>

using namespace NYP;
using namespace NYPReplica;

namespace {
    const TFsPath STORAGE_PATH = "storage";
    const TFsPath BACKUP_PATH = "backup";
    const TFsPath LOG_FILE = "eventlog";

    class TRandomStringReplicaObject {
    public:
        TRandomStringReplicaObject() = default;

        TRandomStringReplicaObject(size_t size) {
            Value_.resize(size);
            constexpr ui32 range = 'z' - 'a';
            for (ui32 i = 0; i < size; ++i) {
                Value_[i] = RandomNumber<ui8>(range) + 'a';
            }
        }

        TString ToString() const {
            return Value_;
        }

    private:
        TString Value_;
    };

    template <typename TReplicaObject>
    using TStorageValue = TVector<TStorageElement<TReplicaObject>>;

    class TRandomKeyValueGenerator {
    public:
        TRandomKeyValueGenerator() = default;

        TRandomKeyValueGenerator(ui64 maxSizeKey, ui64 maxSizeValue, ui64 maxNumRecords)
            : MaxSizeKey_(maxSizeKey)
            , MaxSizeValue_(maxSizeValue)
            , MaxNumRecords_(maxNumRecords)
        {
        }

        std::pair<TString, TStorageValue<TRandomStringReplicaObject>> operator() (TMaybe<TString> key = Nothing()) const {
            if (!key.Defined()) {
                key = TRandomStringReplicaObject(1 + RandomNumber<size_t>(MaxSizeKey_)).ToString();
            }
            TStorageValue<TRandomStringReplicaObject> values;
            const size_t recordsNumber = 1 + RandomNumber<size_t>(MaxNumRecords_);
            for (size_t i = 0; i < recordsNumber; ++i) {
                values.emplace_back(TRandomStringReplicaObject(1 + RandomNumber<size_t>(MaxSizeValue_)));
            }
            return {*key, values};
        }

    private:
        const ui64 MaxSizeKey_ = 10000;
        const ui64 MaxSizeValue_ = 10000;
        const ui64 MaxNumRecords_ = 100;
    };
}

TStorageOptions GetDefaultStorageOptions(TStringBuf storageName = "default", const ui64 maxAllowedSpaceUsage = 0, const ui64 compactionBufferSize = 0) {    
    TStorageOptions options;
    options.ColumnFamilies = {
        "abacaba",
        "abadaba",
    };
    options.Meta = TStorageMeta({TString(storageName), 1});
    options.Paths.StoragePath = STORAGE_PATH / storageName;
    options.Paths.BackupPath = BACKUP_PATH / storageName;
    options.Paths.LogsPath = LOG_FILE / storageName;
    TRocksDBConfig rocksDBConfig;
    rocksDBConfig.SetMaxAllowedSpaceUsageBytes(maxAllowedSpaceUsage);
    rocksDBConfig.SetCompactionBufferSizeBytes(compactionBufferSize);
    options.ReplicaConfig.MutableStorageConfig()->MutableRocksDBConfig()->CopyFrom(rocksDBConfig);

    return options;
}

TVector<std::pair<TString, TStorageValue<TRandomStringReplicaObject>>> GenSample(const ui64 limit) {
    TVector<std::pair<TString, TStorageValue<TRandomStringReplicaObject>>> testSample;
    TRandomKeyValueGenerator generator;

    NYP::NYPReplica::TStorage storage(GetDefaultStorageOptions("unlimited"));
    storage.Open(false);
    int iteration = 0;
    while (true) {
        testSample.emplace_back(generator());
        const auto& [key, value] = testSample.back();
        UNIT_ASSERT(storage.PutReplicaObjects(TWriteOptions(), "abacaba", key, value));

        if (++iteration % 50 == 0 && GetTotalDirectorySizeBytes((STORAGE_PATH / "unlimited").GetPath()) >= 2 * limit) {
           break;
        }
    }

    return testSample;
}

TStatus FillStorage(const TVector<std::pair<TString, TStorageValue<TRandomStringReplicaObject>>>& testSample, NYP::NYPReplica::TStorage& storage, const TDuration maxDelay) {
    storage.Open(false);
    TAdaptiveThreadPool queue;
    queue.Start();
    for (const auto& request : testSample) {
        NThreading::TFuture<TStatus> future = NThreading::Async([&storage, &request] () {
            return storage.PutReplicaObjects(TWriteOptions(), "abacaba", request.first, request.second);
        }, queue);

        try {
            TStatus result = future.GetValue(maxDelay);
            if (!result) {
                return result;
            }
        } catch (const NThreading::TFutureException&) {
            return future.GetValueSync();
        }
    }
    return TStatus::Ok();
}

Y_UNIT_TEST_SUITE(StorageMaxAllowedSpaceUsageBytes) {
    Y_UNIT_TEST(StorageMaxAllowedSpaceUsageBytes) {
        // It isn't recommended to make less than 1GB. This may break test
        constexpr ui64 limit = 1_GB;
        SetRandomSeed(limit);
        TVector<std::pair<TString, TStorageValue<TRandomStringReplicaObject>>> testSample = GenSample(limit);
        ui64 testSampleSizeBytes = GetTotalDirectorySizeBytes((STORAGE_PATH / "unlimited").GetPath());

        // limited, CompactionBufferSizeBytes = 0
        {
            NYP::NYPReplica::TStorage storage(GetDefaultStorageOptions("limited1", limit));

            TStatus result = FillStorage(testSample, storage, TDuration::Seconds(5));

            if (!result) {
                UNIT_ASSERT_EQUAL(result.Error().SubCode(), TError::ESubCode::SpaceLimit);
            }

            // we allow an excess of 256MB (due to the size of the records)
            constexpr ui64 maxExcess = 256_MB;

            UNIT_ASSERT_LE(GetTotalDirectorySizeBytes((STORAGE_PATH / "limited1").GetPath()), limit + maxExcess);
        }

        // limited, CompactionBufferSizeBytes = limit / 2
        {
            NYP::NYPReplica::TStorage storage(GetDefaultStorageOptions("limited2", limit, limit / 2));
            UNIT_ASSERT(FillStorage(testSample, storage, TDuration::Seconds(15)));
            UNIT_ASSERT_LE(GetTotalDirectorySizeBytes((STORAGE_PATH / "limited2").GetPath()), limit);
        }

        // resuming after Status::SpaceLimit
        {
            constexpr int attempts = 3;
            constexpr ui64 smallLimit = 256_MB;
            // we allow an error of 512MB (due to the size of the records)
            constexpr ui64 maxError = 512_MB;
            
            for (int i = 0; i <= attempts; ++i) {
                UNIT_ASSERT_UNEQUAL(i, attempts);

                TString storageName = TStringBuilder{} << "limited" << 3 + i;
                NYP::NYPReplica::TStorage storage(GetDefaultStorageOptions(storageName, smallLimit));

                TStatus result = FillStorage(testSample, storage, TDuration::Seconds(5));

                if (result || (!result && result.Error().SubCode() != TError::ESubCode::SpaceLimit)) {
                    continue;
                }

                TYPClusterConfig clusterConfig;
                clusterConfig.MutableRocksDBConfig()->SetMaxAllowedSpaceUsageBytes(limit * 3);
                storage.UpdateClusterConfig(clusterConfig);

                UNIT_ASSERT(FillStorage(testSample, storage, TDuration::Seconds(15)));

                ui64 storageSizeBytes = GetTotalDirectorySizeBytes((STORAGE_PATH / storageName).GetPath());
                UNIT_ASSERT_GE(storageSizeBytes, testSampleSizeBytes - maxError);
                UNIT_ASSERT_LE(storageSizeBytes, limit * 3);

                break;
            }
        }
    }
}
