#pragma once

#include "make_log_events.h"
#include "sensors.h"
#include "storage.h"
#include "update.h"

#include <infra/libs/yp_replica/protos/events_decl.ev.pb.h>

#include <infra/libs/logger/logger.h>

#include <infra/libs/service_iface/errors.h>

#include "util/string/split.h"

namespace NYP::NYPReplica {

template <typename TReplicaObject>
struct IModifyReplicaElementCallback {
    virtual void Do(const TMaybe<TReplicaObject>& oldValue, TReplicaObject& newValue, EObjectType newValueType) = 0;
    virtual ~IModifyReplicaElementCallback() { }
};

template <typename TReplicaObject>
struct ITableRuleCallback {
    virtual void Do(const TString& key, const TVector<TStorageElementRef<TReplicaObject>>& newValue) = 0;
    virtual ~ITableRuleCallback() { }
};

template <typename TReplicaObject>
class ITableRule {
public:
    virtual ~ITableRule() = default;

    // ID that defines the rule.
    // All the IDs of the rules in TTableRulesHolder for one TReplicaObject must be different
    virtual constexpr TStringBuf GetID() const = 0;

    // Sets the GetTableName functions type. For a stable rule, there can be only one table.
    // For a dynamic rule, you can select a table for each object.
    virtual constexpr bool IsStable() const = 0;

    // Sets the key by which the object will be saved
    virtual TString GetKey(const TReplicaObject& object) const {
        return object.GetObjectId();
    }

    // TODO(azuremint) Оптимизация хранения при дубликации
    //virtual TReplicaObject GetValue(const TReplicaObject& object) const = 0;

    // Sets the filtering of objects for which the rule is applied.
    // Rule tables will contain only objects for which rule.Filter(object) = true
    virtual bool Filter(const TReplicaObject& /* object */) const {
        return true;
    }

    // TODO(azuremint) Нужно ли вообще для стабильных правил название таблицы если уже есть уникальное ID правила?
    // Used only in stable rule
    virtual TString GetTableName() const {
        ythrow NInfra::TNotImplementedError();
    }

    // Used only in dynamic rule
    virtual TString GetTableName(const TReplicaObject& /* object */) const {
        ythrow NInfra::TNotImplementedError();
    }

    // All tables created by the rule use these options
    // Name in TColumnFamilyOptions is not used
    virtual TColumnFamilyOptions GetColumnFamilyOptions() const {
        return ColumnFamilyOptions_.GetOrElse(TColumnFamilyOptions(TString()));
    }

public:
    void SetCallback(THolder<ITableRuleCallback<TReplicaObject>>&& callback) {
        RuleCallback_ = std::move(callback);
    }

    void DoCallback(
        const TString& key,
        const TVector<TStorageElementRef<TReplicaObject>>& newValue,
        const TString& columnFamilyName,
        NInfra::TLogFramePtr logFrame,
        const NInfra::TSensorGroup& sensorGroup
    ) const {
        if (RuleCallback_) {
            try {
                RuleCallback_->Do(key, newValue);
            } catch (...) {
                INFRA_LOG_ERROR(TReplicaTableRuleCallbackError(columnFamilyName, key, CurrentExceptionMessage()));
                NInfra::TRateSensor(sensorGroup, NSensors::TABLE_RULE_CALLBACK_ERROR, {{NSensors::COLUMN_FAMILY, columnFamilyName}}).Inc();
            }
        }
    }

    void SetCFOptionsIfEmpty(TColumnFamilyOptions options) {
        if (ColumnFamilyOptions_.Empty()) {
            ColumnFamilyOptions_ = std::move(options);
        }
    }

protected:
    THolder<ITableRuleCallback<TReplicaObject>> RuleCallback_;
    TMaybe<TColumnFamilyOptions> ColumnFamilyOptions_;
};

template <typename TReplicaObject>
class TMainTableRule final : public ITableRule<TReplicaObject> {
public:
    constexpr TStringBuf GetID() const override {
        return ID;
    }

    constexpr bool IsStable() const override {
        return true;
    }

    TString GetTableName() const override {
        return TString();
    }

    constexpr static TStringBuf ID = "MAIN";
};

template <typename TReplicaObject>
class TReplicaObjectKeyTableRule final : public ITableRule<TReplicaObject> {
public:
    constexpr TStringBuf GetID() const override {
        return ID;
    }

    constexpr bool IsStable() const override {
        return true;
    }

    TString GetKey(const TReplicaObject& object) const override {
        return object.GetKey();
    }

    TString GetTableName() const override {
        return TString();
    }

    constexpr static TStringBuf ID = "REPLICA_OBJECT_KEY";
};

template <typename TReplicaObject>
inline TStringBuf GetDefaultRuleID() {
    return !TReplicaObject::KEY_IS_OBJECT_ID ? TReplicaObjectKeyTableRule<TReplicaObject>::ID : TMainTableRule<TReplicaObject>::ID;
}

template <typename TReplicaObject>
using TTableRulePtr = THolder<ITableRule<TReplicaObject>>;

template <typename... TReplicaObject>
class TTableManagers;

template <typename... TReplicaObjects>
class TTableRulesHolder {
public:
    TTableRulesHolder() {
        InitMainRule<TReplicaObjects...>();
        InitKeyRule<TReplicaObjects...>();
    }

    template <typename TReplicaObject>
    void AddRule(TTableRulePtr<TReplicaObject>&& rule) {
        std::get<TVector<TTableRulePtr<TReplicaObject>>>(Rules_).emplace_back(std::move(rule));
    }

    template <typename TReplicaObject>
    void SetCallbackForMainRule(THolder<ITableRuleCallback<TReplicaObject>>&& callback) {
        std::get<TVector<TTableRulePtr<TReplicaObject>>>(Rules_).front()->SetCallback(std::move(callback));
    }

    template <KeyIsNotObjectId TReplicaObject>
    void SetCallbackForKeyRule(THolder<ITableRuleCallback<TReplicaObject>>&& callback) {
        std::get<TVector<TTableRulePtr<TReplicaObject>>>(Rules_)[1]->SetCallback(std::move(callback));
    }

    template <typename TReplicaObject>
    void SetCFOptionsIfEmptyForMainRule(TColumnFamilyOptions options) {
        std::get<TVector<TTableRulePtr<TReplicaObject>>>(Rules_).front()->SetCFOptionsIfEmpty(std::move(options));
    }

    template <KeyIsNotObjectId TReplicaObject>
    void SetCFOptionsIfEmptyForKeyRule(TColumnFamilyOptions options) {
        std::get<TVector<TTableRulePtr<TReplicaObject>>>(Rules_)[1]->SetCFOptionsIfEmpty(std::move(options));
    }

private:
    template <EmptySet... TRestReplicaObjects>
    void InitMainRule() {
    }

    template <typename TReplicaObject, typename... TRestReplicaObjects>
    void InitMainRule() {
        std::get<TVector<TTableRulePtr<TReplicaObject>>>(Rules_).emplace_back(MakeHolder<TMainTableRule<TReplicaObject>>());
        InitMainRule<TRestReplicaObjects...>();
    }

    template <EmptySet... TRestReplicaObjects>
    void InitKeyRule() {
    }

    template <typename TReplicaObject, typename... TRestReplicaObjects>
    void InitKeyRule() {
        if constexpr (!TReplicaObject::KEY_IS_OBJECT_ID) {
            std::get<TVector<TTableRulePtr<TReplicaObject>>>(Rules_).emplace_back(MakeHolder<TReplicaObjectKeyTableRule<TReplicaObject>>());
        }
        InitKeyRule<TRestReplicaObjects...>();
    }

private:
    friend class TTableManagers<TReplicaObjects...>;
    std::tuple<TVector<TTableRulePtr<TReplicaObjects>>...> Rules_;
};

// Defines a table in requests to the TYPReplica.
// Each table is defined by TReplicaObject and rule ID (and tableName for dynamic rules)
template <typename TReplicaObject>
struct TTableInfo {
public:
    TTableInfo(TString ruleID, TMaybe<TString> tableName = Nothing())
        : RuleID(std::move(ruleID))
        , TableName(std::move(tableName))
    {
    }

public:
    TString RuleID;
    TMaybe<TString> TableName;
};

template <typename TReplicaObject>
class TTableManager {
    template <typename... TReplicaObjects>
    friend class TTableManagers;
public:
    TTableManager(TVector<TTableRulePtr<TReplicaObject>>&& rules) {
        for (auto& rule : rules) {
            Y_ENSURE(!Rules_.contains(rule->GetID()));
            Rules_[rule->GetID()] = std::move(rule);
        }
        MainColumnFamilyName_ = GetStableColumnFamilyName(TMainTableRule<TReplicaObject>::ID);
    }

    ~TTableManager() = default;

public:
    TMaybe<TString> GetColumnFamilyName(const TTableInfo<TReplicaObject>& tableInfo) const {
        auto ruleIt = Rules_.FindPtr(tableInfo.RuleID);
        if (ruleIt == nullptr) {
            return Nothing();
        } else if (tableInfo.TableName.Defined()) {
            return GetColumnFamilyName(tableInfo.RuleID, tableInfo.TableName.GetRef());
        } else {
            return GetColumnFamilyName(tableInfo.RuleID, (*ruleIt)->GetTableName());
        }
    }
    TVector<TString> GetAllStableColumnFamilyNames() const {
        TVector<TString> stableColumnFamilyNames;
        for (const auto& [ruleID, rule] : Rules_) {
            if (rule->IsStable()) {
                stableColumnFamilyNames.push_back(GetColumnFamilyName(ruleID, rule->GetTableName()));
            }
        }
        return stableColumnFamilyNames;
    }

    TColumnFamilyResolveFunc GetColumnFamilyResolveFunc() const {
        return [this](const TString& columnFamilyName) -> TMaybe<TColumnFamilyOptions> {
            TVector<TStringBuf> res = StringSplitter(columnFamilyName).Split('#');
            if (res.size() != 3 || res[0] != TReplicaObject::COLUMN_FAMILY_NAME) {
                return Nothing();
            }
            TString ruleID(res[1]);
            if (auto rulePtr = Rules_.FindPtr(ruleID); rulePtr) {
                TColumnFamilyOptions options = (*rulePtr)->GetColumnFamilyOptions();
                options.Name = columnFamilyName;
                return options;
            } else {
                return Nothing();
            }
        };
    }

    TVector<TColumnFamilyOptions> GetAllStableColumnFamilies() const {
        TVector<TColumnFamilyOptions> stableColumnFamilies;
        for (const auto& [ruleID, rule] : Rules_) {
            if (rule->IsStable()) {
                TColumnFamilyOptions options = rule->GetColumnFamilyOptions();
                options.Name = ruleID;
                stableColumnFamilies.push_back(std::move(options));
            }
        }
        return stableColumnFamilies;
    }

    TString GetMainColumnFamilyName() const {
        return MainColumnFamilyName_;
    }

private:
    inline static TString GetColumnFamilyName(const TStringBuf ruleID, TStringBuf tableName) {
        return TStringBuilder{} << TReplicaObject::COLUMN_FAMILY_NAME << "#" << ruleID << "#" << tableName;
    }

    inline TString GetStableColumnFamilyName(const TStringBuf stableRuleID) const {
        return GetColumnFamilyName(stableRuleID, Rules_.at(stableRuleID)->GetTableName());
    }

private:
    using TColumnFamilyUpdate = THashMap<TString, TVector<TStorageElementRef<TReplicaObject>>>;
    using TStorageUpdates = THashMap<TString, TColumnFamilyUpdate>;

    TStatus ApplyUpdates(
        TStorage& storage,
        TWriteBatch& batch,
        THashMap<TString, ui64>& objectsNumber,
        TUpdates<TReplicaObject>& updates,
        TAtomicSharedPtr<TUpdatesDisabler> updatesDisabler,
        NInfra::TLogFramePtr logFrame,
        const NInfra::TSensorGroup& sensorGroup
    ) {
        objectsNumber[MainColumnFamilyName_] = updates.SnapshotObjectsNumber;

        if (ModifyElementCallback_) {
            for (auto& update : updates.Data) {
                try {
                    ModifyElementCallback_->Do(update.OldValue, update.NewValue.ReplicaObject, update.NewValue.Type);
                } catch (...) {
                    INFRA_LOG_ERROR(TReplicaModifyElementsError(
                        TString{TReplicaObject::NAME}, update.NewValue.ReplicaObject.GetKey(), CurrentExceptionMessage()));
                    NInfra::TRateSensor(sensorGroup, NSensors::MODIFY_ELEMENTS_ERROR, {{NSensors::REPLICA_OBJECT_TYPE, TReplicaObject::NAME}}).Inc();
                }
            }
        }

        for (const auto& [ruleID, rule] : Rules_) {
            OUTCOME_TRYV(updatesDisabler->Status());

            if (rule->IsStable()) {
                OUTCOME_TRYV(ApplyUpdatesForStableRule(*rule, storage, batch, objectsNumber, updates, updatesDisabler, logFrame, sensorGroup));
            } else {
                OUTCOME_TRYV(ApplyUpdatesForDynamicRule(*rule, storage, batch, objectsNumber, updates, updatesDisabler, logFrame, sensorGroup));
            }
        }

        OUTCOME_TRYV(batch.UpdateTimestamp(updates.Timestamp));

        OUTCOME_TRYV(batch.UpdateYpTimestamp(updates.YpTimestamp));

        OUTCOME_TRYV(updatesDisabler->Status());

        return TStatus::Ok();
    }

    TStatus ApplyUpdatesForStableRule(
        const ITableRule<TReplicaObject>& stableRule,
        TStorage& storage,
        TWriteBatch& batch,
        THashMap<TString, ui64>& objectsNumber,
        const TUpdates<TReplicaObject>& updates,
        TAtomicSharedPtr<TUpdatesDisabler> updatesDisabler,
        NInfra::TLogFramePtr logFrame,
        const NInfra::TSensorGroup& sensorGroup
    ) {
        Y_ENSURE(stableRule.IsStable());

        TStorageUpdates storageUpdates;
        TColumnFamilyUpdate& cfUpdates = storageUpdates[GetColumnFamilyName(stableRule.GetID(), stableRule.GetTableName())];

        for (const auto& [oldValue, newValue] : updates.Data) {
            OUTCOME_TRYV(updatesDisabler->Status());

            if (newValue.Type == EObjectType::PUT && stableRule.Filter(newValue.ReplicaObject)) {
                TString newKey = stableRule.GetKey(newValue.ReplicaObject);

                const TReplicaObject* oldValueRef = oldValue.Get();
                if (oldValue.Defined() && stableRule.Filter(*oldValueRef)) {
                    TString oldKey = stableRule.GetKey(*oldValueRef);
                    if (oldKey != newKey) {
                        cfUpdates[std::move(oldKey)].emplace_back(*oldValueRef, EObjectType::DELETE);
                    }
                }

                cfUpdates[std::move(newKey)].emplace_back(newValue.ReplicaObject, EObjectType::PUT);
            } else if (newValue.Type == EObjectType::DELETE || oldValue.Defined()) {
                const TReplicaObject& oldValueRef = oldValue.GetRef();
                if (stableRule.Filter(oldValueRef)) {
                    cfUpdates[stableRule.GetKey(oldValueRef)].emplace_back(oldValueRef, EObjectType::DELETE);
                }
            }
        }

        OUTCOME_TRYV(updatesDisabler->Status());

        return ApplyUpdatesToWriteBatch(stableRule, storage, batch, objectsNumber, storageUpdates, updatesDisabler, logFrame, sensorGroup);
    }

    TStatus ApplyUpdatesForDynamicRule(
        const ITableRule<TReplicaObject>& dynamicRule,
        TStorage& storage,
        TWriteBatch& batch,
        THashMap<TString, ui64>& objectsNumber,
        const TUpdates<TReplicaObject>& updates,
        TAtomicSharedPtr<TUpdatesDisabler> updatesDisabler,
        NInfra::TLogFramePtr logFrame,
        const NInfra::TSensorGroup& sensorGroup
    ) {
        Y_ENSURE(!dynamicRule.IsStable());

        TStringBuf ruleID = dynamicRule.GetID();
        TStorageUpdates storageUpdates;

        for (const auto& [oldValue, newValue] : updates.Data) {
            OUTCOME_TRYV(updatesDisabler->Status());

            if (newValue.Type == EObjectType::PUT && dynamicRule.Filter(newValue.ReplicaObject)) {
                TString newKey = dynamicRule.GetKey(newValue.ReplicaObject);

                TString newCFName = GetColumnFamilyName(ruleID, dynamicRule.GetTableName(newValue.ReplicaObject));

                const TReplicaObject* oldValueRef = oldValue.Get();
                if (oldValue.Defined() && dynamicRule.Filter(*oldValueRef)) {
                    TString oldKey = dynamicRule.GetKey(*oldValueRef);
                    TString oldCFName = GetColumnFamilyName(ruleID, dynamicRule.GetTableName(*oldValueRef));
                    if (oldKey != newKey || oldCFName != newCFName) {
                        storageUpdates[std::move(oldCFName)][std::move(oldKey)].emplace_back(*oldValueRef, EObjectType::DELETE);
                    }
                }

                storageUpdates[std::move(newCFName)][std::move(newKey)].emplace_back(newValue.ReplicaObject, EObjectType::PUT);
            } else if (newValue.Type == EObjectType::DELETE || oldValue.Defined()) {
                const TReplicaObject& oldValueRef = oldValue.GetRef();
                if (dynamicRule.Filter(oldValueRef)) {
                    TString oldCFName = GetColumnFamilyName(ruleID, dynamicRule.GetTableName(oldValueRef));
                    storageUpdates[std::move(oldCFName)][dynamicRule.GetKey(oldValueRef)].emplace_back(oldValueRef, EObjectType::DELETE);
                }
            }
        }

        OUTCOME_TRYV(updatesDisabler->Status());

        if (storageUpdates.empty()) {
            return TStatus::Ok();
        }

        TVector<TColumnFamilyOptions> columnFamilies;
        columnFamilies.reserve(storageUpdates.size());
        for (const auto& [cfName, _] : storageUpdates) {
            TColumnFamilyOptions options = dynamicRule.GetColumnFamilyOptions();
            options.Name = cfName;
            columnFamilies.push_back(options);
        }

        TVector<TString> createdCF = std::move(OUTCOME_TRYX(storage.CreateColumnFamilies(columnFamilies, true)));
        if (!columnFamilies.empty()) {
            INFRA_LOG_INFO(MakeReplicaUpdateColumnFamiliesEvent(createdCF, TReplicaUpdateColumnFamilies::CREATE));
        }

        return ApplyUpdatesToWriteBatch(dynamicRule, storage, batch, objectsNumber, storageUpdates, updatesDisabler, logFrame, sensorGroup);
    }

    TStatus ApplyUpdatesToWriteBatch(
        const ITableRule<TReplicaObject>& rule,
        TStorage& storage,
        TWriteBatch& batch,
        THashMap<TString, ui64>& objectsNumber,
        const TStorageUpdates& updates,
        TAtomicSharedPtr<TUpdatesDisabler> updatesDisabler,
        NInfra::TLogFramePtr logFrame,
        const NInfra::TSensorGroup& sensorGroup
    ) {
        TReadOptions options;

        for (const auto& [columnFamilyName, columnFamily] : updates) {
            ui64 objectsNumberInCF = 0;
            if (columnFamilyName != MainColumnFamilyName_) {
                objectsNumberInCF = storage.template GetObjectsNumber<TReplicaObject>(options, columnFamilyName);
            }

            for (const auto& [key, diff] : columnFamily) {
                OUTCOME_TRYV(updatesDisabler->Status());

                TVector<TStorageElement<TReplicaObject>> oldValue;
                TVector<TStorageElementRef<TReplicaObject>> newValue;

                if (columnFamilyName != MainColumnFamilyName_) {
                    if (auto getResult = storage.template GetReplicaObjects<TReplicaObject>(options, columnFamilyName, key); getResult.IsError()) {
                        if (getResult.Error() != TStatus::NotFound()) {
                            return getResult.Error();
                        }
                    } else {
                        oldValue = std::move(getResult.Success());
                    }
                    newValue = ApplyDiff<TReplicaObject>(oldValue, std::move(diff));
                } else {
                    Y_ENSURE(diff.size() == 1);
                    if (diff.front().Type == EObjectType::PUT) {
                        newValue = {diff.front()};
                    }
                }

                rule.DoCallback(key, newValue, columnFamilyName, logFrame, sensorGroup);

                INFRA_LOG_DEBUG(MakeReplicaStorageUpdateObjectEvent<TReplicaObject>(columnFamilyName, key, newValue));

                if (!newValue.empty()) {
                    if (oldValue.empty()) {
                        ++objectsNumberInCF;
                    }
                    OUTCOME_TRYV(batch.PutReplicaObjects(columnFamilyName, key, newValue));
                } else {
                    --objectsNumberInCF;
                    OUTCOME_TRYV(batch.Delete(columnFamilyName, key));
                }
            }

            if (columnFamilyName == MainColumnFamilyName_) {
                OUTCOME_TRYV(batch.UpdateObjectsNumber<TReplicaObject>(columnFamilyName, objectsNumber[columnFamilyName]));
            } else {
                OUTCOME_TRYV(batch.UpdateObjectsNumber<TReplicaObject>(columnFamilyName, objectsNumberInCF));
                objectsNumber[columnFamilyName] = objectsNumberInCF;
            }
        }

        return TStatus::Ok();
    }

private:
    void SetModifyReplicaElementCallback(THolder<IModifyReplicaElementCallback<TReplicaObject>>&& callback) {
        ModifyElementCallback_ = std::move(callback);
    }

    void SetCFOptionsIfEmpty(const TColumnFamilyOptions& options) {
        for (auto& [RuleId, Rule] : Rules_) {
            Rule->SetCFOptionsIfEmpty(options);
        }
    }

private:
    THashMap<TString, TTableRulePtr<TReplicaObject>> Rules_;
    TString MainColumnFamilyName_;
    THolder<IModifyReplicaElementCallback<TReplicaObject>> ModifyElementCallback_;
};

template <typename... TReplicaObjects>
class TTableManagers {
public:
    TTableManagers(TTableRulesHolder<TReplicaObjects...>&& rulesHolder)
        : TableManagers_(std::make_from_tuple<std::tuple<TTableManager<TReplicaObjects>...>>(std::move(rulesHolder.Rules_)))
    {
    }

public:
    template <typename TReplicaObject>
    TMaybe<TString> GetColumnFamilyName(const TTableInfo<TReplicaObject>& tableInfo) const {
        return std::get<TTableManager<TReplicaObject>>(TableManagers_).GetColumnFamilyName(tableInfo);
    }

    template <EmptySet... TRestReplicaObjects>
    TVector<TString> GetAllStableColumnFamilyNames() const {
        return TVector<TString>();
    }

    template <typename TReplicaObject, typename... TRestReplicaObjects>
    TVector<TString> GetAllStableColumnFamilyNames() const {
        TVector<TString> firstStableColumnFamilyNames = std::get<TTableManager<TReplicaObject>>(TableManagers_).GetAllStableColumnFamilyNames();
        TVector<TString> restStableColumnFamilyNames = GetAllStableColumnFamilyNames<TRestReplicaObjects...>();
        std::copy(
            std::move_iterator(firstStableColumnFamilyNames.begin()), 
            std::move_iterator(firstStableColumnFamilyNames.end()),
            std::back_inserter(restStableColumnFamilyNames)
        );
        return restStableColumnFamilyNames;
    }

    template <EmptySet... TRestReplicaObjects>
    TVector<TColumnFamilyResolveFunc> GetColumnFamilyResolveFuncs() const {
        return TVector<TColumnFamilyResolveFunc>();
    }

    template <typename TReplicaObject, typename... TRestReplicaObjects>
    TVector<TColumnFamilyResolveFunc> GetColumnFamilyResolveFuncs() const {
        TColumnFamilyResolveFunc firstColumnFamilyResolveFunc = std::get<TTableManager<TReplicaObject>>(TableManagers_).GetColumnFamilyResolveFunc();
        TVector<TColumnFamilyResolveFunc> restColumnFamilyResolveFuncs = GetColumnFamilyResolveFuncs<TRestReplicaObjects...>();
        restColumnFamilyResolveFuncs.push_back(std::move(firstColumnFamilyResolveFunc));
        return restColumnFamilyResolveFuncs;
    }

    template <typename TReplicaObject>
    TString GetMainColumnFamilyName() const {
        return std::get<TTableManager<TReplicaObject>>(TableManagers_).GetMainColumnFamilyName();
    }

public:
    template <EmptySet... TRestReplicaObjects>
    TStatus ApplyUpdates(
        TStorage& /* storage */,
        TWriteBatch& /* batch */,
        THashMap<TString, ui64>& /* objectsNumber */,
        std::tuple<TUpdates<TReplicaObjects>...>& /* updates */,
        TAtomicSharedPtr<TUpdatesDisabler> /* updatesDisabler */,
        NInfra::TLogFramePtr /* logFrame */,
        const NInfra::TSensorGroup& /* sensorGroup */
    ) const {
        return TStatus::Ok();
    }

    template <typename TReplicaObject, typename... TRestReplicaObjects>
    TStatus ApplyUpdates(
        TStorage& storage,
        TWriteBatch& batch,
        THashMap<TString, ui64>& objectsNumber,
        std::tuple<TUpdates<TReplicaObjects>...>& updates,
        TAtomicSharedPtr<TUpdatesDisabler> updatesDisabler,
        NInfra::TLogFramePtr logFrame,
        const NInfra::TSensorGroup& sensorGroup
    ) {
        OUTCOME_TRYV(std::get<TTableManager<TReplicaObject>>(TableManagers_).ApplyUpdates(
            storage, batch, objectsNumber, std::get<TUpdates<TReplicaObject>>(updates), updatesDisabler, logFrame, sensorGroup));
        return ApplyUpdates<TRestReplicaObjects...>(storage, batch, objectsNumber, updates, updatesDisabler, logFrame, sensorGroup);
    }

    template <typename TReplicaObject>
    void SetModifyReplicaElementCallback(THolder<IModifyReplicaElementCallback<TReplicaObject>>&& callback) {
        std::get<TTableManager<TReplicaObject>>(TableManagers_).SetModifyReplicaElementCallback(std::move(callback));
    }

    template <EmptySet... TRestReplicaObjects>
    void SetCFOptionsIfEmpty(const TColumnFamilyOptions& /* options */) {
    }

    template <typename TReplicaObject, typename... TRestReplicaObjects>
    void SetCFOptionsIfEmpty(const TColumnFamilyOptions& options) {
        std::get<TTableManager<TReplicaObject>>(TableManagers_).SetCFOptionsIfEmpty(options);
        SetCFOptionsIfEmpty<TRestReplicaObjects...>(options);
    }

private:
    std::tuple<TTableManager<TReplicaObjects>...> TableManagers_;
};

} // namespace NYP::NYPReplica
