#include <infra/libs/yp_replica/testing/decode.h>
#include <infra/libs/yp_replica/testing/encode.h>
#include <infra/libs/yp_replica/testing/testing_storage.h>

#include <library/cpp/getopt/small/last_getopt.h>

#include <util/folder/path.h>

using namespace NYP::NYPReplica;

struct TOptions {
    TMaybe<TFsPath> DecodedJson;
    TMaybe<TFsPath> EncodedJson;
    TMaybe<TFsPath> LoadJson;
    TMaybe<TFsPath> StoragePath;
    TMaybe<TFsPath> BackupPath;
    bool PrintShortJson = false;
    bool PrintFullJson = false;
    bool PrintOnlyIdAndKey = false;
};

TOptions ParseOptions(int argc, const char* argv[]) {
    TOptions result;

    NLastGetopt::TOpts opts;

    opts.AddHelpOption();

    opts
        .AddLongOption('e', "encode", "Encode short/full json description of the storage")
        .Optional()
        .RequiredArgument("PATH")
        .StoreResult(&result.DecodedJson);

    opts
        .AddLongOption('d', "decode", "Decode encoded short/full json description of the storage")
        .Optional()
        .RequiredArgument("PATH")
        .StoreResult(&result.EncodedJson);

    opts
        .AddLongOption('l', "load-storage", "Load storage from short/full json (the value of the storage in the full description should be strings)")
        .Optional()
        .RequiredArgument("PATH")
        .StoreResult(&result.LoadJson);

    opts
        .AddLongOption('s', "storage-path", "Path to storage")
        .Optional()
        .RequiredArgument("PATH")
        .StoreResult(&result.StoragePath);
    
    opts
        .AddLongOption('b', "backup-path", "Path to backup")
        .Optional()
        .RequiredArgument("PATH")
        .StoreResult(&result.BackupPath);

    opts
        .AddLongOption('p', "print-short", "Print storage values in short json format")
        .Optional()
        .NoArgument()
        .StoreTrue(&result.PrintShortJson);

    opts
        .AddLongOption('f', "print-full", "Print storage values in full json format (output only TReplicaObject)")
        .Optional()
        .NoArgument()
        .StoreTrue(&result.PrintFullJson);

    opts
        .AddLongOption('i', "only-id-key", "In Replica object, only the id and key will be displayed (with -p or -f)")
        .Optional()
        .NoArgument()
        .StoreTrue(&result.PrintOnlyIdAndKey);

    NLastGetopt::TOptsParseResult{&opts, argc, argv};

    return result;
}

void Encode(const TFsPath& file) {
    const TString rawJsonReplica = TFileInput(file).ReadAll();
    NJson::TJsonValue json;
    NJson::ReadJsonTree(rawJsonReplica, &json, true);
    NTesting::EncodeJson(json);
    Cout << NJson::WriteJson(json, true, true) << Endl;
}

void Decode(const TFsPath& file) {
    const TString rawJsonReplica = TFileInput(file).ReadAll();
    NJson::TJsonValue json;
    NJson::ReadJsonTree(rawJsonReplica, &json, true);
    NTesting::DecodeJson(json);
    Cout << NJson::WriteJson(json, true, true) << Endl;
}

using TTestingStorage = NTesting::TTestingStorage<
    TEndpointReplicaObject,
    TEndpointSetReplicaObject,
    TPodWithNodeIdKeyReplicaObject,
    TPodReplicaObject,
    TDnsRecordSetReplicaObject,
    TDnsZoneReplicaObject
>;

int main(int argc, const char* argv[]) {
    TOptions options = ParseOptions(argc, argv);
    if (options.EncodedJson.Defined()) {
        Decode(options.EncodedJson.GetRef());
    }
    if (options.DecodedJson.Defined()) {
        Encode(options.DecodedJson.GetRef());
    }
    if (options.LoadJson.Defined() || options.StoragePath.Defined() || options.BackupPath.Defined()) {
        TStorageOptions storageOptions;
        storageOptions.Paths.StoragePath = options.StoragePath.Defined() ? options.StoragePath.GetRef() : TFsPath("storage");
        storageOptions.Paths.BackupPath = options.BackupPath.Defined() ? options.BackupPath.GetRef() : TFsPath("backup");
        TTestingStorage storage(std::move(storageOptions));
        Y_ENSURE(storage.Open(/* validate */ false));
        if (options.LoadJson.Defined()) {
            storage.LoadFromJson(options.LoadJson.GetRef());
        }
        if (options.PrintShortJson || options.PrintFullJson) {
            NTesting::TDescriptionOptions descriptionOptions;
            descriptionOptions.IsFull = options.PrintFullJson;
            descriptionOptions.OnlyIdAndKey = options.PrintOnlyIdAndKey;
            NJson::TJsonValue jsonStorage = storage.GetJsonStorage(descriptionOptions);
            Cout << NJson::WriteJson(jsonStorage, true, true) << Endl;
        }
    }
}
