#include <yp/cpp/yp/client.h>
#include <yp/cpp/yp/semaphore.h>
#include <yp/cpp/yp/token.h>

#include <library/cpp/getopt/small/last_getopt.h>
#include <library/cpp/logger/global/global.h>

#include <util/datetime/base.h>
#include <util/generic/fwd.h>
#include <util/thread/pool.h>

struct TOptions {
    TString SemaphoreSetId;
    ui64 Budget;
    TDuration Frequency;
    TString YpClusterAddress;
    TString YpToken;
};

TOptions ParseOptions(int argc, const char* argv[]) {
    TOptions result;

    NLastGetopt::TOpts opts;

    opts.AddHelpOption();

    opts
        .AddLongOption('i', "semaphore-set-id", "Semaphore set id")
        .Required()
        .RequiredArgument()
        .StoreResult(&result.SemaphoreSetId);

    opts
        .AddLongOption('b', "budget", "Lock volume")
        .Required()
        .RequiredArgument()
        .StoreResult(&result.Budget);

    opts
        .AddLongOption('f', "frequency", "Semaphore lock frequency (sec)")
        .Required()
        .RequiredArgument()
        .StoreResult(&result.Frequency);

    opts
        .AddLongOption('a', "address", "YP cluster address")
        .Required()
        .RequiredArgument()
        .StoreResult(&result.YpClusterAddress);

    opts
        .AddLongOption('t', "yp-token", "YP token")
        .Optional()
        .RequiredArgument()
        .DefaultValue(NYP::NClient::FindToken())
        .StoreResult(&result.YpToken);

    NLastGetopt::TOptsParseResult{&opts, argc, argv};

    return result;
}

void LockSemaphore(const ui64 threadId, const TOptions& options) {
    const NYP::NClient::TClientOptions clientOptions = NYP::NClient::TClientOptions()
        .SetAddress(options.YpClusterAddress)
        .SetEnableSsl(true)
        .SetEnableBalancing(true)
        .SetTimeout(TDuration::Seconds(1))
        .SetToken(options.YpToken);

    NYP::NClient::TClient client(clientOptions);

    for (;;) {
        try {
            NYP::NClient::TSemaphoreSetGuard guard(
                client,
                options.SemaphoreSetId,
                NYP::NClient::GetDefaultSemaphoreSetOptions()
            );
            INFO_LOG << "thread_id: " << threadId << ". Ok, sleep: " << options.Frequency << Endl;
            Sleep(options.Frequency);
        } catch (...) {
            ERROR_LOG << "thread_id: " << threadId << ". " << CurrentExceptionMessage() << Endl;
        }
        INFO_LOG << "thread_id: " << threadId << ". End of cycle" << Endl;
    }
}

void RunMain(const TOptions& options) {
    TThreadPool pool;
    pool.Start(options.Budget);

    for (ui64 i = 0; i < options.Budget; ++i) {
        Y_ENSURE(pool.AddFunc([options, i]{
            LockSemaphore(i, options);
        }));
        Sleep(TDuration::MilliSeconds(50));
    }
}

int main(int argc, const char* argv[]) {
    InitGlobalLog2Console();
    RunMain(ParseOptions(argc, argv));
    return EXIT_SUCCESS;
}
