#include "semaphore.h"

#include <yp/yp_proto/yp/client/api/proto/access_control.pb.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/retry/retry.h>

namespace NYP::NYPReplica::NSemaphoreUpdater {

const TVector<TString> SEMAPHORE_ATTRIBUTE_SELECTORS = {
    "/meta",
    "/spec",
    "/labels",
};

const TVector<TString> SEMAPHORE_SET_ATTRIBUTE_SELECTORS = {
    "/meta",
    "/spec",
    "/labels",
};

bool SemaphoreCmp(const TSemaphore& lhs, const TSemaphore& rhs) {
    return lhs.YpSemaphoreSet.Meta().id() < rhs.YpSemaphoreSet.Meta().id();
}

TString GetFilter(const TStringBuf serviceId) {
    return TString::Join("([/labels/service_id] = \"", serviceId, "\")");
}

TVector<NClient::TSemaphoreSet> SelectYpSemaphoreSets(NClient::TClient& client, const TString& filter, const ui64 timestamp) {
    TVector<NClient::TSemaphoreSet> ypSemaphoreSets;
    INFO_LOG << "Getting semaphore sets" << Endl;
    NYP::NClient::TSelectObjectsResult selectResult;
    try {
        selectResult = *DoWithRetry<NYP::NClient::TSelectObjectsResult, NYP::NClient::TResponseError>(
            [&] {
                return client.template SelectObjects<NClient::TSemaphoreSet>(
                    SEMAPHORE_SET_ATTRIBUTE_SELECTORS,
                    filter,
                    {},
                    timestamp
                ).GetValue(client.Options().Timeout() * 2);
            },
            TRetryOptions().WithCount(3).WithSleep(TDuration::MilliSeconds(250)).WithIncrement(TDuration::MilliSeconds(250)),
            /* throwLast = */ true
        );
        INFO_LOG << "Done" << Endl;
    } catch (...) {
        ERROR_LOG << "Error getting semaphore sets: " << CurrentExceptionMessage() << Endl;
        throw;
    }

    ypSemaphoreSets.reserve(selectResult.Results.size());

    for (const auto& object : selectResult.Results) {
        NClient::TSemaphoreSet ypSemaphoreSet;
        object.Fill(ypSemaphoreSet.MutableMeta(), ypSemaphoreSet.MutableSpec(), ypSemaphoreSet.MutableLabels());
        ypSemaphoreSets.push_back(std::move(ypSemaphoreSet));
    }
    return ypSemaphoreSets;
}

THashMap<TString, TVector<NClient::TSemaphore>> SelectYpSemaphores(NClient::TClient& client, const TString& filter, const ui64 timestamp) {
    THashMap<TString, TVector<NClient::TSemaphore>> ypSemaphores;
    INFO_LOG << "Getting semaphores" << Endl;
    NYP::NClient::TSelectObjectsResult selectResult;
    try {
        selectResult = *DoWithRetry<NYP::NClient::TSelectObjectsResult, NYP::NClient::TResponseError>(
            [&] {
                return client.template SelectObjects<NClient::TSemaphore>(
                    SEMAPHORE_ATTRIBUTE_SELECTORS,
                    filter,
                    {},
                    timestamp
                ).GetValue(client.Options().Timeout() * 2);
            },
            TRetryOptions().WithCount(3).WithSleep(TDuration::MilliSeconds(250)).WithIncrement(TDuration::MilliSeconds(250)),
            /* throwLast = */ true
        );
        INFO_LOG << "Done" << Endl;
    } catch (...) {
        ERROR_LOG << "Error getting semaphore sets: " << CurrentExceptionMessage() << Endl;
        throw;
    }

    for (const auto& object : selectResult.Results) {
        NClient::TSemaphore ypSemaphore;
        object.Fill(ypSemaphore.MutableMeta(), ypSemaphore.MutableSpec(), ypSemaphore.MutableLabels());
        TString semaphoreSetId = ypSemaphore.Meta().semaphore_set_id();
        ypSemaphores[std::move(semaphoreSetId)].push_back(std::move(ypSemaphore));
    }
    return ypSemaphores;
}

TVector<TSemaphore> SelectSemaphores(NClient::TClient& client, const TStringBuf serviceId) {
    INFO_LOG << "Getting timestamp" << Endl;
    ui64 timestamp;
    try {
        timestamp = *DoWithRetry<ui64, NYP::NClient::TResponseError>(
            [&client] { return client.GenerateTimestamp().GetValue(client.Options().Timeout() * 2); },
            TRetryOptions().WithCount(3).WithSleep(TDuration::MilliSeconds(250)).WithIncrement(TDuration::MilliSeconds(250)),
            /* throwLast */ true
        );
        INFO_LOG << "Done. Timestamp: " << timestamp << Endl;
    } catch (...) {
        ERROR_LOG << "Error getting timestamp: " << CurrentExceptionMessage() << Endl;
        throw;
    }

    TString filter = GetFilter(serviceId);

    TVector<NClient::TSemaphoreSet> ypSemaphoreSets = SelectYpSemaphoreSets(client, filter, timestamp);
    THashMap<TString, TVector<NClient::TSemaphore>> ypSemaphores = SelectYpSemaphores(client, filter, timestamp);

    TVector<TSemaphore> result;
    result.reserve(ypSemaphoreSets.size());

    for (auto& ypSemaphoreSet : ypSemaphoreSets) {
        TSemaphore semaphore;
        semaphore.SemaphoreId = ypSemaphoreSet.Meta().id();
        semaphore.YpSemaphoreSet = std::move(ypSemaphoreSet);
        semaphore.YpSemaphores = std::move(ypSemaphores[semaphore.SemaphoreId]);
        Sort(semaphore.YpSemaphores,
            [](const NClient::TSemaphore& lhs, const NClient::TSemaphore& rhs) -> bool {
                return lhs.Meta().id() < rhs.Meta().id();
            }
        );
        result.push_back(std::move(semaphore));
    }

    Sort(result, SemaphoreCmp);

    return result;
}

TVector<TSemaphore> GetSemaphoresFromClusterConfig(const TYpClusterConfig& ypClusterConfig, const TStringBuf serviceId, const TVector<TString>& owners) {
    TVector<TSemaphore> result;
    result.reserve(ypClusterConfig.GetSemaphoreSetList().size());

    for (const auto& semaphoreSetConfig : ypClusterConfig.GetSemaphoreSetList()) {
        TSemaphore semaphore;
        semaphore.SemaphoreId = TString::Join(serviceId, "__", semaphoreSetConfig.GetName());
        semaphore.YpSemaphoreSet.MutableMeta()->set_id(semaphore.SemaphoreId);
        if (!owners.empty()) {
            NClient::NApi::NProto::TAccessControlEntry& accessControlEntry = *semaphore.YpSemaphoreSet.MutableMeta()->add_acl();
            accessControlEntry.set_action(NClient::NApi::NProto::EAccessControlAction::ACA_ALLOW);
            accessControlEntry.add_permissions(NClient::NApi::NProto::EAccessControlPermission::ACP_READ);
            accessControlEntry.add_permissions(NClient::NApi::NProto::EAccessControlPermission::ACA_WRITE);
            for (const TString& owner : owners) {
                accessControlEntry.add_subjects(owner);
            }
            semaphore.NonDefaultOwners = true;
        }
        (*semaphore.YpSemaphoreSet.MutableLabels())["service_id"] = serviceId;
        int allBudget = semaphoreSetConfig.GetBudget();
        int number = semaphoreSetConfig.GetDivide();
        int singleBudget = allBudget / number;
        for (int i = 0; i < number; ++i) {
            NClient::TSemaphore ypSemaphore;
            ypSemaphore.MutableMeta()->set_id(TString::Join(semaphore.SemaphoreId, "__semaphore-", ToString(i)));
            ypSemaphore.MutableMeta()->set_semaphore_set_id(semaphore.SemaphoreId);
            if (i + 1 == number) {
                singleBudget = allBudget;
            }
            ypSemaphore.MutableSpec()->set_budget(singleBudget);
            (*ypSemaphore.MutableLabels())["service_id"] = serviceId;
            allBudget -= singleBudget;
            semaphore.YpSemaphores.push_back(std::move(ypSemaphore));
        }
        Sort(semaphore.YpSemaphores,
            [](const NClient::TSemaphore& lhs, const NClient::TSemaphore& rhs) -> bool {
                return lhs.Meta().id() < rhs.Meta().id();
            }
        );
        result.push_back(std::move(semaphore));
    }

    Sort(result, SemaphoreCmp);

    return result;
}

bool CreateYpSemaphore(NClient::TClient& client, const NClient::TSemaphore& ypSemaphore) {
    INFO_LOG << "Creating semaphore. Id: " << ypSemaphore.Meta().id() << ", budget: " << ypSemaphore.Spec().budget() << Endl;
    try {
        DoWithRetry<NYP::NClient::TResponseError> (
            [&client, &ypSemaphore] {
                client.CreateObject(ypSemaphore).GetValue(client.Options().Timeout() * 2);
            },
            TRetryOptions().WithCount(15).WithSleep(TDuration::MilliSeconds(100)).WithIncrement(TDuration::MilliSeconds(100)),
            /* throwLast */ true
        );
        INFO_LOG << "Done" << Endl;
    } catch (...) {
        ERROR_LOG << "Semaphore creation error: " << CurrentExceptionMessage() << Endl;
        return false;
    }
    return true;
}

bool RemoveYpSemaphore(NClient::TClient& client, const NClient::TSemaphore& ypSemaphore) {
    INFO_LOG << "Removing semaphore set. Id: " << ypSemaphore.Meta().id() << Endl;
    try {
        DoWithRetry<NYP::NClient::TResponseError> (
            [&client, &ypSemaphore] {
                client.RemoveObject<NClient::TSemaphore>(ypSemaphore.Meta().id()).GetValue(client.Options().Timeout() * 2);
            },
            TRetryOptions().WithCount(15).WithSleep(TDuration::MilliSeconds(100)).WithIncrement(TDuration::MilliSeconds(100)),
            /* throwLast */ true
        );
        INFO_LOG << "Done" << Endl;
    } catch (...) {
        ERROR_LOG << "Semaphore set deletion error: " << CurrentExceptionMessage() << Endl;
        return false;
    }
    return true;
}

bool UpdateYpSemaphore(NClient::TClient& client, const NClient::TSemaphore& oldYpSemaphore, const NClient::TSemaphore& newYpSemaphore) {
    if (oldYpSemaphore.Spec().budget() != newYpSemaphore.Spec().budget()) {
        INFO_LOG << "Updating semaphore budget. "
                 << "Semaphore id: " << oldYpSemaphore.Meta().id()
                 << ", old budget: " << oldYpSemaphore.Spec().budget()
                 << ", new budget: " << newYpSemaphore.Spec().budget() << Endl;
        try {
            DoWithRetry<NYP::NClient::TResponseError> (
                [&client, &newYpSemaphore] {
                    client.UpdateObject<NClient::TSemaphore>(
                        newYpSemaphore.Meta().id(),
                        {NYP::NClient::TSetRequest("/spec/budget", newYpSemaphore.Spec().budget())},
                        {}
                    ).GetValue(client.Options().Timeout() * 2);
                },
                TRetryOptions().WithCount(15).WithSleep(TDuration::MilliSeconds(100)).WithIncrement(TDuration::MilliSeconds(100)),
                /* throwLast */ true
            );
            INFO_LOG << "Done" << Endl;
        } catch (...) {
            ERROR_LOG << "Semaphore update error: " << CurrentExceptionMessage() << Endl;
            return false;
        }
    }
    return true;
}

bool CreateSemaphore(NClient::TClient& client, const TSemaphore& semaphore) {
    INFO_LOG << "Creating semaphore set. Id: " << semaphore.SemaphoreId << Endl;
    try {
        DoWithRetry<NYP::NClient::TResponseError> (
            [&client, &semaphore] {
                client.CreateObject(semaphore.YpSemaphoreSet).GetValue(client.Options().Timeout() * 2);
            },
            TRetryOptions().WithCount(3).WithSleep(TDuration::MilliSeconds(250)).WithIncrement(TDuration::MilliSeconds(250)),
            /* throwLast */ true
        );
        INFO_LOG << "Done" << Endl;
    } catch (...) {
        ERROR_LOG << "Semaphore set creation error: " << CurrentExceptionMessage() << Endl;
        return false;
    }

    bool isOk = true;

    for (const auto& ypSemaphore : semaphore.YpSemaphores) {
        isOk &= CreateYpSemaphore(client, ypSemaphore);
    }

    return isOk;
}

bool RemoveSemaphore(NClient::TClient& client, const TSemaphore& semaphore) {
    INFO_LOG << "Removing semaphore set. Id: " << semaphore.SemaphoreId << Endl;
    try {
        DoWithRetry<NYP::NClient::TResponseError> (
            [&client, &semaphore] { 
                client.RemoveObject<NClient::TSemaphoreSet>(semaphore.SemaphoreId).GetValue(client.Options().Timeout() * 2);
            },
            TRetryOptions().WithCount(15).WithSleep(TDuration::MilliSeconds(100)).WithIncrement(TDuration::MilliSeconds(100)),
            /* throwLast */ true
        );
        INFO_LOG << "Done" << Endl;
    } catch (...) {
        ERROR_LOG << "Semaphore set deletion error: " << CurrentExceptionMessage() << Endl;
        return false;
    }
    return true;
}

bool UpdateSemaphore(NClient::TClient& client, const TSemaphore& oldSemaphore, const TSemaphore& newSemaphore) {
    if (newSemaphore.NonDefaultOwners) {
        INFO_LOG << "Updating semaphore set owners. "
                 << "Semaphore set id: " << newSemaphore.YpSemaphoreSet.Meta().id() << Endl;
        try {
            DoWithRetry<NYP::NClient::TResponseError> (
                [&client, &newSemaphore] {
                    client.UpdateObject<NClient::TSemaphoreSet>(
                        newSemaphore.YpSemaphoreSet.Meta().id(),
                        {NYP::NClient::TSetRequest("/meta/acl", newSemaphore.YpSemaphoreSet.Meta().acl())},
                        {}
                    ).GetValue(client.Options().Timeout() * 2);
                },
                TRetryOptions().WithCount(3).WithSleep(TDuration::MilliSeconds(250)).WithIncrement(TDuration::MilliSeconds(250)),
                /* throwLast */ true
            );
            INFO_LOG << "Done" << Endl;
        } catch (...) {
            ERROR_LOG << "Semaphore set update error: " << CurrentExceptionMessage() << Endl;
            return false;
        }
    }
    auto oldElementsIt = oldSemaphore.YpSemaphores.begin();
    auto newElementsIt = newSemaphore.YpSemaphores.begin();
    bool isOk = true;
    while (oldElementsIt != oldSemaphore.YpSemaphores.end() || newElementsIt != newSemaphore.YpSemaphores.end()) {
        if (oldElementsIt == oldSemaphore.YpSemaphores.end()) {
            isOk &= CreateYpSemaphore(client, *newElementsIt);
            newElementsIt++;
        } else if (newElementsIt == newSemaphore.YpSemaphores.end()) {
            isOk &= RemoveYpSemaphore(client, *oldElementsIt);
            oldElementsIt++;
        } else {
            if (oldElementsIt->Meta().id() < newElementsIt->Meta().id()) {
                isOk &= RemoveYpSemaphore(client, *oldElementsIt);
                oldElementsIt++;
            } else if (oldElementsIt->Meta().id() > newElementsIt->Meta().id()) {
                isOk &= CreateYpSemaphore(client, *newElementsIt);
                newElementsIt++;
            } else {
                isOk &= UpdateYpSemaphore(client, *oldElementsIt, *newElementsIt);
                oldElementsIt++;
                newElementsIt++;
            }
        }
    }
    return isOk;
}

} // NYP::NYPReplica::NSemaphoreUpdater


