#include "updater.h"

#include "semaphore.h"

#include <yp/cpp/yp/fwd.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/retry/retry.h>

namespace NYP::NYPReplica::NSemaphoreUpdater {

bool ApplyDiff(
    NClient::TClient& client,
    const TVector<TSemaphore>& oldSemaphores,
    const TVector<TSemaphore>& newSemaphores
) {
    bool isOk = true;

    INFO_LOG << "Start of the update cycle" << Endl;

    auto oldElementsIt = oldSemaphores.begin();
    auto newElementsIt = newSemaphores.begin();

    while (oldElementsIt != oldSemaphores.end() || newElementsIt != newSemaphores.end()) {
        if (oldElementsIt == oldSemaphores.end()) {
            isOk &= CreateSemaphore(client, *newElementsIt);
            newElementsIt++;
        } else if (newElementsIt == newSemaphores.end()) {
            isOk &= RemoveSemaphore(client, *oldElementsIt);
            oldElementsIt++;
        } else {
            if (oldElementsIt->SemaphoreId < newElementsIt->SemaphoreId) {
                isOk &= RemoveSemaphore(client, *oldElementsIt);
                oldElementsIt++;
            } else if (oldElementsIt->SemaphoreId > newElementsIt->SemaphoreId) {
                isOk &= CreateSemaphore(client, *newElementsIt);
                newElementsIt++;
            } else {
                isOk &= UpdateSemaphore(client, *oldElementsIt, *newElementsIt);
                oldElementsIt++;
                newElementsIt++;
            }
        }
    }

    INFO_LOG << "End of the update cycle" << Endl;
    return isOk;
}

int UpdateSemaphores(const TConfig& config, const ui64 retryCount, const TString& ypToken) {
    const TString serviceId = config.GetServiceId();
    const TVector<TString> owners(config.owners().begin(), config.owners().end());
    Y_ENSURE(serviceId);
    bool isOk = true;
    for (const auto& clusterConfig : config.GetYpClusterConfigs()) {
        INFO_LOG << "Updating semaphores in " << clusterConfig.GetName() << " cluster" << Endl;
        const NYP::NClient::TClientOptions clientOptions = NYP::NClient::TClientOptions()
            .SetAddress(clusterConfig.GetName())
            .SetEnableSsl(true)
            .SetEnableBalancing(true)
            .SetTimeout(TDuration::Seconds(1))
            .SetToken(ypToken);
        NClient::TClient client(clientOptions);

        auto semaphoresFromClusterConfig = GetSemaphoresFromClusterConfig(clusterConfig, serviceId, owners);

        for (ui64 i = 0; i < retryCount; ++i) {
            INFO_LOG << "Attempt number " << i << Endl;

            if (ApplyDiff(client, SelectSemaphores(client, serviceId), semaphoresFromClusterConfig)) {
                INFO_LOG << "End of update in " << clusterConfig.GetName() << " cluster" << Endl;
                break;
            } else if (i + 1 == retryCount) {
                ERROR_LOG << "Failed to fully update " << clusterConfig.GetName() << " cluster" << Endl;
                isOk = false;
            } else {
                Sleep(TDuration::Seconds(1));
            }
        }
    }
    return isOk ? EXIT_SUCCESS : EXIT_FAILURE;
}

} // NYP::NYPReplica::NSemaphoreUpdater {
