#pragma once

#include "backup.h"
#include "data_model.h"
#include "make_log_events.h"
#include "sensors.h"
#include "storage.h"
#include "table.h"
#include "update.h"
#include "util.h"

#include <infra/libs/yp_replica/protos/backup/backup.pb.h>
#include <infra/libs/yp_replica/protos/events_decl.ev.pb.h>

#include <infra/libs/background_thread/background_thread.h>
#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/self_closing_logger.h>
#include <infra/libs/sensors/sensor_group.h>
#include <infra/libs/updatable_proto_config/accessor.h>
#include <infra/libs/yp_replica/protos/config/config.pb.h>
#include <infra/libs/yp_updates_coordinator/client/client.h>

#include <yp/cpp/yp/client.h>
#include <yp/cpp/yp/error.h>
#include <yp/cpp/yp/request_model.h>
#include <yp/cpp/yp/selector_result.h>
#include <yp/cpp/yp/semaphore.h>

#include <yp/client/api/misc/public.h>

#include <library/cpp/protobuf/interop/cast.h>
#include <library/cpp/retry/retry.h>

#include <util/datetime/cputimer.h>
#include <util/folder/iterator.h>
#include <util/generic/maybe.h>
#include <util/generic/serialized_enum.h>
#include <util/generic/strbuf.h>
#include <util/system/guard.h>
#include <util/system/mutex.h>
#include <util/system/rwlock.h>

namespace NYP::NYPReplica {

    template <typename TObject>
    struct TReplicaSelectionResult {
        TVector<TObject> Objects;
        ui64 YpTimestamp = 0;
        ui64 Hash = 0;
    };

    template <typename TReplicaObject>
    using TListElementsResult = TVector<std::pair<TString, TVector<TStorageElement<TReplicaObject>>>>;

    template<typename TReplicaObject>
    ui64 GetByteSizeOfReplicaObjects(const TVector<TReplicaObject>& data) {
        ui64 result = 0;
        for (const TReplicaObject& item : data) {
            result += item.Values().ByteSizeLong();
        }
        return result;
    }

    struct IUpdateCallback {
        virtual void Do(ui64 /* timestamp */, const TVector<TUpdate<TEndpointSetReplicaObject>>& updates) { Do(updates); }
        virtual void Do(ui64 /* timestamp */, const TVector<TUpdate<TEndpointReplicaObject>>& updates) { Do(updates); }
        virtual void Do(ui64 /* timestamp */, const TVector<TUpdate<TPodReplicaObject>>& updates) { Do(updates); }
        virtual void Do(ui64 /* timestamp */, const TVector<TUpdate<TPodWithNodeIdKeyReplicaObject>>& updates) { Do(updates); }
        virtual void Do(ui64 /* timestamp */, const TVector<TUpdate<TDnsRecordSetReplicaObject>>& updates) { Do(updates); }
        virtual void Do(ui64 /* timestamp */, const TVector<TUpdate<TDnsZoneReplicaObject>>& updates) { Do(updates); }

        virtual ~IUpdateCallback() { }

    private:
        virtual void Do(const TVector<TUpdate<TEndpointSetReplicaObject>>&) { }
        virtual void Do(const TVector<TUpdate<TEndpointReplicaObject>>&) { }
        virtual void Do(const TVector<TUpdate<TPodReplicaObject>>&) {}
        virtual void Do(const TVector<TUpdate<TPodWithNodeIdKeyReplicaObject>>&) {}
        virtual void Do(const TVector<TUpdate<TDnsRecordSetReplicaObject>>&) { }
        virtual void Do(const TVector<TUpdate<TDnsZoneReplicaObject>>&) { }
    };

    template <typename TReplicaObject>
    using TFilterFunction = std::function<bool(const TStringBuf, const TVector<TStorageElement<TReplicaObject>>&)>;
    template <typename TReplicaObject>
    using TStopFunction = std::function<bool(const TStringBuf, const TVector<TStorageElement<TReplicaObject>>&)>;

    template <typename... TReplicaObjects>
    class TYPReplicaSnapshot;

    template <typename... TReplicaObjects>
    class TYPReplica {
    public:
        using TReplicaStorage = TStorage;
        using TReplicaSnapshot = TYPReplicaSnapshot<TReplicaObjects...>;

        template <typename TReplicaObject>
        struct TListOptions {
            TMaybe<TReplicaSnapshot> Snapshot;
            TFilterFunction<TReplicaObject> Filter;
            TStopFunction<TReplicaObject> Stop;
            ui64 Limit = 0;
            ESeekType SeekType = ESeekType::ToFirst;
            TString SeekKey;
            TTableInfo<TReplicaObject> TableInfo = TTableInfo<TReplicaObject>(TString(GetDefaultRuleID<TReplicaObject>()));
        };

    private:
        static TErrorDesc MakeErrorDesc(const TError& error) {
            return TErrorDesc(error.Code(), error.SubCode(), error.Message());
        }

        static TErrorDesc MakeErrorDesc(const TStatus& status) {
            return MakeErrorDesc(status.Error());
        }

        friend class TYPReplicaSnapshot<TReplicaObjects...>;

        class TInternalSnapshot {
        public:
            ~TInternalSnapshot() {
                Release();
            }

            void Release() {
                Snapshot.Reset();
                Storage.Reset();
            }

            TAtomicSharedPtr<TStorage> Storage;
            TAtomicSharedPtr<TSnapshot> Snapshot;
        };

        class TDisableBackupCreation {
        public:
            TDisableBackupCreation(TYPReplica<TReplicaObjects...>* replica)
                : Replica_(replica)
                , WasRunning_(Replica_->BackupCreator_->IsRunning())
            {
                Replica_->StopBackup();
            }

            ~TDisableBackupCreation() {
                if (WasRunning_) {
                    Replica_->StartBackup();
                }
            }

        private:
            TYPReplica<TReplicaObjects...>* const Replica_;
            const bool WasRunning_;
        };

        class TCurrentStateUpdater {
        public:
            TCurrentStateUpdater(TYPReplica<TReplicaObjects...>* replica, NInfra::TLogFramePtr logFrame)
                : Replica_(replica)
                , LogFrame_(logFrame)
            {
            }

            ~TCurrentStateUpdater() {
                Replica_->TryUpdateCurrentState(LogFrame_);
            }

        private:
            TYPReplica<TReplicaObjects...>* const Replica_;
            NInfra::TLogFramePtr LogFrame_;
        };

        class TYpSemaphoreGuard {
        public:
            TYpSemaphoreGuard(NClient::TClient& client, const TSemaphoreConfig& semaphoreConfig, const NInfra::TSensorGroup& sensorGroup)
                : SemaphoreSetGuard_(NClient::TSemaphoreSetGuard(
                    client,
                    semaphoreConfig.GetID(),
                    NClient::GetDefaultSemaphoreSetOptions()
                ))
                , SemaphoreSensor_(NInfra::TIntGaugeSensor(
                    sensorGroup,
                    NSensors::SEMAPHORE_COUNTER,
                    {{NSensors::SEMAPHORE_ID, semaphoreConfig.GetID()}}
                ))
            {
                SemaphoreSensor_.Add(1);
            }

            bool IsAcquired() const {
                return SemaphoreSetGuard_.IsAcquired();
            }

            ~TYpSemaphoreGuard() {
                SemaphoreSensor_.Add(-1);
            }

        private:
            NClient::TSemaphoreSetGuard SemaphoreSetGuard_;
            NInfra::TIntGaugeSensor SemaphoreSensor_;
        };

        class TUpdatesDisablerWithSemaphore {
        public:
            TUpdatesDisablerWithSemaphore(TAtomicSharedPtr<TUpdatesDisabler> updatesDisabler)
                : UpdatesDisabler_(std::move(updatesDisabler))
            {
            }

            void AcquireSemaphore(NClient::TClient& client, const TSemaphoreConfig& semaphoreConfig, const NInfra::TSensorGroup& sensorGroup, NInfra::TLogFramePtr logFrame) {
                try {
                    INFRA_LOG_INFO(TAcquireSemaphoreStart(semaphoreConfig.GetID()));
                    Semaphore_.Reset(new TYpSemaphoreGuard(client, semaphoreConfig, sensorGroup));
                    SemaphoreSetId_ = semaphoreConfig.GetID();
                    INFRA_LOG_INFO(TAcquireSemaphoreSuccess(semaphoreConfig.GetID()));
                } catch (...) {
                    Status_ = TStatus::SemaphoreAcquireError(semaphoreConfig.GetID(), CurrentExceptionMessage());
                }
            }

            void AcquireSemaphoreIfEmpty(NClient::TClient& client, const TSemaphoreConfig& semaphoreConfig, const NInfra::TSensorGroup& sensorGroup, NInfra::TLogFramePtr logFrame) {
                if (!Semaphore_) {
                    AcquireSemaphore(client, semaphoreConfig, sensorGroup, logFrame);
                }
            }

            bool UpdatesDisabled() const {
                if (!Status_ || (Semaphore_ && !Semaphore_->IsAcquired())) {
                    return false;
                }
                return UpdatesDisabler_->UpdatesDisabled();
            }

            void ThrowIfUpdatesDisabled() const {
                UpdatesDisabler_->ThrowIfUpdatesDisabled();
                Status_.ThrowIfError();
                if (Semaphore_ && !Semaphore_->IsAcquired()) {
                    TStatus::SemaphorePingError(SemaphoreSetId_).ThrowIfError();
                }
            }

        private:
            TStatus Status_;
            TString SemaphoreSetId_;
            TAtomicSharedPtr<TUpdatesDisabler> UpdatesDisabler_;
            THolder<TYpSemaphoreGuard> Semaphore_;
        };

        friend class TDisableBackupCreation;

    public:
        TYPReplica(
            NUpdatableProtoConfig::TAccessor<TYPReplicaConfig> replicaConfig,
            NUpdatableProtoConfig::TAccessor<TYPClusterConfig> clusterConfig,
            TTableRulesHolder<TReplicaObjects...>&& tableRulesHolder,
            const TString& ypToken,
            NInfra::TLogger& logger,
            ui64 storageFormatVersion
        );

        ~TYPReplica();

        TString Name() const;

        template <typename TReplicaObject>
        bool ContainsTable(const TTableInfo<TReplicaObject>& tableInfo) const;

        template <typename TReplicaObject>
        TListElementsResult<TReplicaObject> ListElements(const TListOptions<TReplicaObject>& options = {}) const;

        TMaybe<ui64> GetYpTimestamp(const TMaybe<TReplicaSnapshot> replicaSnapshot = Nothing()) const;

        template <typename TReplicaObject>
        TMaybe<TReplicaSelectionResult<typename TReplicaObject::TObject>> GetByKey(const TStringBuf key, const TMaybe<TReplicaSnapshot> replicaSnapshot = Nothing()) const;

        TReplicaSnapshot GetReplicaSnapshot() const;

        TMaybe<TDuration> Age(const TMaybe<TReplicaSnapshot> replicaSnapshot = Nothing()) const;

        void EnableUpdates();
        void DisableUpdates();

        TVector<TBackupInfo> ListBackups() const;

        TStatus RollbackToBackup(const TRollbackToBackupConfig& rollbackConfig);
        TStatus RollbackToBackup(ui64 backupId);
        TStatus RollbackToBackup(TInstant maxAcceptableTimestamp);

        void Start();

        void Stop();

        void ReconstructBalancing(NInfra::TLogFramePtr logFrame) const;

        void UpdateSensors();

    public:
        void SetUpdateCallback(THolder<IUpdateCallback>&& callback);

        template <typename TReplicaObject>
        void SetModifyReplicaElementCallback(THolder<IModifyReplicaElementCallback<TReplicaObject>>&& callback);

    protected:
        TInternalSnapshot InternalSnapshot() const;

        void InitializeBackupEngine(NInfra::TLogFramePtr logFramePtr);
        void InitializeStorage(NInfra::TLogFramePtr logFramePtr);
        bool RequestUpdates();
        void TryUpdateUntilSuccess(const TDuration& maxSleep);
        bool UpdatesDisabled() const;
        void ThrowIfUpdatesDisabled() const;

        void CreateBackup(const TCreateBackupOptions& options);
        void StartBackup();
        void StopBackup();

        void UpdateCoordinatorClient(const TUpdatesCoordinatorConfig& config);

        TAtomicSharedPtr<TStorage> Storage() const;
        TAtomicSharedPtr<TStorage> SwitchStorage(TAtomicSharedPtr<TStorage> storage);
        void UpdateActiveSnapshot();

    private:
        template <typename TReplicaObject>
        void AddToHistogram(const TStringBuf name, const ui64 value) const;

        template <typename TReplicaObject>
        TIntrusivePtr<NInfra::THistogramRateSensor> GetColumnFamilyHistogram(const TStringBuf name) const;

        TIntrusivePtr<NInfra::THistogramRateSensor> GetHistogram(const TStringBuf name) const;

        template <typename TReplicaObject>
        NInfra::TRateSensor RateSensor(const TStringBuf name) const;

        template <typename TReplicaObject>
        NInfra::TRateSensor RateSensor(const TStringBuf name, const TTableInfo<TReplicaObject>& tableInfo) const;

        template <typename TReplicaObject>
        NInfra::TIntGaugeSensor IntGaugeSensor(const TStringBuf name) const;

        template <typename TReplicaObject>
        NInfra::TIntGaugeSensor IntGaugeSensor(const TStringBuf name, const TTableInfo<TReplicaObject>& tableInfo) const;

        void InitSensors();

        void UpdateSensorsAfterUpdates(const THashMap<TString, ui64>& objectsNumber) const;

        template <EmptySet... TRestReplicaObjects>
        void InitMetaColumnFamilies() {
        }

        template <typename TReplicaObject, typename... TRestReplicaObjects>
        void InitMetaColumnFamilies();

        template <class Function>
        auto DoUsingYpSemaphore(Function&& func, const TSemaphoreConfig& semaphoreSetConfig);

        void UpdateCoordinatorClient(THolder<NYPUpdatesCoordinator::TClient>&& client);

        void StartParticipateInCoordination(NInfra::TLogFramePtr logFramePtr) const;

        bool TryUpdateCurrentState(NInfra::TLogFramePtr logFrame) const;

        bool TryUpdateTimestampUpdateStatus(ui64 timestamp, NYPUpdatesCoordinator::TUpdateStatus status, NInfra::TLogFramePtr logFrame) const;

        ui64 GetTargetState(NInfra::TLogFramePtr logFrame) const;

        ui64 GetTargetState(const TUpdatesCoordinatorConfig& config, NInfra::TLogFramePtr logFrame) const;

        TString GetFilter() const;

        TString GetWatchLog() const;

        ui64 GenerateYpTimestamp(const TGenerateTimestampConfig& config, NInfra::TLogFramePtr logFrame) const;

        /// Second element of returned pair is the number of selected objects
        template <typename TReplicaObject>
        std::pair<TMap<TString, TStorageElement<TReplicaObject>>, ui64> SelectAllObjects(NInfra::TLogFramePtr logFrame, ui64 timestamp, const TUpdatesDisablerWithSemaphore& updatesDisabler);

        template <typename TReplicaObject>
        TVector<TMaybe<TReplicaObject>> SelectObjects(NInfra::TLogFramePtr logFrame, const TVector<TString>& objectIds, ui64 timestamp, const TUpdatesDisablerWithSemaphore& updatesDisabler);

        template <typename TReplicaObject>
        NYP::NClient::TWatchObjectsResult WatchObjects(NInfra::TLogFramePtr logFrame, ui64 startTimestamp, ui64 timestamp, const TUpdatesDisablerWithSemaphore& updatesDisabler);

        template <typename TReplicaObject>
        TUpdates<TReplicaObject> SelectAllAndGetUpdates(const TStringBuf columnFamilyName, NInfra::TLogFramePtr logFrame, const ui64 newTimestamp, TUpdatesDisablerWithSemaphore& updatesDisabler);

        template <typename TReplicaObject>
        TUpdates<TReplicaObject> SelectOnlyUpdates(const TStringBuf columnFamilyName, NInfra::TLogFramePtr logFrame, const ui64 currentTimestamp, const ui64 newTimestamp, const TUpdatesDisablerWithSemaphore& updatesDisabler);

        template <typename TReplicaObject>
        TUpdates<TReplicaObject> GetUpdates(const TStringBuf columnFamilyName, NInfra::TLogFramePtr logFrame, const ui64 currentTimestamp, const TMaybe<ui64>& newYpTimestamp, TUpdatesDisablerWithSemaphore& updatesDisabler);

        template <EmptySet... TRestReplicaObjects>
        std::tuple<TUpdates<TRestReplicaObjects>...> GetUpdates(
            NInfra::TLogFramePtr /* logFrame */,
            const ui64 /* currentTimestamp */,
            TUpdatesDisablerWithSemaphore& /* updatesDisabler */
        ) {
        }

        template <typename TReplicaObject, typename... TRestReplicaObjects>
        std::tuple<TUpdates<TReplicaObject>, TUpdates<TRestReplicaObjects>...> GetUpdates(NInfra::TLogFramePtr logFrame, const ui64 currentTimestamp, TUpdatesDisablerWithSemaphore& updatesDisabler);

        TCreateBackupOptions GenerateBackupOptions() const;

    private:
        NUpdatableProtoConfig::TAccessor<TYPReplicaConfig> ReplicaConfig_;
        NUpdatableProtoConfig::TAccessor<TYPClusterConfig> ClusterConfig_;
        const TString YpAddress_;

        const ui64 StorageFormatVersion_;
        TVector<TString> StorageColumnFamilies_;
        TVector<TColumnFamilyResolveFunc> StorageColumnFamilyResolveFuncs_;
        TTableManagers<TReplicaObjects...> TableManagers_;
        TRWMutex StorageMutex_;
        TAtomicSharedPtr<TStorage> Storage_;
        TAtomicSharedPtr<TSnapshot> ActiveSnapshot_;
        THolder<TBackupEngine> BackupEngine_;

        std::atomic<bool> UseCache_ = {false};

        THolder<NClient::TClient> YPClient_;

        TRWMutex UpdatesCoordinatorClientMutex_;
        THolder<NYPUpdatesCoordinator::TClient> UpdatesCoordinatorClient_;

        TMutex LockUpdates_;

        TAtomicSharedPtr<TUpdatesDisabler> UpdatesDisabler_;

        TMutex StartStopMutex_;
        std::atomic<bool> Started_ = {false};
        std::atomic<bool> Stopped_ = {false};

        const TDuration UpdatingFrequency_;

        THolder<NInfra::IBackgroundThread> UpdatesRequester_;
        THolder<NInfra::IBackgroundThread> BackupCreator_;

        NInfra::TLogger& Logger_;

        const NInfra::TSensorGroup SensorGroup_;
        THashMap<TStringBuf, THolder<NInfra::ISensor>> Sensors_;

        size_t CounterOfFails_ = 0;

        size_t SelectAllRequestsInARow_ = 0;
        size_t WatchRequestsInARow_ = 0;

    protected:
        THashMap<TStringBuf, TIntrusivePtr<NInfra::THistogramRateSensor>> Histograms_;
        TMap<TString, THashMap<TStringBuf, TIntrusivePtr<NInfra::THistogramRateSensor>>> HistogramsByColumnFamily_;

    private:
        THolder<IUpdateCallback> UpdateCallback_;
    };

    template <typename... TReplicaObjects>
    class TYPReplicaSnapshot {
    private:
        friend class TYPReplica<TReplicaObjects...>;
        using TInternalSnapshot = typename TYPReplica<TReplicaObjects...>::TInternalSnapshot;

        TYPReplicaSnapshot(TInternalSnapshot snapshot)
            : Snapshot_(std::move(snapshot))
        {
        }

    public:
        void Release() {
            Snapshot_.Release();
        }

    private:
        TInternalSnapshot Snapshot_;
    };

    template <typename... TReplicaObjects>
    TYPReplica<TReplicaObjects...>::TYPReplica(
        NUpdatableProtoConfig::TAccessor<TYPReplicaConfig> replicaConfig,
        NUpdatableProtoConfig::TAccessor<TYPClusterConfig> clusterConfig,
        TTableRulesHolder<TReplicaObjects...>&& tableRulesHolder,
        const TString& ypToken,
        NInfra::TLogger& logger,
        ui64 storageFormatVersion
    )
        : ReplicaConfig_(std::move(replicaConfig))
        , ClusterConfig_(std::move(clusterConfig))
        , YpAddress_(CONFIG_SNAPSHOT_VALUE(ClusterConfig_, GetAddress()))
        , StorageFormatVersion_(storageFormatVersion)
        , TableManagers_(std::move(tableRulesHolder))
        , UseCache_(CONFIG_SNAPSHOT_VALUE(ReplicaConfig_, GetUseCache()))
        , UpdatesDisabler_(MakeAtomicShared<TUpdatesDisabler>(CONFIG_SNAPSHOT_VALUE(ClusterConfig_, GetEnableUpdates())))
        , UpdatingFrequency_(TDuration::Parse(CONFIG_SNAPSHOT_VALUE(ClusterConfig_, GetUpdatingFrequency())))
        , UpdatesRequester_(MakeHolder<NInfra::TBackgroundThread>([this] { RequestUpdates(); }, UpdatingFrequency_))
        , Logger_(logger)
        , SensorGroup_(
            WITH_CONFIG_SNAPSHOT(ReplicaConfig_, snapshot) {
                return NInfra::TSensorGroup(
                    snapshot->HasName() ? TString::Join(NSensors::NAMESPACE, ".", snapshot->GetName()) : NSensors::NAMESPACE
                ).AddLabels({{NSensors::YP_CLUSTER, CONFIG_SNAPSHOT_VALUE(ClusterConfig_, GetName())}});
            }
        )
    {
        const auto initialReplicaConfig = *ReplicaConfig_.Get();
        const auto initialClusterConfig = *ClusterConfig_.Get();

        NClient::TClientOptions clientOptions;
        clientOptions
            .SetAddress(initialClusterConfig.GetAddress())
            .SetEnableBalancing(initialClusterConfig.GetBalancing())
            .SetEnableSsl(initialClusterConfig.GetEnableSsl())
            .SetTimeout(TDuration::Parse(initialClusterConfig.GetTimeout()))
            .SetToken(ypToken)
            .SetEnableMastersCache(initialClusterConfig.GetCacheDiscoveryResult())
            .SetUseResolvedGrpcAddress(initialClusterConfig.GetUseMasterIpAddress())
            .SetMastersCacheFilePattern(TFsPath(initialReplicaConfig.GetMastersDiscoveryCachePath()) / initialClusterConfig.GetName());
        if (initialClusterConfig.HasTargetFqdn() && !initialClusterConfig.GetTargetFqdn().empty()) {
            clientOptions.SetTargetFqdn(initialClusterConfig.GetTargetFqdn());
        }
        YPClient_ = MakeHolder<NClient::TClient>(clientOptions);

        if (const TGetTargetStateConfig getTargetStateConfig = initialClusterConfig.GetRetrieveTargetStateConfig(); getTargetStateConfig.HasCoordinator()) {
            UpdateCoordinatorClient(getTargetStateConfig.GetCoordinator());
        }

        TableManagers_.template SetCFOptionsIfEmpty<TReplicaObjects...>(GetColumnFamilyOptions(initialReplicaConfig, initialClusterConfig));
        StorageColumnFamilies_ = TableManagers_.template GetAllStableColumnFamilyNames<TReplicaObjects...>();
        StorageColumnFamilyResolveFuncs_ = TableManagers_.template GetColumnFamilyResolveFuncs<TReplicaObjects...>();

        InitMetaColumnFamilies<TReplicaObjects...>();

        TAtomicSharedPtr<TStorage> storage = MakeAtomicShared<TStorage>(TStorageOptions(
            TStorageMeta({initialClusterConfig.GetName(), StorageFormatVersion_}),
            StorageColumnFamilies_,
            MergeColumnFamilyResolverFuncs(StorageColumnFamilyResolveFuncs_),
            initialClusterConfig,
            initialReplicaConfig,
            SensorGroup_
        ));
        SwitchStorage(storage);

        if (initialReplicaConfig.HasBackupConfig()) {
            BackupEngine_ = MakeHolder<TBackupEngine>(TBackupEngineOptions(initialClusterConfig.GetName(), initialReplicaConfig.GetBackupConfig()));
        }

        BackupCreator_ = MakeHolder<NInfra::TBackgroundThread>(
            [this] { CreateBackup(GenerateBackupOptions()); },
            TDuration::Parse(initialReplicaConfig.GetBackupConfig().GetBackupFrequency())
        );

        InitSensors();

        ReplicaConfig_.SubscribeForUpdate([this](const TYPReplicaConfig& oldConfig, const TYPReplicaConfig& newConfig, const NUpdatableProtoConfig::TWatchContext& context = {}) {
            Y_UNUSED(context);
            NInfra::TLogFramePtr logFrame = Logger_.SpawnFrame();
            const TString replicaName = ReplicaConfig_.Get()->GetName();
            const auto clusterConfig = ClusterConfig_.Get();
            bool isEqual = true;
            if (oldConfig.GetStorageConfig().GetRocksDBConfig().GetMaxAllowedSpaceUsageBytes() !=
                newConfig.GetStorageConfig().GetRocksDBConfig().GetMaxAllowedSpaceUsageBytes()) {
                isEqual = false;
                INFRA_LOG_INFO(TReplicaConfigUpdateObject(
                    replicaName,
                    clusterConfig->GetName(),
                    TReplicaConfigUpdateObject::STORAGE,
                    TReplicaConfigUpdateObject::MAX_ALLOWED_SPACE_USAGE,
                    ToString(oldConfig.GetStorageConfig().GetRocksDBConfig().GetMaxAllowedSpaceUsageBytes()),
                    ToString(newConfig.GetStorageConfig().GetRocksDBConfig().GetMaxAllowedSpaceUsageBytes())
                ));
            }
            if (oldConfig.GetStorageConfig().GetRocksDBConfig().GetCompactionBufferSizeBytes() != 
                newConfig.GetStorageConfig().GetRocksDBConfig().GetCompactionBufferSizeBytes()) {
                isEqual = false;
                INFRA_LOG_INFO(TReplicaConfigUpdateObject(
                    replicaName,
                    clusterConfig->GetName(),
                    TReplicaConfigUpdateObject::STORAGE,
                    TReplicaConfigUpdateObject::COMPACTION_BUFFER_SIZE,
                    ToString(oldConfig.GetStorageConfig().GetRocksDBConfig().GetCompactionBufferSizeBytes()),
                    ToString(newConfig.GetStorageConfig().GetRocksDBConfig().GetCompactionBufferSizeBytes())));
            }
            if (oldConfig.GetStorageConfig().GetRocksDBConfig().GetStatsLevel() !=
                newConfig.GetStorageConfig().GetRocksDBConfig().GetStatsLevel()) {
                isEqual = false;
                INFRA_LOG_INFO(TReplicaConfigUpdateObject(
                    replicaName,
                    clusterConfig->GetName(),
                    TReplicaConfigUpdateObject::STORAGE,
                    TReplicaConfigUpdateObject::ROCKS_DB_STATS_LEVEL,
                    TRocksDBConfig_EStatsLevel_Name(oldConfig.GetStorageConfig().GetRocksDBConfig().GetStatsLevel()),
                    TRocksDBConfig_EStatsLevel_Name(newConfig.GetStorageConfig().GetRocksDBConfig().GetStatsLevel())
                ));
            }
            if (UseCache_.load() != newConfig.GetUseCache()) {
                isEqual = false;
                UseCache_.store(newConfig.GetUseCache());
            }
            if (!isEqual) {
                Storage_->UpdateReplicaConfig(newConfig);
            }
        });

        ClusterConfig_.SubscribeForUpdate([this](const TYPClusterConfig& oldConfig, const TYPClusterConfig& newConfig, const NUpdatableProtoConfig::TWatchContext& context = {}) {
            Y_UNUSED(context);
            bool isEqual = true;
            if (const TGetTargetStateConfig& getTargetStateConfig = newConfig.GetRetrieveTargetStateConfig(); getTargetStateConfig.HasCoordinator()) {
                if (!google::protobuf::util::MessageDifferencer::Equivalent(oldConfig.GetRetrieveTargetStateConfig(), newConfig.GetRetrieveTargetStateConfig())) {
                    isEqual = false;
                    UpdateCoordinatorClient(getTargetStateConfig.GetCoordinator());
                    if (UpdatesRequester_->IsRunning()) {
                        StartParticipateInCoordination(Logger_.SpawnFrame());
                    }
                }
            }
            NInfra::TLogFramePtr logFrame = Logger_.SpawnFrame();
            const TString replicaName = ReplicaConfig_.Get()->GetName();
            const TString clusterName = ClusterConfig_.Get()->GetName();
            if (oldConfig.GetRocksDBConfig().GetMaxAllowedSpaceUsageBytes() !=
                newConfig.GetRocksDBConfig().GetMaxAllowedSpaceUsageBytes()) {
                isEqual = false;
                INFRA_LOG_INFO(TReplicaConfigUpdateObject(
                    replicaName,
                    clusterName,
                    TReplicaConfigUpdateObject::CLUSTER,
                    TReplicaConfigUpdateObject::MAX_ALLOWED_SPACE_USAGE,
                    ToString(oldConfig.GetRocksDBConfig().GetMaxAllowedSpaceUsageBytes()),
                    ToString(newConfig.GetRocksDBConfig().GetMaxAllowedSpaceUsageBytes())));
            }
            if (oldConfig.GetRocksDBConfig().GetCompactionBufferSizeBytes() !=
                newConfig.GetRocksDBConfig().GetCompactionBufferSizeBytes()) {
                isEqual = false;
                INFRA_LOG_INFO(TReplicaConfigUpdateObject(
                    replicaName,
                    clusterName,
                    TReplicaConfigUpdateObject::CLUSTER,
                    TReplicaConfigUpdateObject::COMPACTION_BUFFER_SIZE,
                    ToString(oldConfig.GetRocksDBConfig().GetCompactionBufferSizeBytes()),
                    ToString(newConfig.GetRocksDBConfig().GetCompactionBufferSizeBytes())));
            }
            if (!isEqual) {
                Storage_->UpdateClusterConfig(newConfig);
            }
        });
    }

    template <typename... TReplicaObjects>
    TYPReplica<TReplicaObjects...>::~TYPReplica() {
        Stop();
    }

    template <typename... TReplicaObjects>
    TString TYPReplica<TReplicaObjects...>::Name() const {
        return CONFIG_SNAPSHOT_VALUE(ClusterConfig_, GetName());
    }

    template <typename... TReplicaObjects>
    template <typename TReplicaObject, typename... TRestReplicaObjects>
    void TYPReplica<TReplicaObjects...>::InitMetaColumnFamilies() {
        StorageColumnFamilies_.emplace_back(GetMetaColumnFamilyName<TReplicaObject>());
        StorageColumnFamilyResolveFuncs_.emplace_back([](const TString& columnFamilyName) -> TMaybe<TColumnFamilyOptions> {
            if (columnFamilyName == GetMetaColumnFamilyName<TReplicaObject>()) {
                return TColumnFamilyOptions(columnFamilyName);
            } else {
                return Nothing();
            }
        });
        InitMetaColumnFamilies<TRestReplicaObjects...>();
    }

    template <typename... TReplicaObjects>
    template <typename TReplicaObject>
    bool TYPReplica<TReplicaObjects...>::ContainsTable(const TTableInfo<TReplicaObject>& tableInfo) const {
        TMaybe<TString> columnFamily = TableManagers_.template GetColumnFamilyName(tableInfo);
        if (!columnFamily.Defined()) {
            return false;
        }
        auto result = Storage()->ContainsColumnFamily(columnFamily.GetRef());
        return result.IsSuccess() && result.Success();
    }

    template <typename... TReplicaObjects>
    template <typename TReplicaObject>
    TListElementsResult<TReplicaObject> TYPReplica<TReplicaObjects...>::ListElements(const TListOptions<TReplicaObject>& options) const {
        auto [storage, snapshot] = options.Snapshot.Defined() ? options.Snapshot->Snapshot_ : InternalSnapshot();

        auto [logFrame, bio] = NInfra::CreateBiographedLoggerFrame(
            Logger_,
            TListElementsStart(),
            TListElementsEnd()
        );

        TReadOptions readOptions;
        readOptions.Snapshot = snapshot;

        TMaybe<TString> columnFamily = TableManagers_.template GetColumnFamilyName(options.TableInfo);

        if (!columnFamily.Defined()) {
            TStatus status = TStatus::InvalidArgument(TStringBuilder{} << "Invalid TTableInfo (RuleId: " << options.TableInfo.RuleID
                << (options.TableInfo.TableName.Defined() ? (", TableName: " + options.TableInfo.TableName.GetRef()) : "") << ")");
            INFRA_LOG_ERROR(TListElementsError(MakeErrorDesc(status)));
            status.ThrowIfError();
        }

        TIterator it = storage->NewIterator(readOptions, columnFamily.GetRef());
        it.Seek(options.SeekType, options.SeekKey);

        TListElementsResult<TReplicaObject> result;
        result.reserve(options.Limit);
        ui64 objectsSeen = 0;
        for (; it.Valid() && (options.Limit == 0 || result.size() < options.Limit); it.Next()) {
            ++objectsSeen;
            TVector<TStorageElement<TReplicaObject>> objects = it.Value<TReplicaObject>();
            if (options.Stop && options.Stop(it.Key(), objects)) {
                break;
            }
            if (options.Filter && !options.Filter(it.Key(), objects)) {
                continue;
            }

            result.emplace_back(it.Key(), std::move(objects));
        }

        if (TStatus status = it.Status(); !status) {
            INFRA_LOG_ERROR(TListElementsError(MakeErrorDesc(status)));
            status.ThrowIfError();
        }

        INFRA_LOG_INFO(TListElementsSuccess(Name(), columnFamily.GetRef(), result.size(), objectsSeen));

        return result;
    }

    template <typename... TReplicaObjects>
    TMaybe<ui64> TYPReplica<TReplicaObjects...>::GetYpTimestamp(const TMaybe<TReplicaSnapshot> replicaSnapshot) const {
        auto [storage, snapshot] = replicaSnapshot.Defined() ? replicaSnapshot->Snapshot_ : InternalSnapshot();

        TReadOptions readOptions;
        readOptions.Snapshot = snapshot;

        return storage->GetYpTimestamp(readOptions);
    }

    template <typename... TReplicaObjects>
    template <typename TReplicaObject>
    TMaybe<TReplicaSelectionResult<typename TReplicaObject::TObject>> TYPReplica<TReplicaObjects...>::GetByKey(const TStringBuf key, const TMaybe<TReplicaSnapshot> replicaSnapshot) const {
        auto [storage, snapshot] = replicaSnapshot.Defined() ? replicaSnapshot->Snapshot_ : InternalSnapshot();

        TReadOptions readOptions;
        readOptions.UseCache = UseCache_.load();
        readOptions.Snapshot = snapshot;

        TTableInfo<TReplicaObject> tableInfo(TString(GetDefaultRuleID<TReplicaObject>()));
        TVector<TStorageElement<TReplicaObject>> objects;
        if (auto getResult = storage->template GetReplicaObjects<TReplicaObject>(readOptions, TableManagers_.template GetColumnFamilyName(tableInfo).GetRef(), key); getResult.IsError()) {
            return Nothing();
        } else {
            objects = std::move(getResult.Success());
        }

        if (objects.empty()) {
            return Nothing();
        }

        TReplicaSelectionResult<typename TReplicaObject::TObject> result;
        result.YpTimestamp = storage->GetYpTimestamp(readOptions);

        result.Objects.reserve(objects.size());
        for (const auto& object : objects) {
            result.Objects.emplace_back(object.ReplicaObject.GetObject());
        }

        return result;
    }

    template <typename... TReplicaObjects>
    TMaybe<TDuration> TYPReplica<TReplicaObjects...>::Age(const TMaybe<TReplicaSnapshot> replicaSnapshot) const {
        auto [storage, snapshot] = replicaSnapshot.Defined() ? replicaSnapshot->Snapshot_ : InternalSnapshot();

        if (!storage) {
            return Nothing();
        }

        TReadOptions readOptions;
        readOptions.Snapshot = snapshot;

        return storage->Age(readOptions);
    }
    
    template <typename... TReplicaObjects>
    void TYPReplica<TReplicaObjects...>::EnableUpdates() {
        UpdatesDisabler_->EnableUpdates();
    }

    template <typename... TReplicaObjects>
    void TYPReplica<TReplicaObjects...>::DisableUpdates() {
        UpdatesDisabler_->DisableUpdates();
    }

    template <typename... TReplicaObjects>
    TVector<TBackupInfo> TYPReplica<TReplicaObjects...>::ListBackups() const {
        if (!BackupEngine_) {
            return {};
        }

        return BackupEngine_->ListBackups();
    }

    template <typename... TReplicaObjects>
    TStatus TYPReplica<TReplicaObjects...>::RollbackToBackup(const TRollbackToBackupConfig& rollbackConfig) {
        if (!rollbackConfig.GetRollback()) {
            return TStatus::RollbackConfigNotSet();
        }

        if (rollbackConfig.HasId()) {
            return RollbackToBackup(rollbackConfig.GetId());
        } else if (rollbackConfig.HasMaxAcceptableTimestamp()) {
            return RollbackToBackup(TInstant::MilliSeconds(rollbackConfig.GetMaxAcceptableTimestamp()));
        }
        return TStatus::UnknownRollbackOptions();
    }

    template <typename... TReplicaObjects>
    TStatus TYPReplica<TReplicaObjects...>::RollbackToBackup(ui64 backupId) {
        if (!BackupEngine_) {
            return TStatus::BackupEngineNotInitialized();
        }

        TDisableUpdatesGuard disableUpdates(UpdatesDisabler_);
        TDisableBackupCreation disableBackups(this);
        TGuard<TMutex> guard(LockUpdates_);

        const TStorageOptions storageOptions = Storage()->Options();

        TStorageOptions checkpointStorageOptions = storageOptions;
        checkpointStorageOptions.Paths.StoragePath = storageOptions.Paths.StoragePath.Parent() / "checkpoints" / storageOptions.Paths.StoragePath.GetName();
        checkpointStorageOptions.DropOnDestruction = true;
        if (TStatus status = Storage()->CreateCheckpoint(checkpointStorageOptions.Paths.StoragePath); !status) {
            return status;
        }

        TAtomicSharedPtr<TStorage> roStorage = MakeAtomicShared<TStorage>(checkpointStorageOptions);
        if (TStatus status = roStorage->OpenForReadOnly(/* validate */ true); !status) {
            return status;
        }

        TAtomicSharedPtr<TStorage> storage = SwitchStorage(roStorage);

        storage->Drop();
        if (TStatus status = storage->RestoreFromBackup(backupId); !status) {
            storage->Drop();
            roStorage->CreateCheckpoint(storageOptions.Paths.StoragePath);
            storage->Open(/* validate */ false);
            SwitchStorage(storage);
            return status;
        }

        storage->Reopen(/* validate */ false).ThrowIfError();
        SwitchStorage(storage);

        for (const TBackupInfo& backup : ListBackups()) {
            if (backup.Id > backupId) {
                BackupEngine_->DeleteBackup(backup.Id).ThrowIfError();
            }
        }

        return TStatus::Ok();
    }

    template <typename... TReplicaObjects>
    TStatus TYPReplica<TReplicaObjects...>::RollbackToBackup(TInstant maxAcceptableTimestamp) {
        if (!BackupEngine_) {
            return TStatus::BackupEngineNotInitialized();
        }

        TVector<TBackupInfo> backups = ListBackups();
        const TBackupInfo* chosenBackup = nullptr;
        for (const TBackupInfo& backup : backups) {
            TInstant createTimestamp = NProtoInterop::CastFromProto(backup.Meta.GetCreateTimestamp());
            if (createTimestamp < maxAcceptableTimestamp &&
                (!chosenBackup || createTimestamp > NProtoInterop::CastFromProto(chosenBackup->Meta.GetCreateTimestamp()))) {
                chosenBackup = &backup;
            }
        }

        if (!chosenBackup) {
            return TStatus::NoSuitableBackup();
        }

        return RollbackToBackup(chosenBackup->Id);
    }

    template <typename... TReplicaObjects>
    void TYPReplica<TReplicaObjects...>::Start() {
        TGuard<TMutex> guard(StartStopMutex_);
        if (Stopped_.load()) {
            return;
        }
        NInfra::TLogFramePtr logFrame = Logger_.SpawnFrame();
        InitializeBackupEngine(logFrame);
        InitializeStorage(logFrame);

        UpdateActiveSnapshot();
        UpdatesRequester_->Start();
        StartParticipateInCoordination(logFrame);
        StartBackup();

        Started_.store(true);
    }

    template <typename... TReplicaObjects>
    void TYPReplica<TReplicaObjects...>::Stop() {
        // Stop startup process
        Stopped_.store(true);
        DisableUpdates();

        TGuard<TMutex> guard(StartStopMutex_);
        DisableUpdates();
        UpdatesRequester_->Stop();
        UpdateCoordinatorClient(nullptr); // unregister instance
        StopBackup();

        const auto replicaConfig = ReplicaConfig_.Get();
        if (replicaConfig->GetFlushToBackupOnStop()) {
            CreateBackup(GenerateBackupOptions());
        }
    }

    template <typename... TReplicaObjects>
    void TYPReplica<TReplicaObjects...>::InitializeBackupEngine(NInfra::TLogFramePtr logFrame) {
        const auto replicaConfig = ReplicaConfig_.Get();
        if (TStatus status = BackupEngine_->Open(); !status) {
            INFRA_LOG_ERROR(TReplicaBackupEngineInitFail(MakeErrorDesc(status)));
            INFRA_LOG_INFO(TReplicaBackupEngineReset());
            BackupEngine_->Drop();
        } else {
            INFRA_LOG_INFO(TReplicaBackupEngineInitSuccess());
        }
        const ui32 maxBackupsNumber = replicaConfig->GetBackupConfig().GetMaxBackupsNumber();
        INFRA_LOG_INFO(TReplicaPurgeOldBackups(maxBackupsNumber));
        BackupEngine_->PurgeOldBackups(maxBackupsNumber);
    }

    template <typename... TReplicaObjects>
    void TYPReplica<TReplicaObjects...>::InitializeStorage(NInfra::TLogFramePtr logFrame) {
        const auto clusterConfig = ClusterConfig_.Get();
        if (clusterConfig->HasRollbackToBackup()) {
            RollbackToBackup(clusterConfig->GetRollbackToBackup());
        }

        if (TStatus status = Storage()->Open(/* validate = */ true); !status) {
            INFRA_LOG_ERROR(TReplicaRestoreFromStorageFail(MakeErrorDesc(status)));
            INFRA_LOG_INFO(TReplicaStorageReset());
            Storage()->Drop();
            INFRA_LOG_INFO(TReplicaRestoreFromBackup());

            bool restoreFailed = false;
            bool openFailed = false;
            if (TStatus restoreStatus = Storage()->RestoreFromLatestBackup(); !restoreStatus) {
                restoreFailed = true;
                INFRA_LOG_ERROR(TReplicaRestoreFromBackupFail(MakeErrorDesc(restoreStatus)));
            } else if (TStatus openStatus = Storage()->Open(/* validate = */ true); !openStatus) {
                openFailed = true;
                INFRA_LOG_ERROR(TReplicaRestoreFromBackupFail(MakeErrorDesc(openStatus)));
            }

            if (restoreFailed || openFailed) {
                INFRA_LOG_INFO(TReplicaStorageReset());
                Storage()->Drop();
                Storage()->Open(/* validate = */ false).ThrowIfError();
                if (!clusterConfig->GetUpdateIfNoBackup()) {
                    return;
                }
                TryUpdateUntilSuccess(UpdatingFrequency_);
            } else {
                INFRA_LOG_INFO(TReplicaRestoreFromBackupSuccess());
            }
        }

        Storage()->Validate().ThrowIfError();
    }

    template <typename... TReplicaObjects>
    TCreateBackupOptions TYPReplica<TReplicaObjects...>::GenerateBackupOptions() const {
        TCreateBackupOptions result;
        result.FlushBeforeBackup = true;
        result.Meta.SetFormatVersion(StorageFormatVersion_);
        *result.Meta.MutableCreateTimestamp() = NProtoInterop::CastToProto(TInstant::Now());
        return result;
    }

    template <typename... TReplicaObjects>
    void TYPReplica<TReplicaObjects...>::CreateBackup(const TCreateBackupOptions& options) {
        auto [logFrame, bio] = NInfra::CreateBiographedLoggerFrame(
            Logger_,
            TReplicaCreateBackupStart(),
            TReplicaCreateBackupEnd(),
            GetHistogram(NSensors::BACKUP_CREATION_TIME)
        );
        const auto replicaConfig = ReplicaConfig_.Get();

        auto storage = Storage();
        if (!storage || !storage->IsOpened()) {
            INFRA_LOG_ERROR(TReplicaCreateBackupFail(MakeErrorDesc(TStatus::DatabaseNotOpened())));
            return;
        }

        if (!BackupEngine_) {
            INFRA_LOG_ERROR(TReplicaCreateBackupFail(MakeErrorDesc(TStatus::BackupEngineNotInitialized())));
            return;
        }

        if (TStatus status = BackupEngine_->CreateNewBackup(options, storage.Get()); !status) {
            INFRA_LOG_ERROR(TReplicaCreateBackupFail(MakeErrorDesc(status)));
            INFRA_LOG_INFO(TReplicaBackupEngineReopen());

            bool reopenFailed = false;
            bool createFailed = false;
            if (TStatus reopenStatus = BackupEngine_->Reopen(); !reopenStatus) {
                reopenFailed = true;
                INFRA_LOG_ERROR(TReplicaCreateBackupFail(MakeErrorDesc(reopenStatus)));
                INFRA_LOG_INFO(TReplicaBackupEngineReset());
            } else if (TStatus createStatus = BackupEngine_->CreateNewBackup(options, storage.Get()); !createStatus) {
                createFailed = true;
                INFRA_LOG_ERROR(TReplicaCreateBackupFail(MakeErrorDesc(createStatus)));
                INFRA_LOG_INFO(TReplicaBackupEngineReset());
            }

            if (reopenFailed || createFailed) {
                BackupEngine_->Drop();
                if (TStatus createStatus = BackupEngine_->CreateNewBackup(options, storage.Get()); !createStatus) {
                    INFRA_LOG_ERROR(TReplicaCreateBackupFail(MakeErrorDesc(createStatus)));
                    return;
                }
            }
        }
        INFRA_LOG_INFO(TReplicaCreateBackupSuccess());

        INFRA_LOG_INFO(TReplicaPurgeOldBackups(replicaConfig->GetBackupConfig().GetMaxBackupsNumber()));
        BackupEngine_->PurgeOldBackups(replicaConfig->GetBackupConfig().GetMaxBackupsNumber());
    }

    template <typename... TReplicaObjects>
    void TYPReplica<TReplicaObjects...>::StartBackup() {
        if (!BackupEngine_) {
            return;
        }

        BackupCreator_->Start();
    }

    template <typename... TReplicaObjects>
    void TYPReplica<TReplicaObjects...>::StopBackup() {
        if (!BackupEngine_) {
            return;
        }

        if (BackupEngine_->IsOpened()) {
            BackupEngine_->StopBackup();
        }
        BackupCreator_->Stop();
    }

    template <typename... TReplicaObjects>
    void TYPReplica<TReplicaObjects...>::InitSensors() {
        NInfra::TRateSensor(SensorGroup_, NSensors::MASTER_FAILURES);
        NInfra::TRateSensor(SensorGroup_, NSensors::MASTER_REQUESTS);
        NInfra::TRateSensor(SensorGroup_, NSensors::UPDATE_CALLBACK_ERROR);

        for (const auto& [replicaObjectName, columnFamily] : {std::make_pair(TReplicaObjects::NAME, TableManagers_.template GetMainColumnFamilyName<TReplicaObjects>())...}) {
            NInfra::TRateSensor(SensorGroup_, NSensors::MODIFY_ELEMENTS_ERROR, {{NSensors::REPLICA_OBJECT_TYPE, replicaObjectName}});
            for (const auto& [histogramName, baseOfHistogram, scaleOfHistogram] : NSensors::HISTOGRAMS_BY_COLUMN_FAMILY_INIT_PARAMETERS) {
                if (!HistogramsByColumnFamily_[TString{columnFamily}].contains(histogramName)) {
                    HistogramsByColumnFamily_[TString{columnFamily}].emplace(
                        histogramName,
                        MakeIntrusive<NInfra::THistogramRateSensor>(
                            SensorGroup_,
                            histogramName,
                            NMonitoring::ExponentialHistogram(NMonitoring::HISTOGRAM_MAX_BUCKETS_COUNT, baseOfHistogram, scaleOfHistogram),
                            TVector<std::pair<TStringBuf, TStringBuf>>{{NSensors::COLUMN_FAMILY, columnFamily}}
                       )
                    );
                }
            }
        }

        for (const auto& [histogramName, baseOfHistogram, scaleOfHistogram] : NSensors::HISTOGRAMS_INIT_PARAMETERS) {
            if (!Histograms_.contains(histogramName)) {
                Histograms_.emplace(
                    histogramName,
                    MakeIntrusive<NInfra::THistogramRateSensor>(
                       SensorGroup_,
                       histogramName,
                       NMonitoring::ExponentialHistogram(NMonitoring::HISTOGRAM_MAX_BUCKETS_COUNT, baseOfHistogram, scaleOfHistogram)
                   )
                );
            }
        }
    }

    template <typename... TReplicaObjects>
    template <typename TReplicaObject>
    void TYPReplica<TReplicaObjects...>::AddToHistogram(const TStringBuf name, const ui64 value) const {
        if (auto* it = HistogramsByColumnFamily_.at(TString{TableManagers_.template GetMainColumnFamilyName<TReplicaObject>()}).FindPtr(name)) {
            (*it)->Add(value);
        }
    }

    template <typename... TReplicaObjects>
    template <typename TReplicaObject>
    TIntrusivePtr<NInfra::THistogramRateSensor> TYPReplica<TReplicaObjects...>::GetColumnFamilyHistogram(const TStringBuf name) const {
        return HistogramsByColumnFamily_.at(TString{TableManagers_.template GetMainColumnFamilyName<TReplicaObject>()}).at(name);
    }

    template <typename... TReplicaObjects>
    TIntrusivePtr<NInfra::THistogramRateSensor> TYPReplica<TReplicaObjects...>::GetHistogram(const TStringBuf name) const {
        return Histograms_.at(name);
    }

    template <typename... TReplicaObjects>
    template <typename TReplicaObject>
    NInfra::TRateSensor TYPReplica<TReplicaObjects...>::RateSensor(const TStringBuf name) const {
        return NInfra::TRateSensor(SensorGroup_, name, {{NSensors::REPLICA_OBJECT_TYPE, TReplicaObject::NAME}});
    }

    template <typename... TReplicaObjects>
    template <typename TReplicaObject>
    NInfra::TRateSensor TYPReplica<TReplicaObjects...>::RateSensor(const TStringBuf name, const TTableInfo<TReplicaObject>& tableInfo) const {
        return NInfra::TRateSensor(SensorGroup_, name, {{NSensors::COLUMN_FAMILY, TableManagers_.template GetColumnFamilyName(tableInfo).GetRef()}});
    }

    template <typename... TReplicaObjects>
    template <typename TReplicaObject>
    NInfra::TIntGaugeSensor TYPReplica<TReplicaObjects...>::IntGaugeSensor(const TStringBuf name) const {
        return NInfra::TIntGaugeSensor(SensorGroup_, name, {{NSensors::REPLICA_OBJECT_TYPE, TReplicaObject::NAME}});
    }

    template <typename... TReplicaObjects>
    template <typename TReplicaObject>
    NInfra::TIntGaugeSensor TYPReplica<TReplicaObjects...>::IntGaugeSensor(const TStringBuf name, const TTableInfo<TReplicaObject>& tableInfo) const {
        return NInfra::TIntGaugeSensor(SensorGroup_, name, {{NSensors::COLUMN_FAMILY, TableManagers_.template GetColumnFamilyName(tableInfo).GetRef()}});
    }

    template <typename... TReplicaObjects>
    void TYPReplica<TReplicaObjects...>::StartParticipateInCoordination(NInfra::TLogFramePtr logFramePtr) const {
        TReadGuard guard(UpdatesCoordinatorClientMutex_);
        const auto clusterConfig = ClusterConfig_.Get();

        const TGetTargetStateConfig getTargetStateConfig = clusterConfig->GetRetrieveTargetStateConfig();
        if (getTargetStateConfig.HasCoordinator()) {
            Y_ENSURE(UpdatesCoordinatorClient_);
            UpdatesCoordinatorClient_->StartParticipateInCoordination(logFramePtr);
        }
    }

    template <typename... TReplicaObjects>
    bool TYPReplica<TReplicaObjects...>::TryUpdateCurrentState(NInfra::TLogFramePtr logFrame) const {
        TReadGuard guard(UpdatesCoordinatorClientMutex_);
        const auto clusterConfig = ClusterConfig_.Get();
        try {
            const TGetTargetStateConfig getTargetStateConfig = clusterConfig->GetRetrieveTargetStateConfig();
            if (getTargetStateConfig.HasCoordinator()) {
                Y_ENSURE(UpdatesCoordinatorClient_);
                UpdatesCoordinatorClient_->SetCurrentState(NYPUpdatesCoordinator::TTimestampClientInfo(Storage()->GetYpTimestamp(TReadOptions())), logFrame);
            }
        } catch (...) {
            return false;
        }

        return true;
    }

    template <typename... TReplicaObjects>
    bool TYPReplica<TReplicaObjects...>::TryUpdateTimestampUpdateStatus(
        ui64 timestamp,
        NYPUpdatesCoordinator::TUpdateStatus status,
        NInfra::TLogFramePtr logFrame
    ) const {
        TReadGuard guard(UpdatesCoordinatorClientMutex_);
        const auto clusterConfig = ClusterConfig_.Get();
        try {
            const TGetTargetStateConfig getTargetStateConfig = clusterConfig->GetRetrieveTargetStateConfig();
            if (getTargetStateConfig.HasCoordinator()) {
                Y_ENSURE(UpdatesCoordinatorClient_);
                UpdatesCoordinatorClient_->SetTimestampUpdateStatus(timestamp, std::move(status), logFrame);
            }
        } catch (...) {
            return false;
        }

        return true;
    }

    template <typename... TReplicaObjects>
    ui64 TYPReplica<TReplicaObjects...>::GetTargetState(NInfra::TLogFramePtr logFrame) const {
        const auto clusterConfig = ClusterConfig_.Get();
        const TGetTargetStateConfig getTargetStateConfig = clusterConfig->GetRetrieveTargetStateConfig();
        if (getTargetStateConfig.HasCoordinator()) {
            return GetTargetState(getTargetStateConfig.GetCoordinator(), logFrame);
        } else {
            return GenerateYpTimestamp(getTargetStateConfig.GetGenerateTimestamp(), logFrame);
        }
        ythrow yexception() << "Unknown target state retrieving mode";
    }

    template <typename... TReplicaObjects>
    ui64 TYPReplica<TReplicaObjects...>::GetTargetState(const TUpdatesCoordinatorConfig& config, NInfra::TLogFramePtr logFrame) const {
        TReadGuard guard(UpdatesCoordinatorClientMutex_);
        Y_ENSURE(UpdatesCoordinatorClient_);
        return *DoWithRetry<ui64, yexception>(
            [this, logFrame] { return UpdatesCoordinatorClient_->GetTargetState(logFrame); },
            [this, logFrame] (const yexception&) { ReconstructBalancing(logFrame); },
            CreateRetryOptions(config.GetRetryConfig()),
            /* throwLast = */ true
        );
    }

    template <typename... TReplicaObjects>
    TString TYPReplica<TReplicaObjects...>::GetFilter() const {
        const auto replicaConfig = ReplicaConfig_.Get();
        const auto clusterConfig = ClusterConfig_.Get();

        if (!clusterConfig->GetFilter().empty()) {
            return clusterConfig->GetFilter();
        } else {
            return replicaConfig->GetFilter();
        }
    }

    template <typename... TReplicaObjects>
    TString TYPReplica<TReplicaObjects...>::GetWatchLog() const {
        const auto replicaConfig = ReplicaConfig_.Get();
        const auto clusterConfig = ClusterConfig_.Get();

        if (clusterConfig->HasWatchLog()) {
            return clusterConfig->GetWatchLog();
        }
        return replicaConfig->GetWatchLog();
    }

    template <typename... TReplicaObjects>
    ui64 TYPReplica<TReplicaObjects...>::GenerateYpTimestamp(
        const TGenerateTimestampConfig& config,
        NInfra::TLogFramePtr logFrame
    ) const {
        return *DoWithRetry<ui64, yexception>(
            [this] { return YPClient_->GenerateTimestamp().GetValue(YPClient_->Options().Timeout() * 2); },
            [this, logFrame] (const yexception&) { ReconstructBalancing(logFrame); },
            CreateRetryOptions(config.GetRetryConfig()),
            /* throwLast = */ true
        );
    }

    template <typename... TReplicaObjects>
    template <typename TReplicaObject>
    std::pair<TMap<TString, TStorageElement<TReplicaObject>>, ui64> TYPReplica<TReplicaObjects...>::SelectAllObjects(
        NInfra::TLogFramePtr logFrame,
        ui64 timestamp,
        const TUpdatesDisablerWithSemaphore& updatesDisabler
    ) {
        const auto replicaConfig = ReplicaConfig_.Get();
        const auto clusterConfig = ClusterConfig_.Get();

        const size_t selectObjectsChunkSize = replicaConfig->GetChunkSize();
        const size_t maxObjectsNumber = clusterConfig->GetMaxObjectsNumber();

        const TString configFilter = GetFilter();

        auto selectAllSubframe = NInfra::CreateBioChapterForFrame(
            logFrame,
            TReplicaSelectAllObjectsStart(YpAddress_, TString{TReplicaObject::NAME}, timestamp, selectObjectsChunkSize),
            TReplicaSelectAllObjectsEnd(TString{TReplicaObject::NAME}, 0),
            GetColumnFamilyHistogram<TReplicaObject>(NSensors::SELECT_ALL_OBJECTS_RESPONSE_TIME)
        );

        RateSensor<TReplicaObject>(NSensors::MASTER_SELECT_ALL_REQUEST).Inc();

        ui64 selectedObjectsNumber = 0;

        TString filter;
        TMap<TString, TStorageElement<TReplicaObject>> results;

        NYP::NClient::TSelectObjectsOptions options;
        options.SetLimit(selectObjectsChunkSize);
        ui64 totalByteSizeOfResult = 0;

        while (true) {
            updatesDisabler.ThrowIfUpdatesDisabled();

            auto selectAllChunkSubframe = NInfra::CreateBioChapterForFrame(
                logFrame,
                TReplicaChunkSelectionStart(TReplicaObject::TObject::TypeName()),
                TReplicaChunkSelectionEnd(TReplicaObject::TObject::TypeName(), 0),
                GetColumnFamilyHistogram<TReplicaObject>(NSensors::CHUNK_SELECTION_RESPONSE_TIME)
            );

            if (!configFilter.empty()) {
                filter = !filter.empty() ? TString::Join("(", filter, ") and (", configFilter, ")") : configFilter;
            }

            TSimpleTimer chunkTimer;
            TMaybe<NYP::NClient::TSelectObjectsResult> chunk;
            try {
                chunk = DoWithRetry<NYP::NClient::TSelectObjectsResult, NYP::NClient::TResponseError>(
                    [&] {
                        RateSensor<TReplicaObject>(NSensors::MASTER_SELECT_ALL_CHUNK_REQUEST).Inc();
                        return YPClient_->template SelectObjects<typename TReplicaObject::TObject>(TReplicaObject::ATTRIBUTE_SELECTORS, filter, options, timestamp).GetValue(YPClient_->Options().Timeout() * 2);
                    },
                    [this, logFrame, &filter](const NYP::NClient::TResponseError& ex) {
                        RateSensor<TReplicaObject>(NSensors::MASTER_SELECT_ALL_CHUNK_ERROR).Inc();
                        INFRA_LOG_WARN(TReplicaChunkSelectionError(TReplicaObject::TObject::TypeName(), ex.Address(), filter, ex.what()));
                        ReconstructBalancing(logFrame);
                    },
                    CreateRetryOptions(clusterConfig->GetSelectObjectsRetryConfig()),
                    /* throwLast = */ true
                );
            } catch (NYP::NClient::TResponseError& ex) {
                RateSensor<TReplicaObject>(NSensors::MASTER_SELECT_ALL_CHUNK_FAILURE).Inc();
                INFRA_LOG_ERROR(TReplicaChunkSelectionFail(TReplicaObject::TObject::TypeName(), ex.Address(), filter, ex.what()));
                ythrow ex;
            }

            Y_ENSURE(chunk.Defined());

            RateSensor<TReplicaObject>(NSensors::MASTER_SELECT_ALL_CHUNK_SUCCESS).Inc();

            selectedObjectsNumber += chunk->Results.size();

            for (size_t i = 0; i < chunk->Results.size(); ++i) {
                auto& object = chunk->Results[i];
                TReplicaObject ro;
                ro.FromSelectorResult(object);
                if (i + 1 == chunk->Results.size()) {
                    ro.FillBorderFilter(filter);
                }
                Y_ENSURE(!results.contains(ro.GetObjectId()));
                auto& data = results[ro.GetObjectId()];
                data = std::move(ro);
            }

            ui64 byteSizeOfChunk = GetByteSizeOfReplicaObjects(chunk->Results);
            totalByteSizeOfResult += byteSizeOfChunk;
            INFRA_LOG_INFO(TReplicaChunkSelectionSuccess(TReplicaObject::TObject::TypeName(), chunk->Results.size(), filter, byteSizeOfChunk));
            AddToHistogram<TReplicaObject>(NSensors::SELECT_ALL_OBJECTS_CHUNK_SIZE, byteSizeOfChunk);

            if (maxObjectsNumber > 0 && selectedObjectsNumber > maxObjectsNumber) {
                IntGaugeSensor<TReplicaObject>(NSensors::MAX_SELECTED_OBJECTS_NUMBER_REACHED).Set(1);
                ythrow yexception() << "Selected objects number exceeded MaxObjectsNumber: " << selectedObjectsNumber << " > " << maxObjectsNumber;
            }

            if (chunk->Results.size() < options.Limit()) {
                break;
            }
        }

        INFRA_LOG_INFO(TReplicaSelectAllObjectsSuccess(TString{TReplicaObject::NAME}, results.size(), totalByteSizeOfResult));
        AddToHistogram<TReplicaObject>(NSensors::SELECT_ALL_OBJECTS_RESULT_SIZE, totalByteSizeOfResult);
        IntGaugeSensor<TReplicaObject>(NSensors::MAX_SELECTED_OBJECTS_NUMBER_REACHED).Set(0);
        IntGaugeSensor<TReplicaObject>(NSensors::SELECTED_OBJECTS_NUMBER).Set(selectedObjectsNumber);

        return {std::move(results), selectedObjectsNumber};
    }

    template <typename... TReplicaObjects>
    template <typename TReplicaObject>
    TUpdates<TReplicaObject> TYPReplica<TReplicaObjects...>::SelectAllAndGetUpdates(
        const TStringBuf columnFamilyName,
        NInfra::TLogFramePtr logFrame,
        const ui64 newTimestamp,
        TUpdatesDisablerWithSemaphore& updatesDisabler
    ) {
        auto subframe = NInfra::CreateBioChapterForFrame(
            logFrame,
            TReplicaSelectUpdatesStart(YpAddress_, TString{TReplicaObject::NAME}, EGetUpdatesMode_Name(SELECT_ALL), newTimestamp),
            TReplicaSelectUpdatesEnd(TString{TReplicaObject::NAME}, 0),
            GetColumnFamilyHistogram<TReplicaObject>(NSensors::SELECT_UPDATES_RESPONSE_TIME)
        );

        const auto clusterConfig = ClusterConfig_.Get();
        const TSemaphoreConfig semaphoreConfig = clusterConfig->GetSemaphoresConfig().GetSemaphoreForGetUpdates();

        if (semaphoreConfig.GetEnable() && semaphoreConfig.HasID()) {
            updatesDisabler.AcquireSemaphoreIfEmpty(*YPClient_, semaphoreConfig, SensorGroup_, logFrame);
        }

        updatesDisabler.ThrowIfUpdatesDisabled();

        auto [selectedData, selectedObjectsNumber] = SelectAllObjects<TReplicaObject>(logFrame, newTimestamp, updatesDisabler);

        TVector<TUpdate<TReplicaObject>> updatesData;
        auto [storage, snapshot] = InternalSnapshot();
        TReadOptions readOptions;
        readOptions.Snapshot = snapshot;
        {
            ui64 updatesObjectsNumber = 0;
            auto buildUpdatesSubframe = NInfra::CreateBioChapterForFrame(
                logFrame,
                TReplicaBuildUpdatesStart(TString{TReplicaObject::NAME}),
                TReplicaBuildUpdatesEnd(TString{TReplicaObject::NAME}, 0)
            );

            {
                TIterator it = storage->NewIterator(readOptions, columnFamilyName);
                it.SeekToFirst();
                for (; it.Valid(); it.Next()) {
                    updatesDisabler.ThrowIfUpdatesDisabled();

                    const TString key{it.Key()};
                    TVector<TStorageElement<TReplicaObject>> oldValue = it.Value<TReplicaObject>();
                    Y_ENSURE(oldValue.size() == 1);

                    if (auto selectedDataIt = selectedData.find(key); selectedDataIt != selectedData.end()) {
                        if (!oldValue.front().Equals(selectedDataIt->second)) {
                            ++updatesObjectsNumber;
                            updatesData.emplace_back(std::move(oldValue.front().ReplicaObject), std::move(selectedDataIt->second));
                        }
                        selectedData.erase(selectedDataIt);
                    } else  {
                        ++updatesObjectsNumber;
                        updatesData.emplace_back(std::move(oldValue.front().ReplicaObject), TStorageElement<TReplicaObject>(TReplicaObject(), EObjectType::DELETE));
                    }
                }
                it.Status().ThrowIfError();
            }

            updatesObjectsNumber += selectedData.size();
            for (auto& [key, newValue] : selectedData) {
                updatesDisabler.ThrowIfUpdatesDisabled();
                updatesData.emplace_back(Nothing(), std::move(newValue));
            }

            NInfra::TIntGaugeSensor(SensorGroup_, NSensors::UPDATES_OBJECTS_NUMBER).Set(updatesObjectsNumber);
        }

        if (logFrame->AcceptLevel(ELogPriority::TLOG_DEBUG)) {
            for (auto& [key, event] : MakeReplicaFoundUpdatedObjectEvents(updatesData)) {
                updatesDisabler.ThrowIfUpdatesDisabled();
                INFRA_LOG_DEBUG(event);
            }
        }

        return TUpdates<TReplicaObject>{std::move(updatesData), selectedObjectsNumber, newTimestamp, TInstant::Zero()};
    }

    template <typename... TReplicaObjects>
    template <typename TReplicaObject>
    NYP::NClient::TWatchObjectsResult TYPReplica<TReplicaObjects...>::WatchObjects(
        NInfra::TLogFramePtr logFrame,
        ui64 startTimestamp,
        ui64 timestamp,
        const TUpdatesDisablerWithSemaphore& updatesDisabler
    ) {
        const auto replicaConfig = ReplicaConfig_.Get();
        const auto clusterConfig = ClusterConfig_.Get();

        const size_t eventCountLimit = replicaConfig->GetWatchObjectsEventCountLimit();
        const TDuration timeLimit = TDuration::Parse(replicaConfig->GetWatchObjectsTimeLimit());

        const TString configFilter = GetFilter();

        auto watchObjectsSubframe = NInfra::CreateBioChapterForFrame(
            logFrame,
            TReplicaWatchObjectsStart(YpAddress_, TReplicaObject::TObject::TypeName(), startTimestamp, timestamp, eventCountLimit, timeLimit.MilliSeconds()),
            TReplicaWatchObjectsEnd(TReplicaObject::TObject::TypeName(), 0),
            GetColumnFamilyHistogram<TReplicaObject>(NSensors::WATCH_OBJECTS_RESPONSE_TIME)
        );
        RateSensor<TReplicaObject>(NSensors::MASTER_WATCH_REQUEST).Inc();

        const auto getUpdatesMode = (clusterConfig->HasGetUpdatesMode() ? clusterConfig->GetGetUpdatesMode() : replicaConfig->GetGetUpdatesMode());

        const TString configWatchLog = GetWatchLog();

        NYP::NClient::NApi::NProto::TAttributeSelector watchedPaths;
        if (getUpdatesMode == WATCH_SELECTORS) {
            watchedPaths.mutable_paths()->Reserve(TReplicaObject::WATCH_SELECTORS.size());
            for (const auto& path : TReplicaObject::WATCH_SELECTORS) {
                watchedPaths.add_paths(path);
            }
        }

        NYP::NClient::TWatchObjectsResult result;
        while (true) {
            updatesDisabler.ThrowIfUpdatesDisabled();

            NYP::NClient::TWatchObjectsOptions watchOptions;
            watchOptions.SetTimestamp(timestamp)
                        .SetStartTimestamp(result.ContinuationToken ? 0 : startTimestamp)
                        .SetContinuationToken(result.ContinuationToken)
                        .SetEventCountLimit(eventCountLimit)
                        .SetTimeLimit(timeLimit)
                        .SetFilter(configFilter);

            if (getUpdatesMode == WATCH_SELECTORS && !watchedPaths.paths().empty()) {
                watchOptions.SetSelector(watchedPaths);
            }

            if (!configWatchLog.empty()) {
                watchOptions.SetWatchLog(configWatchLog);
            }

            NYP::NClient::TWatchObjectsResult watchObjectsResult;
            try {
                watchObjectsResult = *DoWithRetry<NYP::NClient::TWatchObjectsResult, NYP::NClient::TResponseError>(
                    [&] {
                        RateSensor<TReplicaObject>(NSensors::MASTER_WATCH_CHUNK_REQUEST).Inc();
                        return YPClient_->template WatchObjects<typename TReplicaObject::TObject>(watchOptions).GetValue(YPClient_->Options().Timeout() * 2);
                    },
                    [this, &logFrame] (const NYP::NClient::TResponseError& ex) {
                        RateSensor<TReplicaObject>(NSensors::MASTER_WATCH_CHUNK_ERROR).Inc();

                        const int errorCode = static_cast<int>(ex.Error().GetNonTrivialCode());
                        INFRA_LOG_WARN(TReplicaWatchChunkError(
                            TReplicaObject::TObject::TypeName(), ex.Address(), errorCode, ex.what()));

                        switch (NYP::NClient::NApi::EErrorCode(errorCode)) {
                            case NYP::NClient::NApi::EErrorCode::RowsAlreadyTrimmed:
                            case NYP::NClient::NApi::EErrorCode::InvalidContinuationToken:
                            case NYP::NClient::NApi::EErrorCode::ContinuationTokenVersionMismatch:
                                throw ex;
                            default:
                                break;
                        }

                        ReconstructBalancing(logFrame);
                    },
                    CreateRetryOptions(clusterConfig->GetWatchObjectsRetryConfig()),
                    /* throwLast = */ true
                );
            } catch (NYP::NClient::TResponseError& ex) {
                RateSensor<TReplicaObject>(NSensors::MASTER_WATCH_CHUNK_FAILURE).Inc();
                INFRA_LOG_ERROR(TReplicaWatchChunkFail(TReplicaObject::TObject::TypeName(), ex.Address(), ex.what()));
                throw;
            }

            RateSensor<TReplicaObject>(NSensors::MASTER_WATCH_CHUNK_SUCCESS).Inc();
            INFRA_LOG_INFO(TReplicaWatchChunkSuccess(TReplicaObject::TObject::TypeName(), watchObjectsResult.Events.size()));

            std::move(watchObjectsResult.Events.begin(), watchObjectsResult.Events.end(), std::back_inserter(result.Events));

            result.Timestamp = watchObjectsResult.Timestamp;
            result.ContinuationToken = watchObjectsResult.ContinuationToken;

            if (!eventCountLimit || watchObjectsResult.Events.size() < eventCountLimit) {
                break;
            }
        }

        return result;
    }

    template <typename... TReplicaObjects>
    template <typename TReplicaObject>
    TVector<TMaybe<TReplicaObject>> TYPReplica<TReplicaObjects...>::SelectObjects(
        NInfra::TLogFramePtr logFrame,
        const TVector<TString>& objectIds,
        ui64 timestamp,
        const TUpdatesDisablerWithSemaphore& updatesDisabler
    ) {
        auto selectObjectsSubframe = NInfra::CreateBioChapterForFrame(
            logFrame,
            TReplicaSelectUpdatedObjectsStart(YpAddress_, TString{TReplicaObject::NAME}, timestamp, objectIds.size()),
            TReplicaSelectUpdatedObjectsEnd(TString{TReplicaObject::NAME}, 0),
            GetColumnFamilyHistogram<TReplicaObject>(NSensors::SELECT_UPDATED_OBJECTS_RESPONSE_TIME)
        );
        const auto replicaConfig = ReplicaConfig_.Get();
        const auto clusterConfig = ClusterConfig_.Get();

        const size_t selectObjectsChunkSize = replicaConfig->GetChunkSize();

        TVector<NYP::NClient::TSelectorResult> selectionResults;
        selectionResults.reserve(objectIds.size());

        NYP::NClient::TGetObjectOptions getObjectsOptions;
        getObjectsOptions.SetIgnoreNonexistent(true);

        for (size_t start = 0; start < objectIds.size(); start += selectObjectsChunkSize) {
            updatesDisabler.ThrowIfUpdatesDisabled();

            size_t end = Min(start + selectObjectsChunkSize, objectIds.size());

            TVector<TString> objectIdsChunk;
            objectIdsChunk.reserve(end - start);
            std::move(objectIds.begin() + start, objectIds.begin() + end, std::back_inserter(objectIdsChunk));

            TVector<NYP::NClient::TSelectorResult> chunk;
            try {
                chunk = *DoWithRetry<TVector<NYP::NClient::TSelectorResult>, NYP::NClient::TResponseError>(
                    [&] {
                        RateSensor<TReplicaObject>(NSensors::MASTER_GET_UPDATED_OBJECTS_CHUNK_REQUEST).Inc();
                        return YPClient_->template GetObjects<typename TReplicaObject::TObject>(objectIdsChunk, TReplicaObject::ATTRIBUTE_SELECTORS, timestamp, getObjectsOptions).GetValue(YPClient_->Options().Timeout() * 2);
                    },
                    [this, &logFrame, start, end] (const NYP::NClient::TResponseError& ex) {
                        RateSensor<TReplicaObject>(NSensors::MASTER_GET_UPDATED_OBJECTS_CHUNK_ERROR).Inc();

                        INFRA_LOG_WARN(TReplicaSelectObjectsChunkError(
                            TReplicaObject::TObject::TypeName(), ex.Address(), start, end, ex.what()));
                        ReconstructBalancing(logFrame);
                    },
                    CreateRetryOptions(clusterConfig->GetGetObjectsRetryConfig()),
                    /* throwLast */ true
                );
                Y_ENSURE(chunk.size() == end - start);
            } catch (const NYP::NClient::TResponseError& ex) {
                RateSensor<TReplicaObject>(NSensors::MASTER_GET_UPDATED_OBJECTS_CHUNK_FAILURE).Inc();
                INFRA_LOG_ERROR(TReplicaSelectObjectsChunkFail(
                    TReplicaObject::TObject::TypeName(), ex.Address(), start, end, ex.what()));
                throw;
            }

            INFRA_LOG_INFO(TReplicaSelectObjectsChunkSuccess(TReplicaObject::TObject::TypeName(), start, end));
            RateSensor<TReplicaObject>(NSensors::MASTER_GET_UPDATED_OBJECTS_CHUNK_SUCCESS).Inc();

            std::move(chunk.begin(), chunk.end(), std::back_inserter(selectionResults));
        }

        Y_ENSURE(selectionResults.size() == objectIds.size());

        TVector<TMaybe<TReplicaObject>> results;
        results.reserve(selectionResults.size());

        for (const NYP::NClient::TSelectorResult& selectionResult : selectionResults) {
            TMaybe<TReplicaObject> object;
            if (!selectionResult.Values().value_payloads().empty()) {
                object = TReplicaObject();
                object->FromSelectorResult(selectionResult);
            }
            results.emplace_back(std::move(object));
        }
        return results;
    }

    template <typename... TReplicaObjects>
    template <typename TReplicaObject>
    TUpdates<TReplicaObject> TYPReplica<TReplicaObjects...>::SelectOnlyUpdates(
        const TStringBuf columnFamilyName,
        NInfra::TLogFramePtr logFrame,
        const ui64 currentTimestamp,
        const ui64 newTimestamp,
        const TUpdatesDisablerWithSemaphore& updatesDisabler
    ) {
        const auto replicaConfig = ReplicaConfig_.Get();
        const auto clusterConfig = ClusterConfig_.Get();

        const size_t maxObjectsNumber = clusterConfig->GetMaxObjectsNumber();
        const size_t maxAllowedUpdatedObjectsNumberToGet = replicaConfig->GetMaxAllowedUpdatedObjectsNumberToGet();

        auto selectOnlyUpdatesSubframe = NInfra::CreateBioChapterForFrame(
            logFrame,
            TReplicaSelectUpdatesStart(YpAddress_, TString{TReplicaObject::NAME}, EGetUpdatesMode_Name(WATCH_UPDATES), newTimestamp),
            TReplicaSelectUpdatesEnd(TString{TReplicaObject::NAME}, 0),
            GetColumnFamilyHistogram<TReplicaObject>(NSensors::SELECT_UPDATES_RESPONSE_TIME)
        );

        NYP::NClient::TWatchObjectsResult watchObjectsResult = WatchObjects<TReplicaObject>(logFrame, currentTimestamp, newTimestamp, updatesDisabler);

        auto [storage, snapshot] = InternalSnapshot();
        TReadOptions readOptions;
        readOptions.Snapshot = snapshot;
        i64 newObjectsNumber = storage->template GetObjectsNumber<TReplicaObject>(readOptions, columnFamilyName);
        ui64 createEventsNumber = 0;
        ui64 updateEventsNumber = 0;
        ui64 removeEventsNumber = 0;

        TVector<TString> objectIds;
        objectIds.reserve(watchObjectsResult.Events.size());
        for (const NYP::NClient::NApi::NProto::TEvent& event : watchObjectsResult.Events) {
            objectIds.push_back(event.object_id());
            switch (event.event_type()) {
                case NYP::NClient::NApi::NProto::ET_OBJECT_CREATED:
                    ++newObjectsNumber;
                    ++createEventsNumber;
                    break;
                case NYP::NClient::NApi::NProto::ET_OBJECT_UPDATED:
                    ++updateEventsNumber;
                    break;
                case NYP::NClient::NApi::NProto::ET_OBJECT_REMOVED:
                    --newObjectsNumber;
                    ++removeEventsNumber;
                    break;
                default:
                    ythrow yexception() << "Unexpected event type: " << NYP::NClient::NApi::NProto::EEventType_Name(event.event_type());
            }
        }

        INFRA_LOG_DEBUG(MakeReplicaWatchObjectsResultEvent<TReplicaObject>(watchObjectsResult));

        Y_ENSURE(newObjectsNumber >= 0);

        SortUnique(objectIds);

        RateSensor<TReplicaObject>(NSensors::MASTER_EVENTS_NUMBER).Add(watchObjectsResult.Events.size());
        RateSensor<TReplicaObject>(NSensors::MASTER_CREATE_EVENTS_NUMBER).Add(createEventsNumber);
        RateSensor<TReplicaObject>(NSensors::MASTER_UPDATE_EVENTS_NUMBER).Add(updateEventsNumber);
        RateSensor<TReplicaObject>(NSensors::MASTER_REMOVE_EVENTS_NUMBER).Add(removeEventsNumber);

        if (maxObjectsNumber > 0 && newObjectsNumber > static_cast<i64>(maxObjectsNumber)) {
            IntGaugeSensor<TReplicaObject>(NSensors::MAX_SELECTED_OBJECTS_NUMBER_REACHED).Set(1);
            ythrow yexception() << "Selected objects number exceeded MaxObjectsNumber: " << newObjectsNumber << " > " << maxObjectsNumber;
        }

        if (maxAllowedUpdatedObjectsNumberToGet > 0 && objectIds.size() > maxAllowedUpdatedObjectsNumberToGet) {
            ythrow yexception() << "Number of updated objects exceeded MaxAllowedUpdatedObjectsNumberToGet: " << objectIds.size() << " > " << maxAllowedUpdatedObjectsNumberToGet;
        }

        updatesDisabler.ThrowIfUpdatesDisabled();

        TVector<TUpdate<TReplicaObject>> updatesData;
        TVector<TMaybe<TReplicaObject>> newValues;
        TVector<TMaybe<TReplicaObject>> oldValues;

        {
            TAdaptiveThreadPool queue;
            queue.Start();
            NThreading::TFuture<TVector<TMaybe<TReplicaObject>>> selectObjects = NThreading::Async([this, &logFrame, &objectIds, &newTimestamp, &updatesDisabler]() {
                return SelectObjects<TReplicaObject>(logFrame, objectIds, newTimestamp, updatesDisabler);
            }, queue);

            for (const auto& objectId : objectIds) {
                TVector<TStorageElement<TReplicaObject>> oldValue;
                if (auto getResult = storage->template GetReplicaObjects<TReplicaObject>(readOptions, columnFamilyName, objectId); getResult.IsError()) {
                    if (getResult.Error() != TStatus::NotFound()) {
                        getResult.Error().ThrowIfError();
                    }
                } else {
                    oldValue = std::move(getResult.Success());
                }

                if (oldValue.size() == 1) {
                    oldValues.emplace_back(std::move(oldValue.front().ReplicaObject));
                } else {
                    Y_ENSURE(oldValue.empty());
                    oldValues.emplace_back(Nothing());
                }
            }

            newValues = selectObjects.GetValueSync();
        }

        for (size_t i = 0; i < objectIds.size(); ++i) {
            updatesDisabler.ThrowIfUpdatesDisabled();

            TMaybe<TReplicaObject>& newValue = newValues[i];
            TMaybe<TReplicaObject>& oldValue = oldValues[i];
            if (newValue.Defined()) {
                updatesData.emplace_back(std::move(oldValue), TStorageElement<TReplicaObject>(std::move(*newValue), EObjectType::PUT));
            } else if (oldValue.Defined()) {
                updatesData.emplace_back(std::move(oldValue), TStorageElement<TReplicaObject>(TReplicaObject(), EObjectType::DELETE));
            }
        }

        if (logFrame->AcceptLevel(ELogPriority::TLOG_DEBUG)) {
            for (auto& [key, event] : MakeReplicaFoundUpdatedObjectEvents(updatesData)) {
                updatesDisabler.ThrowIfUpdatesDisabled();
                INFRA_LOG_DEBUG(event);
            }
        }

        IntGaugeSensor<TReplicaObject>(NSensors::MAX_SELECTED_OBJECTS_NUMBER_REACHED).Set(0);
        IntGaugeSensor<TReplicaObject>(NSensors::UPDATES_OBJECTS_NUMBER).Set(objectIds.size());

        return TUpdates<TReplicaObject>{std::move(updatesData), static_cast<ui64>(newObjectsNumber), newTimestamp, TInstant::Zero()};
    }

    template <typename... TReplicaObjects>
    template <typename TReplicaObject>
    TUpdates<TReplicaObject> TYPReplica<TReplicaObjects...>::GetUpdates(
        const TStringBuf columnFamilyName,
        NInfra::TLogFramePtr logFrame,
        const ui64 currentTimestamp,
        const TMaybe<ui64>& newYpTimestamp,
        TUpdatesDisablerWithSemaphore& updatesDisabler
    ) {
        const ui64 newTimestamp = newYpTimestamp.GetOrElse(GetTargetState(logFrame));

        auto selectAllAndGetUpdates = [this, columnFamilyName, &logFrame, currentTimestamp, &newTimestamp, &updatesDisabler] {
            const auto clusterConfig = ClusterConfig_.Get();

            if (SelectAllRequestsInARow_ > 0) {
                const TDuration minimalTimeInterval = TDuration::Parse(clusterConfig->GetSelectAllMinimalInterval());
                Sleep(minimalTimeInterval - UpdatingFrequency_);
            }

            TInstant startTime = Now();
            if (currentTimestamp >= newTimestamp) {
                return TUpdates<TReplicaObject>{{}, 0, newTimestamp, startTime};
            }

            ++SelectAllRequestsInARow_;
            WatchRequestsInARow_ = 0;

            try {
                auto result = SelectAllAndGetUpdates<TReplicaObject>(columnFamilyName, logFrame, newTimestamp, updatesDisabler);
                RateSensor<TReplicaObject>(NSensors::MASTER_SELECT_ALL_SUCCESS).Inc();
                result.Timestamp = startTime;
                return result;
            } catch (const NYP::NClient::TResponseError& error) {
                TryUpdateTimestampUpdateStatus(newTimestamp, error, logFrame);
                RateSensor<TReplicaObject>(NSensors::MASTER_SELECT_ALL_FAILURE).Inc();
                throw;
            } catch (...) {
                RateSensor<TReplicaObject>(NSensors::MASTER_SELECT_ALL_FAILURE).Inc();
                throw;
            }
        };

        auto selectOnlyUpdates = [this, columnFamilyName, &logFrame, currentTimestamp, &newTimestamp, &updatesDisabler] {
            const auto clusterConfig = ClusterConfig_.Get();

            if (WatchRequestsInARow_ > 0) {
                const TDuration minimalTimeInterval = TDuration::Parse(clusterConfig->GetWatchMinimalInterval());
                Sleep(minimalTimeInterval - UpdatingFrequency_);
            }

            TInstant startTime = Now();
            if (currentTimestamp >= newTimestamp) {
                return TUpdates<TReplicaObject>{{}, 0, newTimestamp, startTime};
            }

            ++WatchRequestsInARow_;
            SelectAllRequestsInARow_ = 0;

            try {
                auto result = SelectOnlyUpdates<TReplicaObject>(columnFamilyName, logFrame, currentTimestamp, newTimestamp, updatesDisabler);
                RateSensor<TReplicaObject>(NSensors::MASTER_WATCH_SUCCESS).Inc();
                result.Timestamp = startTime;
                return result;
            } catch (...) {
                RateSensor<TReplicaObject>(NSensors::MASTER_WATCH_FAILURE).Inc();
                throw;
            }
        };

        const auto replicaConfig = ReplicaConfig_.Get();
        const auto clusterConfig = ClusterConfig_.Get();

        const auto getUpdatesMode = (clusterConfig->HasGetUpdatesMode() ? clusterConfig->GetGetUpdatesMode() : replicaConfig->GetGetUpdatesMode());

        switch (getUpdatesMode) {
            case SELECT_ALL:
                INFRA_LOG_INFO(TReplicaChooseUpdateMode(TReplicaChooseUpdateMode::SELECT_ALL, "By GetUpdatesMode set in config"));
                return selectAllAndGetUpdates();
            case WATCH_UPDATES:
            case WATCH_SELECTORS: {
                if (currentTimestamp == 0) {
                    INFRA_LOG_INFO(TReplicaChooseUpdateMode(TReplicaChooseUpdateMode::SELECT_ALL, "Current timestamp is zero"));
                    return selectAllAndGetUpdates();
                }
                try {
                    INFRA_LOG_INFO(TReplicaChooseUpdateMode(TReplicaChooseUpdateMode::SELECT_ONLY_UPDATES, "By GetUpdatesMode set in config"));
                    return selectOnlyUpdates();
                } catch (...) {
                    INFRA_LOG_ERROR(TReplicaSelectUpdatesError(TString{TReplicaObject::NAME}, CurrentExceptionMessage()));
                    return selectAllAndGetUpdates();
                }
            }
            default:
                ythrow yexception() << "Unknown mode to get updates: \"" << EGetUpdatesMode_Name(getUpdatesMode);
        }
    }

    template <typename... TReplicaObjects>
    template <typename TReplicaObject, typename... TRestReplicaObjects>
    std::tuple<TUpdates<TReplicaObject>, TUpdates<TRestReplicaObjects>...> TYPReplica<TReplicaObjects...>::GetUpdates(
        NInfra::TLogFramePtr logFrame,
        const ui64 currentTimestamp,
        TUpdatesDisablerWithSemaphore& updatesDisabler
    ) {
        TUpdates<TReplicaObject> firstUpdates = GetUpdates<TReplicaObject>(TableManagers_.template GetMainColumnFamilyName<TReplicaObject>(), logFrame, currentTimestamp, Nothing(), updatesDisabler);
        std::tuple<TUpdates<TRestReplicaObjects>...> restUpdates(GetUpdates<TRestReplicaObjects>(
                TableManagers_.template GetMainColumnFamilyName<TRestReplicaObjects>(),
                logFrame,
                currentTimestamp,
                firstUpdates.YpTimestamp,
                updatesDisabler
            )...
        );
        return std::tuple_cat(std::tuple(std::move(firstUpdates)), restUpdates);
    }

    template <typename... TReplicaObjects>
    bool TYPReplica<TReplicaObjects...>::RequestUpdates() {
        auto [logFrame, bio] = NInfra::CreateBiographedLoggerFrame(
            Logger_,
            TReplicaRequestUpdatesStart(YpAddress_),
            TReplicaRequestUpdatesEnd(),
            GetHistogram(NSensors::REQUEST_UPDATES_RESPONSE_TIME)
        );

        TGuard<TMutex> guard(LockUpdates_);
        TCurrentStateUpdater updateCurrentState(this, logFrame);

        if (UpdatesDisabled()) {
            return false;
        }

        const auto clusterConfig = ClusterConfig_.Get();

        TStatus applyStatus;

        try {
            NInfra::TRateSensor(SensorGroup_, NSensors::MASTER_REQUESTS).Inc();

            ReconstructBalancing(logFrame);

            const ui64 oldYpTimestamp = Storage()->GetYpTimestamp(TReadOptions());

            std::tuple<TUpdates<TReplicaObjects>...> updates;

            {
                TUpdatesDisablerWithSemaphore updatesDisabler(UpdatesDisabler_);
                updates = GetUpdates<TReplicaObjects...>(logFrame, oldYpTimestamp, updatesDisabler);
            }

            const ui64 ypTimestamp = std::get<0>(updates).YpTimestamp;

            if (oldYpTimestamp >= ypTimestamp) {
                INFRA_LOG_INFO(TReplicaRequestUpdatesNothing(YpAddress_));
                return true;
            }

            auto isEmptyUpdate = [](const auto&... updates) {
                return (updates.Data.empty() && ...);
            };
            if (std::apply(isEmptyUpdate, updates)) {
                INFRA_LOG_INFO(TReplicaRequestUpdatesNothing(YpAddress_));
            }

            if (UpdatesDisabled()) {
                return false;
            }

            {
                auto [logFrame, bio] = NInfra::CreateBiographedLoggerFrame(
                    Logger_,
                    TReplicaSyncStorageStart(YpAddress_),
                    TReplicaSyncStorageEnd(YpAddress_, 0),
                    GetHistogram(NSensors::COPY_UPDATES_TO_STORAGE_RESPONSE_TIME)
                );

                THashMap<TString, ui64> objectsNumber;

                auto deleteColumnFamilies = [&objectsNumber](TStorage& storage, NInfra::TLogFramePtr logFrame) {
                    TVector<TString> columnFamilies;
                    for (const auto& [cfName, objectNumber] : objectsNumber) {
                        if (objectNumber == 0) {
                            columnFamilies.push_back(cfName);
                        }
                    }
                    columnFamilies = std::move(OUTCOME_TRYX(storage.DeleteColumnFamilies(columnFamilies, true)));
                    if (!columnFamilies.empty()) {
                        INFRA_LOG_INFO(MakeReplicaUpdateColumnFamiliesEvent(columnFamilies, TReplicaUpdateColumnFamilies::DELETE));
                    }
                    return TStatus::Ok();
                };

                try {
                    TWriteBatch batch = Storage()->CreateWriteBatch();

                    applyStatus = TableManagers_.template ApplyUpdates<TReplicaObjects...>(*Storage(), batch, objectsNumber, updates, UpdatesDisabler_, logFrame, SensorGroup_);
                    if (!applyStatus) {
                        INFRA_LOG_ERROR(TReplicaStorageApplyUpdatesError(MakeErrorDesc(applyStatus)));
                        applyStatus.ThrowIfError();
                    }

                    TStatus writeStatus = Storage()->Write(TWriteOptions{}, batch);

                    if (!writeStatus) {
                        INFRA_LOG_ERROR(TReplicaStorageWriteError(MakeErrorDesc(writeStatus)));
                        writeStatus.ThrowIfError();
                    }

                    TStatus deleteStatus = deleteColumnFamilies(*Storage(), logFrame);

                    if (!deleteStatus) {
                        INFRA_LOG_ERROR(TReplicaStorageWriteError(MakeErrorDesc(writeStatus)));
                        deleteStatus.ThrowIfError();
                    }
                    UpdateActiveSnapshot();
                } catch (...) {
                    INFRA_LOG_ERROR(TReplicaSyncStorageError(YpAddress_, CurrentExceptionMessage()));
                    throw;
                }

                TryUpdateTimestampUpdateStatus(ypTimestamp, NYPUpdatesCoordinator::TUpdateStatus::OK, logFrame);

                UpdateSensorsAfterUpdates(objectsNumber);
            }

            if (UpdateCallback_) {
                try {
                    std::apply([this](const auto&... updates) {
                        (UpdateCallback_->Do(updates.YpTimestamp, updates.Data), ...);
                    }, updates);
                } catch (...) {
                    NInfra::TRateSensor(SensorGroup_, NSensors::UPDATE_CALLBACK_ERROR).Inc();
                    INFRA_LOG_ERROR(TReplicaUpdateCallbackError(CurrentExceptionMessage()));
                }
            }

            CounterOfFails_ = 0;
            NInfra::TIntGaugeSensor(SensorGroup_, NSensors::FAILS_IN_A_ROW).Set(CounterOfFails_);
        } catch (...) {
            if (!applyStatus && applyStatus.Error().Code() == TError::ECode::UpdatesDisabled) {
                CounterOfFails_ = 0;
                NInfra::TIntGaugeSensor(SensorGroup_, NSensors::FAILS_IN_A_ROW).Set(CounterOfFails_);
            } else {
                NInfra::TRateSensor(SensorGroup_, NSensors::MASTER_FAILURES).Inc();
                CounterOfFails_++;
                NInfra::TIntGaugeSensor(SensorGroup_, NSensors::FAILS_IN_A_ROW).Set(CounterOfFails_);

                INFRA_LOG_ERROR(TReplicaRequestUpdatesError(YpAddress_, CurrentExceptionMessage()));

                return false;
            }
        }
        return true;
    }

    template <typename... TReplicaObjects>
    void TYPReplica<TReplicaObjects...>::TryUpdateUntilSuccess(const TDuration& maxSleep) {
        auto tryUpdate = [this] {
            if (Stopped_.load()) {
                ythrow TError(TError::ReplicaStopped);
            }
            return RequestUpdates();
        };

        const TDuration startSleep = Min(TDuration::Seconds(1), maxSleep);
        constexpr TDuration increment = TDuration::Seconds(5);
        constexpr TDuration randomDelta = TDuration::Seconds(5);
        const ui32 maxRetries = (maxSleep - startSleep) / increment;
        TRetryOptions retryOptions = TRetryOptions().WithCount(maxRetries)
                                                    .WithSleep(startSleep)
                                                    .WithRandomDelta(randomDelta)
                                                    .WithIncrement(increment);
        while (!DoWithRetryOnRetCode(tryUpdate, retryOptions)) {
            retryOptions = TRetryOptions().WithCount(Max<ui32>()).WithSleep(maxSleep).WithRandomDelta(randomDelta);
        }
    }

    template <typename... TReplicaObjects>
    bool TYPReplica<TReplicaObjects...>::UpdatesDisabled() const {
        return UpdatesDisabler_->UpdatesDisabled();
    }

    template <typename... TReplicaObjects>
    void TYPReplica<TReplicaObjects...>::ThrowIfUpdatesDisabled() const {
        UpdatesDisabler_->ThrowIfUpdatesDisabled();
    }

    template <typename... TReplicaObjects>
    void TYPReplica<TReplicaObjects...>::UpdateCoordinatorClient(const TUpdatesCoordinatorConfig& config) {
        UpdateCoordinatorClient(MakeHolder<NYPUpdatesCoordinator::TClient>(MakeClientOptions(config.GetClientConfig()), Logger_.SpawnFrame()));
    }

    template <typename... TReplicaObjects>
    void TYPReplica<TReplicaObjects...>::UpdateCoordinatorClient(THolder<NYPUpdatesCoordinator::TClient>&& client) {
        TWriteGuard guard(UpdatesCoordinatorClientMutex_);
        UpdatesCoordinatorClient_ = std::move(client);
    }

    template <typename... TReplicaObjects>
    TAtomicSharedPtr<TStorage> TYPReplica<TReplicaObjects...>::Storage() const {
        TReadGuard guard(StorageMutex_);
        return Storage_;
    }

    template <typename... TReplicaObjects>
    typename TYPReplica<TReplicaObjects...>::TReplicaSnapshot TYPReplica<TReplicaObjects...>::GetReplicaSnapshot() const {
        return TReplicaSnapshot{InternalSnapshot()};
    }

    template <typename... TReplicaObjects>
    typename TYPReplica<TReplicaObjects...>::TInternalSnapshot TYPReplica<TReplicaObjects...>::InternalSnapshot() const {
        TReadGuard guard(StorageMutex_);
        return TInternalSnapshot{Storage_, ActiveSnapshot_};
    }

    template <typename... TReplicaObjects>
    void TYPReplica<TReplicaObjects...>::UpdateActiveSnapshot() {
        TWriteGuard guard(StorageMutex_);
        ActiveSnapshot_ = Storage_->CreateSnapshot();
    }

    template <typename... TReplicaObjects>
    TAtomicSharedPtr<TStorage> TYPReplica<TReplicaObjects...>::SwitchStorage(TAtomicSharedPtr<TStorage> storage) {
        TWriteGuard guard(StorageMutex_);
        ActiveSnapshot_.Reset();
        Storage_.Swap(storage);
        if (Storage_->IsOpened()) {
            ActiveSnapshot_ = Storage_->CreateSnapshot();
        }
        return storage;
    }

    template <typename... TReplicaObjects>
    void TYPReplica<TReplicaObjects...>::ReconstructBalancing(NInfra::TLogFramePtr logFrame) const {
        INFRA_LOG_INFO(TReplicaReconstructBalancing(YPClient_->Options().EnableBalancing()));
        YPClient_->ReconstructBalancing([logFrame](const NClient::NApi::NProto::TRspGetMasters& rsp) {
            TReplicaYpClientGetMastersResult event;
            for (const NClient::NApi::NProto::TRspGetMasters::TMasterInfo& info : rsp.master_infos()) {
                TMasterInfo* eventMasterInfo = event.AddMasterInfos();
                eventMasterInfo->SetFqdn(info.fqdn());
                eventMasterInfo->SetGrpcAddress(info.grpc_address());
                eventMasterInfo->SetGrpcIp6Address(info.grpc_ip6_address());
                eventMasterInfo->SetInstanceTag(info.instance_tag());
                eventMasterInfo->SetAlive(info.alive());
                eventMasterInfo->SetLeading(info.leading());
            }
            INFRA_LOG_INFO(std::move(event));
        });
    }

    template <typename... TReplicaObjects>
    void TYPReplica<TReplicaObjects...>::UpdateSensors() {
        const auto replicaConfig = ReplicaConfig_.Get();
        const auto clusterConfig = ClusterConfig_.Get();
        if (TTryGuard<TMutex> guard(StartStopMutex_); guard.WasAcquired()) {
            if (auto [storage, snapshot] = InternalSnapshot(); storage) {
                TReadOptions readOptions;
                readOptions.Snapshot = snapshot;

                const TDuration age = storage->Age(readOptions);
                const TDuration ageThreshold = TDuration::Parse(replicaConfig->GetStorageConfig().GetAgeAlertThreshold());

                // TODO(dima-zakharov): Set right backup age values
                NInfra::TIntGaugeSensor(SensorGroup_, NSensors::BACKUP_AGE).Set(age.MilliSeconds());
                NInfra::TIntGaugeSensor(SensorGroup_, NSensors::STORAGE_AGE).Set(age.MilliSeconds());
                NInfra::TIntGaugeSensor(SensorGroup_, NSensors::BACKUP_AGE_INDICATOR).Set(Started_.load() && age > ageThreshold ? 1 : 0);
                NInfra::TIntGaugeSensor(SensorGroup_, NSensors::STORAGE_AGE_INDICATOR).Set(Started_.load() && age > ageThreshold ? 1 : 0);

                const ui64 backupSizeBytes = GetTotalDirectorySizeBytes(Storage()->Options().Paths.BackupPath);

                NInfra::TIntGaugeSensor(SensorGroup_, NSensors::BACKUP_SIZE).Set(backupSizeBytes);

                Storage()->UpdateSensors();
            }
        }
        NInfra::TIntGaugeSensor(SensorGroup_, NSensors::MAX_OBJECTS_NUMBER).Set(clusterConfig->GetMaxObjectsNumber());
    }

    template <typename... TReplicaObjects>
    void TYPReplica<TReplicaObjects...>::UpdateSensorsAfterUpdates(const THashMap<TString, ui64>& objectsNumber) const {
        for (const auto& [columnFamily, number] : objectsNumber) {
            NInfra::TIntGaugeSensor(
                SensorGroup_,
                NSensors::STORAGE_OBJECTS_NUMBER,
                {{NSensors::COLUMN_FAMILY, columnFamily}}
            ).Set(number);
        }
    }

    template <typename... TReplicaObjects>
    void TYPReplica<TReplicaObjects...>::SetUpdateCallback(THolder<IUpdateCallback>&& callback) {
        UpdateCallback_ = std::move(callback);
    }

    template <typename... TReplicaObjects>
    template <typename TReplicaObject>
    void TYPReplica<TReplicaObjects...>::SetModifyReplicaElementCallback(THolder<IModifyReplicaElementCallback<TReplicaObject>>&& callback) {
        TableManagers_.SetModifyReplicaElementCallback(std::move(callback));
    }
} // namespace NYP::NYPReplica
