#include "client.h"

#include "provider_client.h"
#include "provider_client_options.h"

#include <infra/libs/yp_updates_coordinator/api/api.pb.h>
#include <infra/libs/yp_updates_coordinator/instance_state/writer/writer.h>
#include <infra/libs/yp_updates_coordinator/logger/events/events_decl.ev.pb.h>
#include <infra/libs/yp_updates_coordinator/logger/make_events.h>

#include <util/generic/guid.h>

namespace NYPUpdatesCoordinator {

using namespace NYPUpdatesCoordinator::NDetail;

static TString GetInstanceNameWithFallback(const TClientOptions& options, NInfra::TLogFramePtr logFrame) {
    logFrame->LogEvent(MakeYPUpdatesCoordinatorGetInstanceNameEvent(options.InstanceName));

    constexpr static std::initializer_list<NEventlog::EGetInstanceNameSource> sources = {
        NEventlog::OPTIONS,
        NEventlog::FQDN_HOSTNAME,
        NEventlog::HOSTNAME,
        NEventlog::RANDOM
    };

    for (const NEventlog::EGetInstanceNameSource from : sources) {
        TMaybe<TString> result;
        TString errorMessage;
        switch (from) {
            case NEventlog::OPTIONS: {
                if (!options.InstanceName.Defined()) {
                    errorMessage = "Instance name is not set in options";
                } else if (options.InstanceName->empty()) {
                    errorMessage = "Instance name is empty in options";
                } else {
                    result = *options.InstanceName;
                }
                break;
            }
            case NEventlog::FQDN_HOSTNAME: {
                try {
                    result = FQDNHostName();
                } catch (...) {
                    errorMessage = CurrentExceptionMessage();
                }
                break;
            }
            case NEventlog::HOSTNAME: {
                try {
                    result = HostName();
                } catch (...) {
                    errorMessage = CurrentExceptionMessage();
                }
                break;
            }
            case NEventlog::RANDOM: {
                result = TString::Join("instance-", CreateGuidAsString());
                break;
            }
            default: {
                errorMessage = TStringBuilder() << "Unknown source " << NEventlog::EGetInstanceNameSource_Name(from);
                break;
            }
        }
        if (!result.Defined()) {
            logFrame->LogEvent(from == NEventlog::OPTIONS ? TLOG_INFO : TLOG_ERR,
                NEventlog::TYPUpdatesCoordinatorGetInstanceNameError(
                    from,
                    std::move(errorMessage)));
        } else {
            logFrame->LogEvent(NEventlog::TYPUpdatesCoordinatorGetInstanceNameResult(
                from,
                *result));
            return std::move(*result);
        }
    }

    ythrow TWithBackTrace<yexception>() << "Failed to get instance name";
}

TClientOptions MakeClientOptions(const TClientConfig& config) {
    TClientOptions options;
    if (config.HasProviderAddress()) {
        options.ProviderAddress = config.GetProviderAddress();
    }
    if (config.HasProviderRequestTimeout()) {
        options.ProviderRequestTimeout = TDuration::Parse(config.GetProviderRequestTimeout());
    }

    if (config.HasYtProxy()) {
        options.YtProxy = config.GetYtProxy();
    }
    if (config.HasCypressRootPath()) {
        options.CypressRootPath = config.GetCypressRootPath();
    }
    if (config.HasService()) {
        options.Service = config.GetService();
    }
    if (config.HasPrimaryMedium()) {
        options.PrimaryMedium = config.GetPrimaryMedium();
    }

    if (config.HasInstanceName()) {
        options.InstanceName = config.GetInstanceName();
    }
    if (config.HasInstanceLocation()) {
        options.InstanceLocation = config.GetInstanceLocation();
    }
    if (config.HasParticipateInCoordination()) {
        options.ParticipateInCoordination = config.GetParticipateInCoordination();
    }
    if (config.HasLockTimeout()) {
        options.LockTimeout = TDuration::Parse(config.GetLockTimeout());
    }

    return options;
}

class TClient::TImpl {
public:
    TImpl(TClientOptions options, NInfra::TLogFramePtr logFrame)
        : Options_(std::move(options))
        , InstanceName_(GetInstanceNameWithFallback(Options_, logFrame))
        , MainLogFrame_(logFrame)
        , ProviderClient_(MakeHolder<TProviderClient>(
            InstanceName_,
            TProviderClientOptions{
                Options_.ProviderAddress,
                Options_.ProviderRequestTimeout}))
    {
        TInstanceStateClientWriterOptions stateWriterOptions;
        stateWriterOptions.YtProxy = Options_.YtProxy;
        stateWriterOptions.CypressRootPath = Options_.CypressRootPath;
        stateWriterOptions.Service = Options_.Service;
        stateWriterOptions.PrimaryMedium = Options_.PrimaryMedium;
        stateWriterOptions.InstanceName = InstanceName_;
        stateWriterOptions.LockTimeout = Options_.LockTimeout;
        stateWriterOptions.Meta = NYT::TNode()
            ("location", Options_.InstanceLocation);
        StateWriter_ = MakeHolder<TInstanceStateClientWriter>(std::move(stateWriterOptions), MainLogFrame_);
    }

    void StartParticipateInCoordination(NInfra::TLogFramePtr logFrame) const {
        Y_ENSURE(StateWriter_);

        StateWriter_->StartParticipateInCoordination(logFrame);
    }

    ui64 GetTargetState(NInfra::TLogFramePtr logFrame) const {
        Y_ENSURE(logFrame);

        const NApi::TRspGetTargetState response = RequestTargetState(logFrame);

        const TTimestampClientInfo timestampInfo = TTimestampClientInfo(response.timestamp()).ReceiveTime(Now());
        TryUpdateTargetState(timestampInfo, logFrame);

        return response.timestamp();
    }

    void SetCurrentState(TTimestampClientInfo timestampInfo, NInfra::TLogFramePtr logFrame) const {
        Y_ENSURE(StateWriter_);

        if (!timestampInfo.UpdateTime().Defined()) {
            timestampInfo.UpdateTime(Now());
        }

        StateWriter_->UpdateCurrentState(std::move(timestampInfo), logFrame);
    }

    void SetTimestampUpdateStatus(const ui64 timestamp, TUpdateStatus status, NInfra::TLogFramePtr logFrame) const {
        Y_ENSURE(StateWriter_);

        StateWriter_->UpdateTimestampUpdateStatus(timestamp, std::move(status), logFrame);
    }

private:
    NApi::TRspGetTargetState RequestTargetState(NInfra::TLogFramePtr logFrame) const {
        logFrame->LogEvent(NEventlog::TYPUpdatesCoordinatorClientRequestTargetState());
        NApi::TRspGetTargetState response;
        try {
            response = ProviderClient_->GetTargetState(Options_.Service, logFrame);
        } catch (...) {
            logFrame->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TYPUpdatesCoordinatorClientRequestTargetStateFailure(CurrentExceptionMessage()));
            throw;
        }

        logFrame->LogEvent(MakeYPUpdatesCoordinatorClientRequestTargetStateResponseEvent(response));

        if (response.status() != NApi::TRspGetTargetState::OK) {
            ythrow yexception() << "Failed to get timestamp. Result status: " << NApi::TRspGetTargetState::EGetTargetStateStatus_Name(response.status()) << "\n"
                                << "error_message: \"" << response.error_message() << "\"";
        }

        return response;
    }

    bool TryUpdateTargetState(const TTimestampClientInfo& timestampInfo, NInfra::TLogFramePtr logFrame) const {
        logFrame->LogEvent(MakeYPUpdatesCoordinatorClientUpdateTargetStateEvent(timestampInfo));
        try {
            StateWriter_->UpdateTargetState(timestampInfo, logFrame);
            logFrame->LogEvent(NEventlog::TYPUpdatesCoordinatorClientUpdateTargetStateSuccess());
        } catch (...) {
            logFrame->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TYPUpdatesCoordinatorClientUpdateTargetStateFailure(CurrentExceptionMessage()));
            return false;
        }
        return true;
    }

private:
    const TClientOptions Options_;

    const TString InstanceName_;

    NInfra::TLogFramePtr MainLogFrame_;

    THolder<TProviderClient> ProviderClient_;

    THolder<TInstanceStateClientWriter> StateWriter_;
};

TClient::TClient(TClientOptions options, NInfra::TLogFramePtr logFrame)
    : Impl_(MakeHolder<TImpl>(std::move(options), logFrame))
{
}

TClient::~TClient() = default;

void TClient::StartParticipateInCoordination(NInfra::TLogFramePtr logFrame) const {
    Impl_->StartParticipateInCoordination(logFrame);
}

ui64 TClient::GetTargetState(NInfra::TLogFramePtr logFrame) const {
    return Impl_->GetTargetState(logFrame);
}

void TClient::SetCurrentState(TTimestampClientInfo timestampInfo, NInfra::TLogFramePtr logFrame) const {
    Impl_->SetCurrentState(std::move(timestampInfo), logFrame);
}

void TClient::SetTimestampUpdateStatus(const ui64 timestamp, TUpdateStatus status, NInfra::TLogFramePtr logFrame) const {
    Impl_->SetTimestampUpdateStatus(timestamp, std::move(status), logFrame);
}

} // namespace NYPUpdatesCoordinator
