#include "raw_provider_client.h"

#include <contrib/libs/grpc/include/grpc++/channel.h>
#include <contrib/libs/grpc/include/grpc++/create_channel.h>

#include <util/stream/output.h>

namespace NYPUpdatesCoordinator::NDetail {

class TRawProviderClient::TImpl {
public:
    TImpl(TRawProviderClientOptions options)
        : Options_(std::move(options))
        , Channel_(grpc::CreateChannel(Options_.Address, grpc::InsecureChannelCredentials()))
        , Stub_(NApi::TYPTimestampProviderService::NewStub(Channel_))
    {
    }

    void Ping(NInfra::TRequestPtr<NApi::TReqPing> request, NInfra::TReplyPtr<NApi::TRspPing> reply) const {
        GrpcRequest(request, reply, &NApi::TYPTimestampProviderService::Stub::Ping);
    }

    void GetTargetState(NInfra::TRequestPtr<NApi::TReqGetTargetState> request, NInfra::TReplyPtr<NApi::TRspGetTargetState> reply) const {
        GrpcRequest(request, reply, &NApi::TYPTimestampProviderService::Stub::GetTargetState);
    }

private:
    template <typename TRequest, typename TReply, typename TCallback>
    void GrpcRequest(NInfra::TRequestPtr<TRequest> request, NInfra::TReplyPtr<TReply> reply, TCallback&& callback) const {
        TReply result;

        grpc::ClientContext context;
        context.set_deadline(std::chrono::system_clock::now() + std::chrono::milliseconds(Options_.Timeout.MilliSeconds()));
        for (const auto& [key, value] : request->Attributes()) {
            context.AddMetadata(key, value);
        }

        if (grpc::Status status = ((*Stub_).*callback)(&context, request->Get(), &result); !status.ok()) {
            ythrow yexception() << "Code: " << static_cast<ui64>(status.error_code()) << "\nMessage: " << status.error_message();
        }

        for (const auto& [key, value] : context.GetServerInitialMetadata()) {
            reply->SetAttribute(TString{key.data(), key.size()}, TString{value.data(), value.size()});
        }
        for (const auto& [key, value] : context.GetServerTrailingMetadata()) {
            reply->SetAttribute(TString{key.data(), key.size()}, TString{value.data(), value.size()});
        }

        reply->Set(result);
    }

private:
    const TRawProviderClientOptions Options_;

    std::shared_ptr<grpc::Channel> Channel_;
    std::shared_ptr<NApi::TYPTimestampProviderService::Stub> Stub_;
};

TRawProviderClient::TRawProviderClient(TRawProviderClientOptions options)
    : Impl_(MakeHolder<TImpl>(std::move(options)))
{
}

TRawProviderClient::~TRawProviderClient() = default;

void TRawProviderClient::Ping(NInfra::TRequestPtr<NApi::TReqPing> request, NInfra::TReplyPtr<NApi::TRspPing> reply) {
    Impl_->Ping(request, reply);
}

void TRawProviderClient::GetTargetState(NInfra::TRequestPtr<NApi::TReqGetTargetState> request, NInfra::TReplyPtr<NApi::TRspGetTargetState> reply) {
    Impl_->GetTargetState(request, reply);
}

} // namespace NYPUpdatesCoordinator::NDetail
