#include "coordinator.h"

#include <infra/libs/yp_updates_coordinator/logger/events/events_decl.ev.pb.h>

#include <yp/cpp/yp/client.h>

#include <library/cpp/retry/retry.h>

namespace NYPUpdatesCoordinator {

class ICoordinator::TImpl {
public:
    void SetYpClient(const NYP::NClient::TClientPtr client) {
        YpClient_ = client;
    }

    ui64 GenerateTimestamp(NInfra::TLogFramePtr logFrame) const {
        if (!YpClient_) {
            ythrow yexception() << "YP client is not set";
        }

        try {
            return *DoWithRetry<ui64, NYP::NClient::TResponseError>(
                [this, logFrame] {
                    logFrame->LogEvent(NEventlog::TYpGenerateTimestampAttempt(YpClient_->Options().Address(), YpClient_->Options().EnableBalancing()));
                    const ui64 result = YpClient_->GenerateTimestamp().GetValue(YpClient_->Options().Timeout() * 2);
                    logFrame->LogEvent(NEventlog::TYpGenerateTimestampSuccess(result));
                    return result;
                },
                [logFrame] (const NYP::NClient::TResponseError& ex) {
                    logFrame->LogEvent(ELogPriority::TLOG_WARNING, NEventlog::TYpGenerateTimestampError(ex.what(), ex.Address()));
                },
                TRetryOptions()
                    .WithCount(3)
                    .WithSleep(TDuration::MilliSeconds(100))
                    .WithIncrement(TDuration::MilliSeconds(500)),
                /* throwLast */ true
            );
        } catch (const NYP::NClient::TResponseError& ex) {
            logFrame->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TYpGenerateTimestampFailure(ex.what(), ex.Address()));
            throw;
        }
    }

private:
    NYP::NClient::TClientPtr YpClient_;
};

ICoordinator::ICoordinator()
    : Impl_(MakeHolder<TImpl>())
{
}

ICoordinator::~ICoordinator() = default;

void ICoordinator::SetYpClient(const NYP::NClient::TClientPtr client) {
    Impl_->SetYpClient(client);
}

ui64 ICoordinator::GenerateTimestamp(NInfra::TLogFramePtr logFrame) const {
    return Impl_->GenerateTimestamp(logFrame);
}


TMaybe<ui64> TFakeCoordinator::GetTargetState(const NApi::TReqGetTargetState&, const TVector<TInstanceInfo>&, NInfra::TLogFramePtr, NInfra::TSensorGroup) const {
    return Nothing();
}

} // namespace NYPUpdatesCoordinator
