#include "location_coordinator.h"

#include <infra/libs/yp_updates_coordinator/api/api.pb.h>
#include <infra/libs/yp_updates_coordinator/logger/events/events_decl.ev.pb.h>
#include <infra/libs/yp_updates_coordinator/logger/make_events.h>

#include <yp/client/api/misc/public.h>
#include <yp/cpp/yp/error.h>

#include <mapreduce/yt/util/wait_for_tablets_state.h>
#include <mapreduce/yt/util/ypath_join.h>

#include <library/cpp/yson/node/node_io.h>

namespace NYPUpdatesCoordinator {

namespace {

TMaybe<TString> GetLocation(const TInstanceInfo& info) {
    if (!info.Meta.Defined()) {
        return Nothing();
    }

    const NYT::TNode& locationEntry = (*info.Meta)["location"];
    if (!locationEntry.IsString()) {
        return Nothing();
    }

    return locationEntry.AsString();
}

TMaybe<TString> GetLocation(const TInstanceInfo* info) {
    return info ? GetLocation(*info) : Nothing();
}

TMaybe<ui64> GetCurrentTimestamp(const TInstanceInfo& info) {
    if (!info.CurrentState.Defined()) {
        return Nothing();
    }

    if (!info.CurrentState->Timestamp.Defined()) {
        return Nothing();
    }

    return info.CurrentState->Timestamp->Value();
}

TMaybe<ui64> GetCurrentTimestamp(const TInstanceInfo* info) {
    return info ? GetCurrentTimestamp(*info) : Nothing();
}

} // namespace

TLocationCoordinatorOptions MakeLocationCoordinatorOptions(const TLocationCoordinatorConfig& config) {
    TLocationCoordinatorOptions options;
    options.Service = config.GetService();
    options.Locations = TVector<TString>{config.GetLocations().begin(), config.GetLocations().end()};
    if (config.HasDelay()) {
        options.Delay = TDuration::Parse(config.GetDelay());
    }
    const TTimestampReelectionConfig& reelectionConfig = config.GetTimestampReelectionConfig();
    if (!reelectionConfig.GetMinAllowedTimeToUpdateSinceReceiving().empty()) {
        options.TimestampReelectionOptions.MinAllowedTimeToUpdateSinceReceiving = TDuration::Parse(reelectionConfig.GetMinAllowedTimeToUpdateSinceReceiving());
    }
    options.TimestampReelectionOptions.MinAllowedErrorAttemptsToUpdate = reelectionConfig.GetMinAllowedErrorAttemptsToUpdate();

    options.YtProxy = config.GetYtConfig().GetProxy();
    options.CypressRootPath = config.GetYtConfig().GetCypressRootPath();
    if (config.GetYtConfig().HasPrimaryMedium()) {
        options.PrimaryMedium = config.GetYtConfig().GetPrimaryMedium();
    }
    return options;
}

TLocationCoordinatorConfig MakeLocationCoordinatorConfig(const TServedServiceConfig& serviceConfig) {
    TLocationCoordinatorConfig config = serviceConfig.GetCoordinatorConfig().GetLocationCoordinatorConfig();
    if (!config.HasService()) {
        config.SetService(serviceConfig.GetName());
    }
    if (!config.HasYtConfig()) {
        *config.MutableYtConfig() = serviceConfig.GetYtConfig();
    }
    return config;
}

TLocationCoordinatorOptions MakeLocationCoordinatorOptions(const TServedServiceConfig& serviceConfig) {
    return MakeLocationCoordinatorOptions(MakeLocationCoordinatorConfig(serviceConfig));
}

TLocationCoordinator::TLocationCoordinator(TLocationCoordinatorOptions options)
    : Options_(std::move(options))
    , YtClient_(NYT::CreateClient(Options_.YtProxy))
    , CoordinatorPath_(NYT::JoinYPaths(Options_.CypressRootPath, Options_.Service, "coordinator"))
{
    CreateTargetVersionNodes();
    CreateAndMountVersionsTables();
}

void TLocationCoordinator::CreateTargetVersionNodes() const {
    for (const TString& location : Options_.Locations) {
        NYT::TCreateOptions options = NYT::TCreateOptions().IgnoreExisting(true).Recursive(true);
        NYT::TNode attributes;
        if (Options_.PrimaryMedium.Defined()) {
            attributes("primary_medium", *Options_.PrimaryMedium);
        }

        if (attributes.IsMap()) {
            options.Attributes(attributes);
        }

        YtClient_->Create(NYT::JoinYPaths(CoordinatorPath_, "locations", location, "target"), NYT::ENodeType::NT_UINT64, options);
    }
}

void TLocationCoordinator::CreateAndMountVersionsTables() const {
    const NYT::TTableSchema schema = NYT::TTableSchema()
        .AddColumn("timestamp", NYT::EValueType::VT_UINT64, NYT::ESortOrder::SO_ASCENDING)
        .AddColumn(NYT::TColumnSchema().Name("update_time").Type(NYT::EValueType::VT_UINT64));

    NYT::TNode attributes = NYT::TNode()
        ("dynamic", true)
        ("schema", schema.ToNode())
        ("min_data_versions", 0)
        ("max_data_versions", 1)
        ("min_data_ttl", 6 * 60 * 60 * 1000)
        ("max_data_ttl", 12 * 60 * 60 * 1000);

    if (Options_.PrimaryMedium.Defined()) {
        attributes("primary_medium", *Options_.PrimaryMedium);
    }

    for (const TString& location : Options_.Locations) {
        const NYT::TYPath tablePath = NYT::JoinYPaths(CoordinatorPath_, "locations", location, "versions");
        YtClient_->Create(
            tablePath,
            NYT::ENodeType::NT_TABLE,
            NYT::TCreateOptions()
                .IgnoreExisting(true)
                .Recursive(true)
                .Attributes(attributes)
        );

        YtClient_->MountTable(tablePath);

        WaitForTabletsState(YtClient_, tablePath, NYT::ETabletState::TS_MOUNTED);
    }
}

TMaybe<ui64> TLocationCoordinator::GetPreviousTimestampByTime(const TString& location, const TDuration& ago, NInfra::TLogFramePtr logFrame) const {
    const TInstant timestamp = Now() - ago;
    const TString query = TStringBuilder()
        << "* from [" << NYT::JoinYPaths(CoordinatorPath_, "locations", location, "versions") << "] "
        << "where update_time <= " << timestamp.MilliSeconds() << "u "
        << "order by update_time desc limit 1";

    logFrame->LogEvent(NEventlog::TLocationCoordinatorGetPreviousTimestampByTime(query));
    const NYT::TNode::TListType result = YtClient_->SelectRows(query);
    logFrame->LogEvent(MakeLocationCoordinatorGetPreviousTimestampByTimeQueryResponseEvent(result));

    if (result.empty()) {
        return Nothing();
    }

    return result.front()["timestamp"].AsUint64();
}

TMaybe<ui64> TLocationCoordinator::GetNextLocationTargetTimestamp(const TVector<TString>::const_iterator locationIt, NInfra::TLogFramePtr logFrame) const {
    logFrame->LogEvent(MakeLocationCoordinatorGetNextLocationTargetTimestampEvent(locationIt == Options_.Locations.cend() ? nullptr : &*locationIt));

    NEventlog::TLocationCoordinatorGetNextLocationTargetTimestampResult::ESource source;
    TMaybe<ui64> targetTimestamp;
    if (locationIt == Options_.Locations.cbegin()) {
        source = NEventlog::TLocationCoordinatorGetNextLocationTargetTimestampResult::GENERATE_TIMESTAMP;
        targetTimestamp = GenerateTimestamp(logFrame);
    } else {
        // TODO synchronize previous location state
        source = NEventlog::TLocationCoordinatorGetNextLocationTargetTimestampResult::PREVIOUS_LOCATION_TIMESTAMP;
        targetTimestamp = GetPreviousTimestampByTime(*(locationIt - 1), Options_.Delay, logFrame);
    }

    logFrame->LogEvent(MakeLocationCoordinatorGetNextLocationTargetTimestampResultEvent(source, targetTimestamp));
    return targetTimestamp;

}

TMaybe<ui64> TLocationCoordinator::GetLocationTargetTimestamp(const TString& location, NInfra::TLogFramePtr logFrame) const {
    const NYT::TYPath targetTimestampPath = NYT::JoinYPaths(CoordinatorPath_, "locations", location, "target");
    logFrame->LogEvent(NEventlog::TLocationCoordinatorGetLocationTargetTimestamp(location, targetTimestampPath));

    const NYT::TNode targetTimestampNode = YtClient_->Get(targetTimestampPath);
    logFrame->LogEvent(NEventlog::TLocationCoordinatorGetLocationTargetTimestampResponse(NYT::NodeToYsonString(targetTimestampNode)));

    if (!targetTimestampNode.IsUint64() || !targetTimestampNode.AsUint64()) {
        return Nothing();
    }

    return targetTimestampNode.UncheckedAsUint64();
}

TMaybe<ui64> TLocationCoordinator::TryUpdateLocationTargetTimestamp(const TString& location, const ui64 expected, const ui64 target, NInfra::TLogFramePtr logFrame) const {
    logFrame->LogEvent(NEventlog::TLocationCoordinatorTryUpdateLocationTargetTimestamp(location, expected, target));

    const NYT::ITransactionPtr transaction = YtClient_->StartTransaction();
    const NYT::TYPath targetTimestampPath = NYT::JoinYPaths(CoordinatorPath_, "locations", location, "target");

    NYT::ILockPtr lock;
    try {
        lock = transaction->Lock(targetTimestampPath, NYT::ELockMode::LM_EXCLUSIVE);
        lock->GetAcquiredFuture().Wait();

        if (const ui64 actual = transaction->Get(targetTimestampPath).AsUint64(); actual == expected) {
            logFrame->LogEvent(NEventlog::TLocationCoordinatorTryUpdateLocationTargetTimestampSetNewTarget(targetTimestampPath, target));
            transaction->Set(targetTimestampPath, target);
            transaction->Commit();
            return target;
        } else {
            logFrame->LogEvent(NEventlog::TLocationCoordinatorTryUpdateLocationTargetTimestampConflict::FromFields(expected, actual));
            return actual;
        }
    } catch (...) {
        logFrame->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TLocationCoordinatorTryUpdateLocationTargetTimestampFailed(targetTimestampPath, CurrentExceptionMessage()));
        return Nothing();
    }
}

void TLocationCoordinator::UpdateLocationTimestampInfo(const TString& location, const ui64 timestamp, const TInstant& updateTime, NInfra::TLogFramePtr logFrame) const {
    const NYT::TYPath locationVersionsPath = NYT::JoinYPaths(CoordinatorPath_, "locations", location, "versions");
    logFrame->LogEvent(NEventlog::TLocationCoordinatorUpdateLocationTimestampInfo(location, locationVersionsPath, timestamp, updateTime.MilliSeconds()));

    YtClient_->InsertRows(
        locationVersionsPath,
        {
            NYT::TNode()
                ("timestamp", timestamp)
                ("update_time", updateTime.MilliSeconds())
        }
    );
}

bool TLocationCoordinator::IsTimestampInvalid(const ui64 timestamp, const TVector<TInstanceInfo>& instanceInfos) const {
    auto tryGetTimestampUpdateStatus = [&timestamp](const TMaybe<TInstanceState>& state) -> const TMaybe<TUpdateStatus>& {
        if (state.Defined() && state->Timestamp.Defined() && state->Timestamp->Value() == timestamp) {
            return state->Timestamp->UpdateStatus();
        }
        return Default<TMaybe<TUpdateStatus>>();
    };

    auto hasTimestampInvalidError = [&tryGetTimestampUpdateStatus](const TMaybe<TInstanceState>& state) {
        const TMaybe<TUpdateStatus>& status = tryGetTimestampUpdateStatus(state);
        if (!status.Defined() || status->Status() != TUpdateStatus::YP_ERROR || !status->YpRequest().Defined()) {
            return false;
        }
        const NYP::NClient::TResponseError& responseError = *status->YpRequest();
        switch (responseError.Error().GetNonTrivialCode()) {
            case static_cast<int>(NYP::NClient::NApi::EErrorCode::TimestampOutOfRange):
                return true;
        }
        return false;
    };

    auto allowToReelect = [this, &tryGetTimestampUpdateStatus](const TMaybe<TInstanceState>& state) {
        if (!Options_.TimestampReelectionOptions.MinAllowedTimeToUpdateSinceReceiving &&
            !Options_.TimestampReelectionOptions.MinAllowedErrorAttemptsToUpdate)
        {
            return false;
        }

        const TMaybe<TUpdateStatus>& status = tryGetTimestampUpdateStatus(state);
        if (!status.Defined() || status->Status() != TUpdateStatus::YP_ERROR || !state->Timestamp->ReceiveTime().Defined()) {
            // not enough info or has no error
            return false;
        }
        if (state->Timestamp->UpdateTime().Defined()) {
            // successfully updated to timestamp
            return false;
        }

        bool reelect = true;
        if (Options_.TimestampReelectionOptions.MinAllowedTimeToUpdateSinceReceiving) {
            reelect &= Now() - *state->Timestamp->ReceiveTime() > Options_.TimestampReelectionOptions.MinAllowedTimeToUpdateSinceReceiving;
        }
        if (Options_.TimestampReelectionOptions.MinAllowedErrorAttemptsToUpdate) {
            reelect &= state->Timestamp->UpdateAttempts().GetOrElse(0) > Options_.TimestampReelectionOptions.MinAllowedErrorAttemptsToUpdate;
        }
        return reelect;
    };

    return AnyOf(instanceInfos, [&hasTimestampInvalidError, &allowToReelect](const TInstanceInfo& info) {
        if (info.Role != EInstanceRole::LEADER) {
            return false;
        }

        for (const TMaybe<TInstanceState>& timestampState : {info.TargetState, info.SentState, info.CurrentState}) {
            try {
                if (hasTimestampInvalidError(timestampState) || allowToReelect(timestampState)) {
                    return true;
                }
            } catch (...) {
            }
        }
        return false;
    });
}

TMaybe<ui64> TLocationCoordinator::GetTargetState(const NApi::TReqGetTargetState& request, const TVector<TInstanceInfo>& instanceInfos, NInfra::TLogFramePtr logFrame, NInfra::TSensorGroup sensorGroup) const {
    Y_UNUSED(sensorGroup);

    logFrame->LogEvent(MakeLocationCoordinatorGetTargetStateEvent(request, instanceInfos));

    const TInstanceInfo* requestInstanceInfo = nullptr;
    for (const TInstanceInfo& instanceInfo : instanceInfos) {
        if (request.instance_name() == instanceInfo.Name) {
            requestInstanceInfo = &instanceInfo;
            break;
        }
    }

    logFrame->LogEvent(MakeLocationCoordinatorRequestInstanceInfoEvent(requestInstanceInfo));

    const TMaybe<TString> requestInstanceLocation = GetLocation(requestInstanceInfo);
    const TMaybe<ui64> requestInstanceCurrentTimestamp = GetCurrentTimestamp(requestInstanceInfo);

    const auto locationIt = requestInstanceLocation.Defined() ? Find(Options_.Locations, *requestInstanceLocation) : Options_.Locations.end();

    TMaybe<ui64> locationTargetTimestamp;
    if (locationIt != Options_.Locations.end()) {
        locationTargetTimestamp = GetLocationTargetTimestamp(*requestInstanceLocation, logFrame);
        logFrame->LogEvent(MakeLocationCoordinatorRequestInstanceLocationInfoEvent(*requestInstanceLocation, locationTargetTimestamp));
    } else {
        logFrame->LogEvent(MakeLocationCoordinatorRequestInstanceLocationIsInvalidEvent(requestInstanceLocation, Options_.Locations));
    }

    TMaybe<ui64> nextLocationTargetTimestamp = GetNextLocationTargetTimestamp(locationIt, logFrame);

    TMaybe<ui64> requestInstanceTargetTimestamp;
    if (locationTargetTimestamp.Defined()) {
        Y_ENSURE(requestInstanceLocation.Defined());

        const bool allInstancesAreAtTarget = AllOf(instanceInfos,
            [&requestInstanceLocation, &locationTargetTimestamp](const TInstanceInfo& info) {
                const bool isInstanceRelevant =
                    info.Role == EInstanceRole::LEADER &&
                    GetLocation(info) == requestInstanceLocation;

                return isInstanceRelevant
                    ? GetCurrentTimestamp(info) >= locationTargetTimestamp
                    : true; // do not account if irrelevant
            });

        // update timestamp update time for location
        if (allInstancesAreAtTarget) {
            TInstant maxUpdateTime;
            for (const TInstanceInfo& info : instanceInfos) {
                if (info.Role == EInstanceRole::LEADER && GetLocation(info) == requestInstanceLocation) {
                    maxUpdateTime = Max(maxUpdateTime, info.CurrentState->Timestamp->UpdateTime().GetOrElse(TInstant::Zero()));
                }
            }

            logFrame->LogEvent(NEventlog::TLocationCoordinatorAllInstancesInLocationAreAtTarget::FromFields(*locationTargetTimestamp, maxUpdateTime.MilliSeconds()));
            UpdateLocationTimestampInfo(*requestInstanceLocation, *locationTargetTimestamp, maxUpdateTime, logFrame);
        }

        const bool timestampIsInvalid = IsTimestampInvalid(*locationTargetTimestamp, instanceInfos);

        const bool useLocationTargetTimestamp =
            requestInstanceCurrentTimestamp < locationTargetTimestamp ||
            !allInstancesAreAtTarget;

        const bool doNotUseLocationTargetTimestamp =
            timestampIsInvalid;

        if (useLocationTargetTimestamp && !doNotUseLocationTargetTimestamp) {
            logFrame->LogEvent(NEventlog::TLocationCoordinatorSetTargetTimestampToLocationTargetTimestamp(*locationTargetTimestamp));
            requestInstanceTargetTimestamp = *locationTargetTimestamp;
        }
    }

    if (!requestInstanceTargetTimestamp.Defined()) {
        logFrame->LogEvent(MakeLocationCoordinatorSetTargetTimestampToLocationNextTargetTimestampEvent(nextLocationTargetTimestamp));
        requestInstanceTargetTimestamp = nextLocationTargetTimestamp;
    }

    if (locationIt != Options_.Locations.end() && requestInstanceTargetTimestamp.Defined() && requestInstanceTargetTimestamp != locationTargetTimestamp) {
        // try update target timestamp for location
        requestInstanceTargetTimestamp = TryUpdateLocationTargetTimestamp(*requestInstanceLocation, locationTargetTimestamp.GetOrElse(0), *requestInstanceTargetTimestamp, logFrame);
        logFrame->LogEvent(MakeLocationCoordinatorSetTargetTimestampToLocationNewTargetTimestampEvent(requestInstanceTargetTimestamp));
    }

    if (!requestInstanceTargetTimestamp.Defined()) {
        // unable to define target timestamp, so keep current state for now
        logFrame->LogEvent(MakeLocationCoordinatorSetTargetTimestampToInstanceCurrentTimestampEvent(requestInstanceCurrentTimestamp));
        requestInstanceTargetTimestamp = requestInstanceCurrentTimestamp;
    }

    logFrame->LogEvent(MakeLocationCoordinatorGettargetStateResultEvent(requestInstanceTargetTimestamp));

    return requestInstanceTargetTimestamp;
}

} // namespace NYPUpdatesCoordinator
