#include "server.h"

#include "service.h"

#include <util/string/cast.h>
#include <util/system/condvar.h>

namespace NYPUpdatesCoordinator {

class TGrpcServer::TImpl {
public:
    TImpl(const TGrpcServerConfig& config, IService& service)
        : GrpcService_(service)
        , ServerAddress_("0.0.0.0:" + ToString(config.GetPort()))
    {
        ServerBuilder_.AddListeningPort(ServerAddress_, grpc::InsecureServerCredentials());
        ServerBuilder_.RegisterService(&GrpcService_);
    }

    ~TImpl() {
        try {
            Stop();
        } catch (...) {
        }
    }

    void Start(NInfra::TLogFramePtr logFrame) {
        Server_.Reset(ServerBuilder_.BuildAndStart().release());
        Y_UNUSED(logFrame);
    }

    void Wait(NInfra::TLogFramePtr logFrame) {
        {
            TGuard<TMutex> guard(Mutex_);

            while (!Shutdown_) {
                CondVar_.Wait(Mutex_);
            }
        }

        Stop();
        Y_UNUSED(logFrame);
    }

    void Stop() {
        if (Server_) {
            Server_->Shutdown();
            Server_->Wait();
        }
    }

    void Shutdown() {
        TGuard<TMutex> guard(Mutex_);

        if (!Shutdown_) {
            Shutdown_ = true;

            CondVar_.Signal();
        }
    }

private:
    TGrpcService GrpcService_;
    const TString ServerAddress_;

    grpc::ServerBuilder ServerBuilder_;
    THolder<grpc::Server> Server_;

    TMutex Mutex_;
    TCondVar CondVar_;
    bool Shutdown_;
};

TGrpcServer::TGrpcServer(const TGrpcServerConfig& config, IService& service)
    : Impl_(MakeHolder<TImpl>(config, service))
{
}

TGrpcServer::~TGrpcServer() = default;

void TGrpcServer::Start(NInfra::TLogFramePtr logFrame) {
    Impl_->Start(logFrame);
}

void TGrpcServer::Wait(NInfra::TLogFramePtr logFrame) {
    Impl_->Wait(logFrame);
}

void TGrpcServer::Stop() {
    Impl_->Stop();
}

void TGrpcServer::Shutdown() {
    Impl_->Shutdown();
}

} // namespace NYPUpdatesCoordinator
