#include "service.h"

#include <infra/libs/service_iface/router.h>
#include <infra/libs/service_iface/str_iface.h>

namespace NYPUpdatesCoordinator {

class TGrpcService::TImpl {
public:
    TImpl(IService& service)
        : Service_(service)
    {
    }

    Status GetTargetState(ServerContext* ctx, const NApi::TReqGetTargetState* request, NApi::TRspGetTargetState* response) {
        return Call(ctx, request, response, [this](const auto& request, auto& reply) { Service_.GetTargetState(request, reply); });
    }

private:
    template <typename TRequest, typename TReply, typename TCallback>
    Status Call(ServerContext* ctx, const TRequest* request, TReply* reply, TCallback&& callback) {
        try {
            NInfra::TAttributes attributes;
            for (const auto& [key, value] : ctx->client_metadata()) {
                attributes.insert({TString(key.data(), key.size()), TString(value.data(), value.size())});
            }
            attributes.emplace(NInfra::NServiceAttributes::RemoteAddressAttributeName(), ctx->peer());

            NInfra::TAttributes responseAttributes;
            if (const auto requestIdIt = attributes.find("request-id"); requestIdIt != attributes.end()) {
                responseAttributes.emplace("request-id", requestIdIt->second);
            }

            for (const auto& [key, value] : responseAttributes) {
                ctx->AddInitialMetadata(key, value);
            }

            NInfra::TRequestPtr<TRequest> requestPtr = new NInfra::TProtoRequest<TRequest>(TStringBuf(), *request, attributes);

            NInfra::TReplyPtr<TReply> replyPtr = new NInfra::TProtoReply<TReply>(*reply, responseAttributes);
            callback(requestPtr, replyPtr);

            for (const auto& [key, value] : responseAttributes) {
                ctx->AddTrailingMetadata(key, value);
            }

            return Status::OK;
        } catch (const NInfra::TServiceError& ex) {
            return Status(ex.GetGrpcStatusCode(), std::string(CurrentExceptionMessage().data()));
        } catch (...) {
            return Status(grpc::StatusCode::INTERNAL, std::string(CurrentExceptionMessage().data()));
        }
    }

private:
    IService& Service_;
};

TGrpcService::TGrpcService(IService& service)
    : Impl_(MakeHolder<TImpl>(service))
{
}

TGrpcService::~TGrpcService() = default;

TGrpcService::Status TGrpcService::GetTargetState(ServerContext* ctx, const NApi::TReqGetTargetState* request, NApi::TRspGetTargetState* response) {
    return Impl_->GetTargetState(ctx, request, response);
}

} // namespace NYPUpdatesCoordinator
