#include "reader.h"

#include <infra/libs/yp_updates_coordinator/logger/events/events_decl.ev.pb.h>
#include <infra/libs/yp_updates_coordinator/logger/make_events.h>

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/util/ypath_join.h>

#include <library/cpp/yson/node/node_io.h>

namespace NYPUpdatesCoordinator {

class TInstanceStateReader::TImpl {
public:
    TImpl(TInstanceStateReaderOptions options)
        : Options_(std::move(options))
        , Client_(NYT::CreateClient(Options_.YtProxy))
        , ServicePath_(NYT::JoinYPaths(Options_.CypressRootPath, Options_.Service))
        , InstancesPath_(NYT::JoinYPaths(ServicePath_, "instances"))
        , StateTablePath_(NYT::JoinYPaths(ServicePath_, "meta", "state"))
        , VersionsTablePath_(NYT::JoinYPaths(ServicePath_, "meta", "versions"))
    {
    }

    TVector<TInstanceInfo> GetInstanceInfos(NInfra::TLogFramePtr logFrame) const {
        logFrame->LogEvent(NEventlog::TInstanceStateReaderGetInstanceInfos(ServicePath_));
        TVector<TInstanceInfo> instances = ListInstances(logFrame);
        FillStates(instances, logFrame);
        return instances;
    }

private:
    TVector<TInstanceInfo> ListInstances(NInfra::TLogFramePtr logFrame) const {
        const NYT::TListOptions listOptions = NYT::TListOptions()
            .AttributeFilter(
                NYT::TAttributeFilter()
                    .AddAttribute("locks")
                    .AddAttribute("meta")
            );

        logFrame->LogEvent(MakeInstanceStateReaderListInstancesEvent(InstancesPath_, listOptions));
        const NYT::TNode::TListType instanceNodes = Client_->List(InstancesPath_, listOptions);
        logFrame->LogEvent(MakeInstanceStateReaderListPathResultEvent(instanceNodes));

        TVector<TInstanceInfo> result;
        for (const NYT::TNode& instanceNode : instanceNodes) {
            bool hasAcquiredExclusiveLock = false;
            bool hasAcquiredSnapshotLock = false;
            for (const auto& lock : instanceNode.GetAttributes().AsMap().at("locks").AsList()) {
                if (lock["state"] != "acquired") {
                    continue;
                }

                switch (FromString<NYT::ELockMode>(lock["mode"].AsString())) {
                    case NYT::ELockMode::LM_EXCLUSIVE:
                        hasAcquiredExclusiveLock = true;
                        break;
                    case NYT::ELockMode::LM_SNAPSHOT:
                        hasAcquiredSnapshotLock = true;
                        break;
                    default:
                        break;
                }
                if (Y_UNLIKELY(hasAcquiredExclusiveLock && hasAcquiredSnapshotLock)) {
                    break;
                }
            }

            if (!hasAcquiredExclusiveLock && !hasAcquiredSnapshotLock) {
                continue;
            }

            EInstanceRole role = hasAcquiredExclusiveLock
                ? EInstanceRole::LEADER
                : EInstanceRole::FOLLOWER;

            TInstanceInfo& info = result.emplace_back(instanceNode.AsString(), role);
            if (const auto* metaIt = instanceNode.GetAttributes().AsMap().FindPtr("meta")) {
                info.Meta = *metaIt;
            }
        }

        logFrame->LogEvent(MakeInstanceStateReaderListInstancesResultEvent(result));

        return result;
    }

    void FillStates(TVector<TInstanceInfo>& instances, NInfra::TLogFramePtr logFrame) const {
        logFrame->LogEvent(NEventlog::TInstanceStateReaderFillStates());

        NYT::TNode::TListType stateTableKeys;
        stateTableKeys.reserve(instances.size());
        for (const TInstanceInfo& instance : instances) {
            stateTableKeys.push_back(NYT::TNode()("instance", instance.Name));
        }

        const NYT::TLookupRowsOptions stateLookupOptions = NYT::TLookupRowsOptions().KeepMissingRows(true);
        logFrame->LogEvent(MakeInstanceStateReaderFillStatesLookupStateRowsEvent(StateTablePath_, stateTableKeys, stateLookupOptions));
        const NYT::TNode::TListType states = Client_->LookupRows(StateTablePath_, stateTableKeys, stateLookupOptions);
        logFrame->LogEvent(MakeInstanceStateReaderFillStatesLookupStateRowsResultEvent(states));
        Y_ENSURE(states.size() == instances.size());

        NYT::TNode::TListType versionsTableKeys;
        versionsTableKeys.reserve(states.size() * 3);

        constexpr std::array<TStringBuf, 3> timestampKeys = {{
            TStringBuf("current_timestamp"),
            TStringBuf("target_timestamp"),
            TStringBuf("sent_timestamp"),
        }};

        THashMap<TStringBuf, THashMap<ui64, size_t>> timestampInfoIdx;
        timestampInfoIdx.reserve(instances.size());
        for (const NYT::TNode& state : states) {
            if (!state.IsMap()) {
                continue;
            }

            for (const TStringBuf timestampKey : timestampKeys) {
                if (const auto* timestamp = state.AsMap().FindPtr(timestampKey); timestamp->IsUint64()) {
                    if (const auto& [it, inserted] = timestampInfoIdx[state["instance"].AsString()].emplace(timestamp->AsUint64(), versionsTableKeys.size()); inserted) {
                        versionsTableKeys.push_back(
                            NYT::TNode()
                                ("instance", state["instance"].AsString())
                                ("timestamp", timestamp->AsUint64())
                        );
                    }
                }
            }
        }

        NYT::TLookupRowsOptions versionsLookupOptions = NYT::TLookupRowsOptions().KeepMissingRows(true);
        logFrame->LogEvent(MakeInstanceStateReaderFillStatesLookupVersionInfosRowsEvent(VersionsTablePath_, versionsTableKeys, versionsLookupOptions));
        const NYT::TNode::TListType versions = Client_->LookupRows(VersionsTablePath_, versionsTableKeys, versionsLookupOptions);
        logFrame->LogEvent(MakeInstanceStateReaderFillStatesLookupVersionInfosRowsResultEvent(versions));

        for (size_t i = 0; i < instances.size(); ++i) {
            if (!states[i].IsMap()) {
                continue;
            }

            for (const TStringBuf timestampKey : timestampKeys) {
                if (const auto* timestamp = states[i].AsMap().FindPtr(timestampKey); timestamp->IsUint64()) {
                    const NYT::TNode& timestampInfo = versions.at(timestampInfoIdx.at(instances[i].Name).at(timestamp->AsUint64()));
                    if (!timestampInfo.IsMap()) {
                        continue;
                    }

                    TInstanceState* state = nullptr;
                    if (timestampKey == "current_timestamp") {
                        state = &instances[i].CurrentState.ConstructInPlace();
                    } else if (timestampKey == "target_timestamp") {
                        state = &instances[i].TargetState.ConstructInPlace();
                    } else if (timestampKey == "sent_timestamp") {
                        state = &instances[i].SentState.ConstructInPlace();
                    }

                    if (!state) {
                        continue;
                    }

                    state->Timestamp.ConstructInPlace(timestamp->AsUint64());
                    if (const auto* receiveTime = timestampInfo.AsMap().FindPtr("receive_time"); receiveTime->IsUint64()) {
                        state->Timestamp->ReceiveTime(TInstant::MilliSeconds(receiveTime->AsUint64()));
                    }
                    if (const auto* updateTime = timestampInfo.AsMap().FindPtr("update_time"); updateTime->IsUint64()) {
                        state->Timestamp->UpdateTime(TInstant::MilliSeconds(updateTime->AsUint64()));
                    }
                    if (const auto* sentTime = timestampInfo.AsMap().FindPtr("sent_time"); sentTime->IsUint64()) {
                        state->Timestamp->SentTime(TInstant::MilliSeconds(sentTime->AsUint64()));
                    }
                    if (const auto* updateStatus = timestampInfo.AsMap().FindPtr("update_status"); updateStatus->HasValue()) {
                        state->Timestamp->UpdateStatus(*updateStatus);
                    }
                    if (const auto* updateAttempts = timestampInfo.AsMap().FindPtr("update_attempts"); updateAttempts->IsUint64()) {
                        state->Timestamp->UpdateAttempts(updateAttempts->AsUint64());
                    }
                }
            }
        }

        logFrame->LogEvent(MakeInstanceStateReaderFillStatesResult(instances));
    }

private:
    const TInstanceStateReaderOptions Options_;
    const NYT::IClientPtr Client_;

    const NYT::TYPath ServicePath_;
    const NYT::TYPath InstancesPath_;
    const NYT::TYPath StateTablePath_;
    const NYT::TYPath VersionsTablePath_;
};

TInstanceStateReader::TInstanceStateReader(TInstanceStateReaderOptions options)
    : Impl_(MakeHolder<TImpl>(std::move(options)))
{
}

TInstanceStateReader::~TInstanceStateReader() = default;

TVector<TInstanceInfo> TInstanceStateReader::GetInstanceInfos(NInfra::TLogFramePtr logFrame) const {
    return Impl_->GetInstanceInfos(logFrame);
}

} // namespace NYPUpdatesCoordinator
