#include "state.h"

#include <yt/yt/core/misc/public.h>
#include <yt/yt/core/misc/serialize.h>

namespace NYPUpdatesCoordinator {

TUpdateStatus::TUpdateStatus(const NYT::TNode& node) {
    FromNode(node);
}

TUpdateStatus::TUpdateStatus(EStatus status)
    : Status_(status)
{
}

TUpdateStatus::TUpdateStatus(NYP::NClient::TResponseError error)
    : Status_(YP_ERROR)
    , YpRequestStatus_(std::move(error))
{
}

TUpdateStatus::EStatus TUpdateStatus::Status() const {
    return Status_;
}

const TMaybe<NYP::NClient::TResponseError>& TUpdateStatus::YpRequest() const {
    return YpRequestStatus_;
}

NYT::TNode TUpdateStatus::ToNode() const {
    NYT::TNode result;
    result["status"] = ToString(Status_);
    if (YpRequestStatus_.Defined()) {
        result["yp-req"] = YpRequestStatusToNode(*YpRequestStatus_);
    }
    return result;
}

void TUpdateStatus::FromNode(const NYT::TNode& node) {
    if (!node.HasValue()) {
        Status_ = OK;
        return;
    }
    Status_ = FromString<EStatus>(node.ChildAsString("status"));
    switch (Status_) {
        case YP_ERROR:
            YpRequestStatus_ = YpRequestStatusFromNode(node["yp-req"]);
            break;
        case OK:
        default:
            break;
    }
}

NYT::TNode TUpdateStatus::YpRequestStatusToNode(const NYP::NClient::TResponseError& error) {
    NYT::TNode result;
    result["id"] = error.RequestId();
    result["rsp-code"] = static_cast<int>(error.Code());
    result["rsp-msg"] = error.Message();
    result["yt-err"]["code"] = static_cast<int>(error.Error().GetCode());
    result["yt-err"]["generic-code"] = static_cast<int>(error.Error().GetNonTrivialCode());
    result["yt-err"]["msg"] = error.Error().GetMessage();
    TStringStream ytError;
    NYT::TStreamSaveContext context(&ytError);
    error.Error().Save(context);
    result["yt-err"]["raw"] = ytError.Str();
    return result;
}

NYP::NClient::TResponseError TUpdateStatus::YpRequestStatusFromNode(const NYT::TNode& node) {
    NYT::TError error;
    TStringInput input(node["yt-err"]["raw"].AsString());
    NYT::TStreamLoadContext context(&input);
    error.Load(context);
    return NYP::NClient::TResponseError{
        /* code */ static_cast<NYP::NClient::TResponseError::ECode>(node["rsp-code"].AsInt64()),
        /* message */ node["rsp-msg"].AsString(),
        /* details */ {},
        /* requestId */ node["id"].AsString(),
        /* request */ {},
        /* address */ {},
        /* error */ std::move(error)
    };
}

NYT::TNode TTimestampInfo::ToNode() const {
    NYT::TNode result = NYT::TNode()("timestamp", Value());
    if (const TMaybe<TInstant>& receiveTime = ReceiveTime(); receiveTime.Defined()) {
        result("receive_time", receiveTime->MilliSeconds());
    }
    if (const TMaybe<TInstant>& updateTime = UpdateTime(); updateTime.Defined()) {
        result("update_time", updateTime->MilliSeconds());
    }
    if (const TMaybe<TInstant>& sentTime = SentTime(); sentTime.Defined()) {
        result("sent_time", sentTime->MilliSeconds());
    }
    if (const TMaybe<TUpdateStatus>& updateStatus = UpdateStatus(); updateStatus.Defined()) {
        result("update_status", updateStatus->ToNode());
    }
    if (const TMaybe<ui64>& updateAttempts = UpdateAttempts(); updateAttempts.Defined()) {
        result("update_attempts", *updateAttempts);
    }
    return result;
}

} // namespace NYPUpdatesCoordinator
