#pragma once

#include <mapreduce/yt/interface/node.h>

#include <yp/cpp/yp/error.h>

#include <util/datetime/base.h>
#include <util/generic/maybe.h>
#include <util/system/types.h>

namespace NYPUpdatesCoordinator {

struct TUpdateStatus {
public:
    enum EStatus {
        OK = 0,
        YP_ERROR = 1,
    };

    TUpdateStatus(const NYT::TNode& node);
    TUpdateStatus(EStatus status);
    TUpdateStatus(NYP::NClient::TResponseError error);

    EStatus Status() const;

    const TMaybe<NYP::NClient::TResponseError>& YpRequest() const;

    NYT::TNode ToNode() const;

    void FromNode(const NYT::TNode& node);

private:
    static NYT::TNode YpRequestStatusToNode(const NYP::NClient::TResponseError& error);
    static NYP::NClient::TResponseError YpRequestStatusFromNode(const NYT::TNode& node);

private:
    EStatus Status_;
    TMaybe<NYP::NClient::TResponseError> YpRequestStatus_;
};

struct TTimestampBaseInfo {
public:
    explicit TTimestampBaseInfo(ui64 value)
        : Value_(value)
    {
    }

    ui64 Value() const {
        return Value_;
    }

private:
    ui64 Value_ = 0;
};

struct TTimestampClientInfo: virtual public TTimestampBaseInfo {
public:
    using TTimestampBaseInfo::TTimestampBaseInfo;

    TTimestampClientInfo& ReceiveTime(TInstant value) {
        ReceiveTime_ = std::move(value);
        return *this;
    }

    const TMaybe<TInstant>& ReceiveTime() const {
        return ReceiveTime_;
    }

    TTimestampClientInfo& UpdateTime(TInstant value) {
        UpdateTime_ = std::move(value);
        return *this;
    }

    const TMaybe<TInstant>& UpdateTime() const {
        return UpdateTime_;
    }

    TTimestampClientInfo& UpdateStatus(TUpdateStatus value) {
        UpdateStatus_.ConstructInPlace(std::move(value));
        return *this;
    }

    const TMaybe<TUpdateStatus>& UpdateStatus() const {
        return UpdateStatus_;
    }

    TTimestampClientInfo& UpdateAttempts(ui32 value) {
        UpdateAttempts_ = value;
        return *this;
    }

    TTimestampClientInfo& IncreaseUpdateAttempts() {
        UpdateAttempts_ = UpdateAttempts_.GetOrElse(0) + 1;
        return *this;
    }

    const TMaybe<ui64>& UpdateAttempts() const {
        return UpdateAttempts_;
    }

private:
    TMaybe<TInstant> ReceiveTime_;
    TMaybe<TInstant> UpdateTime_;
    TMaybe<TUpdateStatus> UpdateStatus_;
    TMaybe<ui64> UpdateAttempts_;
};

struct TTimestampProviderInfo: virtual public TTimestampBaseInfo {
public:
    using TTimestampBaseInfo::TTimestampBaseInfo;

    TTimestampProviderInfo& SentTime(TInstant value) {
        SentTime_ = std::move(value);
        return *this;
    }

    const TMaybe<TInstant>& SentTime() const {
        return SentTime_;
    }

private:
    TMaybe<TInstant> SentTime_;
};

struct TTimestampInfo: public TTimestampClientInfo, public TTimestampProviderInfo {
public:
    TTimestampInfo(ui64 value)
        : TTimestampBaseInfo(value)
        , TTimestampClientInfo(value)
        , TTimestampProviderInfo(value)
    {
    }

    TTimestampInfo(TTimestampClientInfo info)
        : TTimestampBaseInfo(info.Value())
        , TTimestampClientInfo(std::move(info))
        , TTimestampProviderInfo(info.Value())
    {
    }

    TTimestampInfo(TTimestampProviderInfo info)
        : TTimestampBaseInfo(info.Value())
        , TTimestampClientInfo(info.Value())
        , TTimestampProviderInfo(std::move(info))
    {
    }

    NYT::TNode ToNode() const;
};

struct TInstanceState {
    TMaybe<TTimestampInfo> Timestamp;
};

enum class EInstanceRole {
    LEADER = 0,
    FOLLOWER = 1,
};

struct TInstanceInfo {
    TInstanceInfo(TString name, EInstanceRole role)
        : Name(std::move(name))
        , Role(role)
    {
    }

    const TString Name;
    EInstanceRole Role = EInstanceRole::LEADER;
    TMaybe<NYT::TNode> Meta;
    TMaybe<TInstanceState> CurrentState;
    TMaybe<TInstanceState> TargetState;
    TMaybe<TInstanceState> SentState;
};

} // namespace NYPUpdatesCoordinator
