#include "writer.h"

#include <infra/libs/yp_updates_coordinator/logger/make_events.h>

#include <infra/libs/leading_invader/leading_invader.h>

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/util/wait_for_tablets_state.h>
#include <mapreduce/yt/util/ypath_join.h>

#include <library/cpp/yson/node/node_io.h>
#include <library/cpp/threading/future/async.h>

#include <util/datetime/base.h>
#include <util/system/rwlock.h>
#include <util/thread/pool.h>

namespace NYPUpdatesCoordinator::NDetail {

class TInstanceStateBaseWriter::TImpl {
public:
    TImpl(TInstanceStateBaseWriterOptions options, NInfra::TLogFramePtr logFrame)
        : Options_(std::move(options))
        , MainLogFrame_(logFrame)
        , Client_(NYT::CreateClient(Options_.YtProxy))
        , InstanceName_(Options_.InstanceName)
        , InstanceRole_(Options_.ParticipateInCoordination
            ? EInstanceRole::LEADER
            : EInstanceRole::FOLLOWER)
        , ServicePath_(NYT::JoinYPaths(Options_.CypressRootPath, Options_.Service))
        , InstancePath_(NYT::JoinYPaths(ServicePath_, "instances", InstanceName_))
        , StateTablePath_(NYT::JoinYPaths(ServicePath_, "meta", "state"))
        , VersionsTablePath_(NYT::JoinYPaths(ServicePath_, "meta", "versions"))
        , InitEnvPool_(MakeHolder<TThreadPool>())
    {
        InitEnvPool_->Start(1);
        InitYtEnvironment(MainLogFrame_);
    }

    ~TImpl() {
        try {
            InitEnvPool_->Stop();
            if (Options_.Register) {
                {
                    TWriteGuard guard(LockMutex_);
                    Lock_.Destroy();
                }
                RemoveInstancePath(MainLogFrame_);
            }
        } catch (...) {
            MainLogFrame_->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TInstanceStateWriterStopFailure(CurrentExceptionMessage()));
        }
    }

    void StartParticipateInCoordination(NInfra::TLogFramePtr logFrame) {
        Y_UNUSED(logFrame);

        Y_ENSURE(Options_.Register);

        bool needReacquireLock;
        {
            TReadGuard guard(LockMutex_);
            // We need to reacquire lock manually when the role is changed and some lock has
            // already been acquired (if it is not yet then it will be acquired automatically
            // within the initialization of YT environment later).
            needReacquireLock =
                InstanceRole_.exchange(EInstanceRole::LEADER) != EInstanceRole::LEADER &&
                static_cast<bool>(Lock_);
        }
        if (needReacquireLock) {
            LockInstancePath();
        }
    }

    void WaitForReadiness(NInfra::TLogFramePtr logFrame) {
        WaitForReadiness(true, logFrame);
    }

    void UpdateCurrentState(const TTimestampClientInfo& info, NInfra::TLogFramePtr logFrame) {
        ThrowIfEnvIsNotReady();

        UpdateTimestampInfo(info, logFrame);
        SetCurrentTimestamp(info.Value(), logFrame);
    }

    void UpdateTargetState(const TTimestampClientInfo& info, NInfra::TLogFramePtr logFrame) {
        ThrowIfEnvIsNotReady();

        UpdateTimestampInfo(info, logFrame);
        SetTargetTimestamp(info.Value(), logFrame);
    }

    void UpdateSentState(const TString& instanceName, const TTimestampProviderInfo& info, NInfra::TLogFramePtr logFrame) {
        ThrowIfEnvIsNotReady();

        UpdateTimestampInfo(instanceName, info, logFrame);
        SetSentTimestamp(instanceName, info.Value(), logFrame);
    }

    void UpdateTimestampUpdateStatus(const ui64 timestamp, TUpdateStatus status, NInfra::TLogFramePtr logFrame) {
        ThrowIfEnvIsNotReady();

        TTimestampClientInfo info(timestamp);
        info.UpdateStatus(std::move(status));
        info.IncreaseUpdateAttempts();
        UpdateTimestampInfo(info, logFrame);
    }

private:
    void SetCurrentTimestamp(ui64 timestamp, NInfra::TLogFramePtr logFrame) {
        SetTimestamp("current_timestamp", timestamp, logFrame);
    }

    void SetTargetTimestamp(ui64 timestamp, NInfra::TLogFramePtr logFrame) {
        SetTimestamp("target_timestamp", timestamp, logFrame);
    }

    void SetSentTimestamp(const TString& instanceName, ui64 timestamp, NInfra::TLogFramePtr logFrame) {
        SetTimestamp(instanceName, "sent_timestamp", timestamp, logFrame);
    }

    void SetTimestamp(const TString& type, ui64 timestamp, NInfra::TLogFramePtr logFrame) {
        SetTimestamp(InstanceName_, type, timestamp, logFrame);
    }

    void SetTimestamp(const TString& instanceName, const TString& type, ui64 timestamp, NInfra::TLogFramePtr logFrame) {
        const NYT::TNode::TListType rows = {
            NYT::TNode()
                ("instance", instanceName)
                (type, timestamp)
        };
        const NYT::TInsertRowsOptions options = NYT::TInsertRowsOptions().Update(true);

        logFrame->LogEvent(MakeInstanceStateWriterSetTimestampEvent(StateTablePath_, rows, options));
        try {
            Client_->InsertRows(StateTablePath_, rows, options);
            logFrame->LogEvent(NEventlog::TInstanceStateWriterSetTimestampSuccess());
        } catch (...) {
            logFrame->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TInstanceStateWriterSetTimestampFailure(CurrentExceptionMessage()));
        }
    }

    void UpdateTimestampInfo(const TTimestampInfo& info, NInfra::TLogFramePtr logFrame) {
        UpdateTimestampInfo(InstanceName_, info, logFrame);
    }

    void UpdateTimestampInfo(const TString& instanceName, const TTimestampInfo& info, NInfra::TLogFramePtr logFrame) {
        const NYT::TInsertRowsOptions options = NYT::TInsertRowsOptions().Update(true).Aggregate(true);
        const NYT::TNode::TListType rows = {info.ToNode()("instance", instanceName)};
        logFrame->LogEvent(MakeInstanceStateWriterUpdateTimestampInfoEvent(VersionsTablePath_, rows, options));
        try {
            Client_->InsertRows(VersionsTablePath_, rows, options);
            logFrame->LogEvent(NEventlog::TInstanceStateWriterUpdateTimestampInfoSuccess());
        } catch (...) {
            logFrame->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TInstanceStateWriterUpdateTimestampInfoFailure(CurrentExceptionMessage()));
            throw;
        }
    }

private:
    void InitYtEnvironment(NInfra::TLogFramePtr logFrame) {
        logFrame->LogEvent(NEventlog::TInstanceStateWriterInitializeYtEnvironment());

        TWriteGuard guard(InitYtEnvFutureMutex_);
        if (InitYtEnvFuture_.Initialized() && !InitYtEnvFuture_.HasValue() && !InitYtEnvFuture_.HasException()) {
            ythrow yexception() << "Initialization is still running";
        }

        InitYtEnvFuture_ = NThreading::Async([this, logFrame] {
            CreateAndMountStateTable(logFrame);
            CreateAndMountVersionsTable(logFrame);
            if (Options_.Register) {
                RegisterInstance(logFrame);
            }
        }, *InitEnvPool_);
    }

    NThreading::TFuture<void> GetInitYtEnvFuture() const {
        TReadGuard guard(InitYtEnvFutureMutex_);
        return InitYtEnvFuture_;
    }

    void CreateAndMountStateTable(NInfra::TLogFramePtr logFrame) const {
        const NYT::TTableSchema schema = NYT::TTableSchema()
            .AddColumn("instance", NYT::EValueType::VT_STRING, NYT::ESortOrder::SO_ASCENDING)
            .AddColumn(NYT::TColumnSchema().Name("current_timestamp").Type(NYT::EValueType::VT_UINT64).Lock("client"))
            .AddColumn(NYT::TColumnSchema().Name("target_timestamp").Type(NYT::EValueType::VT_UINT64).Lock("client"))
            .AddColumn(NYT::TColumnSchema().Name("sent_timestamp").Type(NYT::EValueType::VT_UINT64).Lock("coordinator"));

        NYT::TNode attributes = NYT::TNode()
            ("dynamic", true)
            ("schema", schema.ToNode())
            ("min_data_versions", 0)
            ("max_data_versions", 1)
            ("min_data_ttl", 6 * 60 * 60 * 1000)
            ("max_data_ttl", 12 * 60 * 60 * 1000);

        if (Options_.PrimaryMedium.Defined()) {
            attributes("primary_medium", *Options_.PrimaryMedium);
        }

        const NYT::TCreateOptions createOptions = NYT::TCreateOptions()
            .IgnoreExisting(true)
            .Recursive(true)
            .Attributes(attributes);

        logFrame->LogEvent(MakeInstanceStateWriterCreateStateTableEvent(StateTablePath_, NYT::NT_TABLE, createOptions));
        try {
            Client_->Create(StateTablePath_, NYT::NT_TABLE, createOptions);
            logFrame->LogEvent(NEventlog::TInstanceStateWriterCreateStateTableSuccess());
        } catch (...) {
            logFrame->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TInstanceStateWriterCreateStateTableFailure(CurrentExceptionMessage()));
            throw;
        }

        logFrame->LogEvent(NEventlog::TInstanceStateWriterMountStateTable(StateTablePath_));
        try {
            Client_->MountTable(StateTablePath_);
            logFrame->LogEvent(NEventlog::TInstanceStateWriterMountStateTableSuccess());
        } catch (...) {
            logFrame->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TInstanceStateWriterMountStateTableFailure(CurrentExceptionMessage()));
            throw;
        }

        const NYT::TWaitForTabletsStateOptions waitForTabletsStateOptions = NYT::TWaitForTabletsStateOptions()
            .CheckInterval(TDuration::Seconds(1))
            .Timeout(TDuration::Seconds(60));

        logFrame->LogEvent(MakeInstanceStateWriterWaitForMountedStateTableEvent(StateTablePath_, NYT::ETabletState::TS_MOUNTED, waitForTabletsStateOptions));
        try {
            NYT::WaitForTabletsState(Client_, StateTablePath_, NYT::ETabletState::TS_MOUNTED, waitForTabletsStateOptions);
            logFrame->LogEvent(NEventlog::TInstanceStateWriterWaitForMountedStateTableSuccess());
        } catch (...) {
            logFrame->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TInstanceStateWriterWaitForMountedStateTableFailure(CurrentExceptionMessage()));
            throw;
        }
    }

    void CreateAndMountVersionsTable(NInfra::TLogFramePtr logFrame) const {
        const NYT::TTableSchema schema = NYT::TTableSchema()
            .AddColumn("instance", NYT::EValueType::VT_STRING, NYT::ESortOrder::SO_ASCENDING)
            .AddColumn("timestamp", NYT::EValueType::VT_UINT64, NYT::ESortOrder::SO_ASCENDING)
            .AddColumn(NYT::TColumnSchema().Name("receive_time").Type(NYT::EValueType::VT_UINT64).Lock("client").Aggregate("min"))
            .AddColumn(NYT::TColumnSchema().Name("update_time").Type(NYT::EValueType::VT_UINT64).Lock("client").Aggregate("min"))
            .AddColumn(NYT::TColumnSchema().Name("sent_time").Type(NYT::EValueType::VT_UINT64).Lock("coordinator").Aggregate("min"))
            .AddColumn(NYT::TColumnSchema().Name("update_status").Type(NYT::EValueType::VT_ANY).Lock("client"))
            .AddColumn(NYT::TColumnSchema().Name("update_attempts").Type(NYT::EValueType::VT_UINT64).Lock("client").Aggregate("sum"));

        NYT::TNode attributes = NYT::TNode()
            ("dynamic", true)
            ("schema", schema.ToNode())
            ("min_data_versions", 0)
            ("max_data_versions", 1)
            ("min_data_ttl", 6 * 60 * 60 * 1000)
            ("max_data_ttl", 12 * 60 * 60 * 1000);

        if (Options_.PrimaryMedium.Defined()) {
            attributes("primary_medium", *Options_.PrimaryMedium);
        }

        const NYT::TCreateOptions createOptions = NYT::TCreateOptions()
            .IgnoreExisting(true)
            .Recursive(true)
            .Attributes(attributes);

        logFrame->LogEvent(MakeInstanceStateWriterCreateVersionsTableEvent(VersionsTablePath_, NYT::NT_TABLE, createOptions));
        try {
            Client_->Create(VersionsTablePath_, NYT::NT_TABLE, createOptions);
            logFrame->LogEvent(NEventlog::TInstanceStateWriterMountVersionsTableSuccess());
        } catch (...) {
            logFrame->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TInstanceStateWriterMountVersionsTableFailure(CurrentExceptionMessage()));
            throw;
        }

        logFrame->LogEvent(NEventlog::TInstanceStateWriterMountVersionsTable(VersionsTablePath_));
        try {
            Client_->MountTable(VersionsTablePath_);
            logFrame->LogEvent(NEventlog::TInstanceStateWriterMountVersionsTableSuccess());
        } catch (...) {
            logFrame->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TInstanceStateWriterMountVersionsTableFailure(CurrentExceptionMessage()));
            throw;
        }

        const NYT::TWaitForTabletsStateOptions waitForTabletsVersionsOptions = NYT::TWaitForTabletsStateOptions()
            .CheckInterval(TDuration::Seconds(1))
            .Timeout(TDuration::Seconds(60));

        logFrame->LogEvent(MakeInstanceStateWriterWaitForMountedVersionsTableEvent(VersionsTablePath_, NYT::ETabletState::TS_MOUNTED, waitForTabletsVersionsOptions));
        try {
            NYT::WaitForTabletsState(Client_, VersionsTablePath_, NYT::ETabletState::TS_MOUNTED, waitForTabletsVersionsOptions);
            logFrame->LogEvent(NEventlog::TInstanceStateWriterWaitForMountedVersionsTableSuccess());
        } catch (...) {
            logFrame->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TInstanceStateWriterWaitForMountedVersionsTableFailure(CurrentExceptionMessage()));
            throw;
        }
    }

    void WaitForReadiness(bool retryUntilSuccess, NInfra::TLogFramePtr logFrame) {
        logFrame->LogEvent(NEventlog::TInstanceStateWriterWaitForReadiness(retryUntilSuccess));
        while (true) {
            NThreading::TFuture<void> initFuture = GetInitYtEnvFuture();
            const TDuration waitTimeout = TDuration::Seconds(5);
            logFrame->LogEvent(NEventlog::TInstanceStateWriterWaitForYtEnvironmentReadiness(ToString(waitTimeout)));
            if (!initFuture.Wait(waitTimeout)) {
                logFrame->LogEvent(ELogPriority::TLOG_WARNING, NEventlog::TInstanceStateWriterYtEnvironmentIsNotReady());
                continue;
            }

            if (initFuture.HasException()) {
                try {
                    initFuture.TryRethrow();
                } catch (...) {
                    logFrame->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TInstanceStateWriterYtEnvironmentSetupFailed(CurrentExceptionMessage()));
                    if (retryUntilSuccess) {
                        InitYtEnvironment(logFrame);
                    } else {
                        throw;
                    }
                }
           } else {
                logFrame->LogEvent(NEventlog::TInstanceStateWriterYtEnvironmentIsReady());
                break;
            }
        }
        InitEnvPool_->Stop();

        if (Options_.Register) {
            WaitForRegistrationCompleted();
        }
    }

    void WaitForRegistrationCompleted() const {
        TReadGuard guard(LockMutex_);
        if (!Lock_) {
            ythrow yexception() << "Lock is not initialized";
        }

        while (true) {
            if (auto leading = Lock_->EnsureLeading(); !leading) {
                // TODO log
            } else {
                break;
            }
            Sleep(TDuration::MilliSeconds(100));
        }
    }

    void ThrowIfEnvIsNotReady() {
        ThrowIfYtEnvIsNotReady();

        if (Options_.Register) {
            ThrowIfNotRegistered();
        }
    }

    void ThrowIfYtEnvIsNotReady() {
        NThreading::TFuture<void> initFuture = GetInitYtEnvFuture();
        if (initFuture.HasException()) {
            InitYtEnvironment(MainLogFrame_);
            initFuture.TryRethrow();
        }

        if (!initFuture.HasValue()) {
            ythrow yexception() << "YT environment is not ready yet";
        }
    }

    void ThrowIfNotRegistered() const {
        TReadGuard guard(LockMutex_);
        if (!Lock_) {
            ythrow yexception() << "Lock is not initialized";
        }

        if (auto locked = Lock_->EnsureLeading(); !locked) {
            ythrow yexception() << "Lost lock '" << locked.Error().Reason << "'";
        }
    }

    void RegisterInstance(NInfra::TLogFramePtr logFrame) {
        NYT::TCreateOptions createOptions = NYT::TCreateOptions().Recursive(true).Force(true);

        NYT::TNode attributes;
        if (Options_.Meta.Defined()) {
            attributes["meta"] = *Options_.Meta;
        }
        if (Options_.PrimaryMedium.Defined()) {
            attributes["primary_medium"] = *Options_.PrimaryMedium;
        }

        if (attributes.IsMap()) {
            createOptions.Attributes(attributes);
        }

        logFrame->LogEvent(MakeInstanceStateWriterCreateInstanceDocumentEvent(InstancePath_, NYT::NT_DOCUMENT, createOptions));
        try {
            Client_->Create(InstancePath_, NYT::NT_DOCUMENT, createOptions);
            logFrame->LogEvent(NEventlog::TInstanceStateWriterCreateInstanceDocumentSuccess());
        } catch (...) {
            logFrame->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TInstanceStateWriterCreateInstanceDocumentFailure(CurrentExceptionMessage()));
            throw;
        }

        LockInstancePath();
    }

    void LockInstancePath() {
        // Access to *InstanceRole_* here must be under the lock on *LockMutex_* since it
        // can be changed before creating a leading invader.
        TWriteGuard guard(LockMutex_);

        NInfra::NLeadingInvader::TConfig config;
        config.SetProxy(Options_.YtProxy);
        config.SetToken(Options_.YtToken);
        config.SetPath(InstancePath_);
        config.SetTimeout(Options_.LockTimeout.ToString());
        config.SetRetryAcquireLockInterval("10ms");
        switch (InstanceRole_.load()) {
            case EInstanceRole::LEADER:
                config.SetLockMode(NInfra::NLeadingInvader::TConfig::EXCLUSIVE);
                break;
            case EInstanceRole::FOLLOWER:
                config.SetLockMode(NInfra::NLeadingInvader::TConfig::SHARED);
                break;
        }

        Lock_ = NInfra::NLeadingInvader::CreateLeadingInvader(config);
    }

    void RemoveInstancePath(NInfra::TLogFramePtr logFrame) {
        const NYT::TRemoveOptions removeOptions = NYT::TRemoveOptions().Force(true);
        logFrame->LogEvent(MakeInstanceStateWriterRemoveInstanceDocumentEvent(InstancePath_, removeOptions));
        try {
            Client_->Remove(InstancePath_, removeOptions);
            logFrame->LogEvent(NEventlog::TInstanceStateWriterRemoveInstanceDocumentSuccess());
        } catch (...) {
            logFrame->LogEvent(NEventlog::TInstanceStateWriterRemoveInstanceDocumentFailure(CurrentExceptionMessage()));
            throw;
        }
    }

private:
    const TInstanceStateBaseWriterOptions Options_;
    NInfra::TLogFramePtr MainLogFrame_;

    const NYT::IClientPtr Client_;
    const TString InstanceName_;
    std::atomic<EInstanceRole> InstanceRole_;

    const NYT::TYPath ServicePath_;
    const NYT::TYPath InstancePath_;
    const NYT::TYPath StateTablePath_;
    const NYT::TYPath VersionsTablePath_;

    THolder<IThreadPool> InitEnvPool_;
    TRWMutex InitYtEnvFutureMutex_;
    NThreading::TFuture<void> InitYtEnvFuture_;

    TRWMutex LockMutex_;
    NInfra::NLeadingInvader::TLeadingInvaderHolder Lock_;
};

TInstanceStateBaseWriter::TInstanceStateBaseWriter(TInstanceStateBaseWriterOptions options, NInfra::TLogFramePtr logFrame)
    : Impl_(MakeHolder<TImpl>(std::move(options), logFrame))
{
}

TInstanceStateBaseWriter::~TInstanceStateBaseWriter() = default;

void TInstanceStateBaseWriter::StartParticipateInCoordination(NInfra::TLogFramePtr logFrame) {
    Impl_->StartParticipateInCoordination(logFrame);
}

void TInstanceStateBaseWriter::WaitForReadiness(NInfra::TLogFramePtr logFrame) {
    Impl_->WaitForReadiness(logFrame);
}

void TInstanceStateBaseWriter::UpdateCurrentState(const TTimestampClientInfo& info, NInfra::TLogFramePtr logFrame) {
    Impl_->UpdateCurrentState(info, logFrame);
}

void TInstanceStateBaseWriter::UpdateTargetState(const TTimestampClientInfo& info, NInfra::TLogFramePtr logFrame) {
    Impl_->UpdateTargetState(info, logFrame);
}

void TInstanceStateBaseWriter::UpdateSentState(const TString& instanceName, const TTimestampProviderInfo& info, NInfra::TLogFramePtr logFrame) {
    Impl_->UpdateSentState(instanceName, info, logFrame);
}

void TInstanceStateBaseWriter::UpdateTimestampUpdateStatus(const ui64 timestamp, TUpdateStatus status, NInfra::TLogFramePtr logFrame) {
    Impl_->UpdateTimestampUpdateStatus(timestamp, std::move(status), logFrame);
}

TInstanceStateClientWriter::TInstanceStateClientWriter(TInstanceStateClientWriterOptions options, NInfra::TLogFramePtr logFrame)
    : TInstanceStateBaseWriter(std::move(options), logFrame)
{
}

TInstanceStateProviderWriter::TInstanceStateProviderWriter(TInstanceStateProviderWriterOptions options, NInfra::TLogFramePtr logFrame)
    : TInstanceStateBaseWriter(std::move(options), logFrame)
{
}

} // namespace NYPUpdatesCoordinator::NDetail
