#include "make_events.h"

#include <library/cpp/yson/node/node_io.h>

namespace NYPUpdatesCoordinator {

namespace {

NEventlog::TLookupRowsOptions MakeLookupRowsOptionsEvent(const NYT::TLookupRowsOptions& options) {
    NEventlog::TLookupRowsOptions event;
    if (options.Columns_.Defined()) {
        event.MutableColumns()->Reserve(options.Columns_->Parts_.size());
        for (const TString& column : options.Columns_->Parts_) {
            *event.AddColumns() = column;
        }
    }
    if (options.Timeout_.Defined()) {
        event.SetTimeout(ToString(*options.Timeout_));
    }
    event.SetKeepMissingRows(options.KeepMissingRows_);
    return event;
}

NEventlog::TLookupRowsRequest MakeLookupRowsRequestEvent(const NYT::TYPath& path, const NYT::TNode::TListType& keys, const NYT::TLookupRowsOptions& options) {
    NEventlog::TLookupRowsRequest event;
    event.SetPath(path);
    event.MutableKeys()->Reserve(keys.size());
    for (const NYT::TNode& key : keys) {
        *event.AddKeys() = NYT::NodeToYsonString(key);
    }
    *event.MutableOptions() = MakeLookupRowsOptionsEvent(options);
    return event;
}

NEventlog::TInsertRowsOptions MakeInsertRowsOptionsEvent(const NYT::TInsertRowsOptions& options) {
    NEventlog::TInsertRowsOptions event;
    if (options.Aggregate_.Defined()) {
        event.SetAggregate(*options.Aggregate_);
    }
    if (options.Update_.Defined()) {
        event.SetUpdate(*options.Update_);
    }
    return event;
}

NEventlog::TInsertRowsRequest MakeInsertRowsRequestEvent(const NYT::TYPath& path, const NYT::TNode::TListType& rows, const NYT::TInsertRowsOptions& options) {
    NEventlog::TInsertRowsRequest event;
    event.SetPath(path);
    event.MutableRows()->Reserve(rows.size());
    for (const NYT::TNode& row : rows) {
        *event.AddRows() = NYT::NodeToYsonString(row);
    }
    *event.MutableOptions() = MakeInsertRowsOptionsEvent(options);
    return event;
}

NEventlog::TCreateOptions MakeCreateOptionsEvent(const NYT::TCreateOptions& options) {
    NEventlog::TCreateOptions event;
    event.SetIgnoreExisting(options.IgnoreExisting_);
    event.SetRecursive(options.Recursive_);
    event.SetForce(options.Force_);
    if (options.Attributes_.Defined()) {
        event.SetAttributes(NYT::NodeToYsonString(*options.Attributes_));
    }
    return event;
}

NEventlog::TCreateRequest MakeCreateRequestEvent(const NYT::TYPath& path, NYT::ENodeType nodeType, const NYT::TCreateOptions& options) {
    NEventlog::TCreateRequest event;
    event.SetPath(path);
    event.SetType(ToString(nodeType));
    *event.MutableOptions() = MakeCreateOptionsEvent(options);
    return event;
}

NEventlog::TWaitForTabletsStateOptions MakeWaitForTabletsStateOptionsEvent(const NYT::TWaitForTabletsStateOptions& options) {
    NEventlog::TWaitForTabletsStateOptions event;
    event.SetTimeout(ToString(options.Timeout_));
    event.SetCheckInterval(ToString(options.CheckInterval_));
    return event;
}

NEventlog::TWaitForTabletsStateRequest MakeWaitForTabletsStateRequestEvent(
    const NYT::TYPath& path,
    const NYT::ETabletState& targetTabletsState,
    const NYT::TWaitForTabletsStateOptions& waitForTabletsStateOptions
) {
    NEventlog::TWaitForTabletsStateRequest event;
    event.SetPath(path);
    event.SetTargetState(ToString(targetTabletsState));
    *event.MutableOptions() = MakeWaitForTabletsStateOptionsEvent(waitForTabletsStateOptions);
    return event;
}

NEventlog::TRemoveOptions MakeRemoveOptionsEvent(const NYT::TRemoveOptions& options) {
    NEventlog::TRemoveOptions event;
    event.SetRecursive(options.Recursive_);
    event.SetForce(options.Force_);
    return event;
}

NEventlog::TRemoveRequest MakeRemoveRequestEvent(const NYT::TYPath& path, const NYT::TRemoveOptions& options) {
    NEventlog::TRemoveRequest event;
    event.SetPath(path);
    *event.MutableOptions() = MakeRemoveOptionsEvent(options);
    return event;
}

} // anonymous namespace

NEventlog::TAttributes MakeAttributesEvent(const NInfra::TAttributes& attributes) {
    NEventlog::TAttributes event;
    event.MutableValues()->Reserve(attributes.size());
    for (const auto& [key, value] : attributes) {
        NEventlog::TAttribute& attribute = *event.AddValues();
        attribute.SetKey(key);
        attribute.SetValue(value);
    }
    return event;
}

NEventlog::TGetTargetStateRequest MakeGetTargetStateRequestEvent(const NApi::TReqGetTargetState& request) {
    NEventlog::TGetTargetStateRequest event;
    event.SetService(request.service());
    event.SetInstanceName(request.instance_name());
    return event;
}

NEventlog::TGetTargetStateRequest MakeGetTargetStateRequestEvent(const NApi::TReqGetTargetState& request, const NInfra::TAttributes& attributes) {
    NEventlog::TGetTargetStateRequest event = MakeGetTargetStateRequestEvent(request);
    *event.MutableAttributes() = MakeAttributesEvent(attributes);
    return event;
}

NEventlog::TGetTargetStateRequest MakeGetTargetStateRequestEvent(NInfra::TRequestPtr<NApi::TReqGetTargetState> request) {
    return MakeGetTargetStateRequestEvent(request->Get(), request->Attributes());
}

NEventlog::TGetTargetStateResponse MakeGetTargetStateResponseEvent(const NApi::TRspGetTargetState& response) {
    NEventlog::TGetTargetStateResponse event;
    event.SetStatus(NApi::TRspGetTargetState_EGetTargetStateStatus_Name(response.status()));
    event.SetErrorMessage(response.error_message());
    event.SetTimestamp(response.timestamp());
    return event;
}

NEventlog::TGetTargetStateResponse MakeGetTargetStateResponseEvent(const NApi::TRspGetTargetState& response, const NInfra::TAttributes& attributes) {
    NEventlog::TGetTargetStateResponse event = MakeGetTargetStateResponseEvent(response);
    *event.MutableAttributes() = MakeAttributesEvent(attributes);
    return event;
}

NEventlog::TTimestampInfo MakeTimestampInfoEvent(const TTimestampInfo& timestampInfo) {
    NEventlog::TTimestampInfo event;
    event.SetValue(timestampInfo.Value());

    if (const TMaybe<TInstant>& receiveTime = timestampInfo.ReceiveTime(); receiveTime.Defined()) {
        event.SetReceiveTime(receiveTime->MilliSeconds());
    }

    if (const TMaybe<TInstant>& updateTime = timestampInfo.UpdateTime(); updateTime.Defined()) {
        event.SetUpdateTime(updateTime->MilliSeconds());
    }

    if (const TMaybe<TInstant>& sentTime = timestampInfo.SentTime(); sentTime.Defined()) {
        event.SetSentTime(sentTime->MilliSeconds());
    }

    if (const TMaybe<TUpdateStatus>& updateStatus = timestampInfo.UpdateStatus(); updateStatus.Defined()) {
        event.SetUpdateStatus(NYT::NodeToYsonString(updateStatus->ToNode()));
    }

    return event;
}

NEventlog::TInstanceState MakeInstanceStateEvent(const TInstanceState& instanceState) {
    NEventlog::TInstanceState event;
    if (instanceState.Timestamp.Defined()) {
        *event.MutableTimestampInfo() = MakeTimestampInfoEvent(*instanceState.Timestamp);
    }
    return event;
}

NEventlog::TInstanceInfo MakeInstanceInfoEvent(const TInstanceInfo& instanceInfo) {
    NEventlog::TInstanceInfo event;
    event.SetName(instanceInfo.Name);
    if (instanceInfo.Meta.Defined()) {
        event.SetMeta(NYT::NodeToYsonString(*instanceInfo.Meta));
    }
    switch (instanceInfo.Role) {
        case EInstanceRole::LEADER:
            event.SetRole(NEventlog::EInstanceRole::LEADER);
            break;
        case EInstanceRole::FOLLOWER:
            event.SetRole(NEventlog::EInstanceRole::FOLLOWER);
            break;
    }
    if (instanceInfo.CurrentState.Defined()) {
        *event.MutableCurrentState() = MakeInstanceStateEvent(*instanceInfo.CurrentState);
    }
    if (instanceInfo.TargetState.Defined()) {
        *event.MutableTargetState() = MakeInstanceStateEvent(*instanceInfo.TargetState);
    }
    if (instanceInfo.SentState.Defined()) {
        *event.MutableSentState() = MakeInstanceStateEvent(*instanceInfo.SentState);
    }
    return event;
}

NEventlog::TLocationCoordinatorGetTargetState MakeLocationCoordinatorGetTargetStateEvent(const NApi::TReqGetTargetState& request, const TVector<TInstanceInfo>& instanceInfos) {
    NEventlog::TLocationCoordinatorGetTargetState event;
    *event.MutableRequest() = MakeGetTargetStateRequestEvent(request);
    event.MutableInstanceInfos()->Reserve(instanceInfos.size());
    for (const TInstanceInfo& info : instanceInfos) {
        *event.AddInstanceInfos() = MakeInstanceInfoEvent(info);
    }
    return event;
}

NEventlog::TLocationCoordinatorRequestInstanceInfo MakeLocationCoordinatorRequestInstanceInfoEvent(const TInstanceInfo* instanceInfo) {
    NEventlog::TLocationCoordinatorRequestInstanceInfo event;
    if (instanceInfo) {
        *event.MutableInfo() = MakeInstanceInfoEvent(*instanceInfo);
    }
    return event;
}

NEventlog::TLocationCoordinatorRequestInstanceLocationInfo MakeLocationCoordinatorRequestInstanceLocationInfoEvent(
    const TString& location,
    const TMaybe<ui64>& targetTimestamp
) {
    NEventlog::TLocationCoordinatorRequestInstanceLocationInfo event;
    event.SetLocation(location);
    if (targetTimestamp.Defined()) {
        event.SetTargetTimestamp(*targetTimestamp);
    }
    return event;
}

NEventlog::TLocationCoordinatorRequestInstanceLocationIsInvalid MakeLocationCoordinatorRequestInstanceLocationIsInvalidEvent(
    const TMaybe<TString>& location,
    const TVector<TString>& validLocations
) {
    NEventlog::TLocationCoordinatorRequestInstanceLocationIsInvalid event;
    if (location.Defined()) {
        event.SetLocation(*location);
    }
    event.MutableValidLocations()->Reserve(validLocations.size());
    for (const TString& validLocation : validLocations) {
        *event.AddValidLocations() = validLocation;
    }
    return event;
}

NEventlog::TLocationCoordinatorGetNextLocationTargetTimestamp MakeLocationCoordinatorGetNextLocationTargetTimestampEvent(const TString* location) {
    NEventlog::TLocationCoordinatorGetNextLocationTargetTimestamp event;
    if (location) {
        event.SetLocation(*location);
    }
    return event;
}

NEventlog::TLocationCoordinatorGetNextLocationTargetTimestampResult MakeLocationCoordinatorGetNextLocationTargetTimestampResultEvent(
    NEventlog::TLocationCoordinatorGetNextLocationTargetTimestampResult::ESource source,
    const TMaybe<ui64>& targetTimestamp
) {
    NEventlog::TLocationCoordinatorGetNextLocationTargetTimestampResult event;
    event.SetSource(source);
    if (targetTimestamp.Defined()) {
        event.SetTimestamp(*targetTimestamp);
    }
    return event;
}

NEventlog::TLocationCoordinatorGetPreviousTimestampByTimeQueryResponse MakeLocationCoordinatorGetPreviousTimestampByTimeQueryResponseEvent(
    const NYT::TNode::TListType& result
) {
    NEventlog::TLocationCoordinatorGetPreviousTimestampByTimeQueryResponse event;
    event.MutableResponses()->Reserve(result.size());
    for (const NYT::TNode& element : result) {
        *event.AddResponses() = NYT::NodeToYsonString(element);
    }
    return event;
}

NEventlog::TLocationCoordinatorSetTargetTimestampToLocationNextTargetTimestamp MakeLocationCoordinatorSetTargetTimestampToLocationNextTargetTimestampEvent(
    const TMaybe<ui64>& locationNextTargetTimestamp
) {
    NEventlog::TLocationCoordinatorSetTargetTimestampToLocationNextTargetTimestamp event;
    if (locationNextTargetTimestamp.Defined()) {
        event.SetLocationNextTargetTimestamp(*locationNextTargetTimestamp);
    }
    return event;
}

NEventlog::TLocationCoordinatorSetTargetTimestampToLocationNewTargetTimestamp MakeLocationCoordinatorSetTargetTimestampToLocationNewTargetTimestampEvent(
    const TMaybe<ui64>& locationNewTargetTimestamp
) {
    NEventlog::TLocationCoordinatorSetTargetTimestampToLocationNewTargetTimestamp event;
    if (locationNewTargetTimestamp.Defined()) {
        event.SetLocationNewTargetTimestamp(*locationNewTargetTimestamp);
    }
    return event;
}

NEventlog::TLocationCoordinatorSetTargetTimestampToInstanceCurrentTimestamp MakeLocationCoordinatorSetTargetTimestampToInstanceCurrentTimestampEvent(
    const TMaybe<ui64>& instanceCurrentTimestamp
) {
    NEventlog::TLocationCoordinatorSetTargetTimestampToInstanceCurrentTimestamp event;
    if (instanceCurrentTimestamp.Defined()) {
        event.SetInstanceCurrentTimestamp(*instanceCurrentTimestamp);
    }
    return event;
}

NEventlog::TLocationCoordinatorGetTargetStateResult MakeLocationCoordinatorGettargetStateResultEvent(const TMaybe<ui64>& timestamp) {
    NEventlog::TLocationCoordinatorGetTargetStateResult event;
    if (timestamp.Defined()) {
        event.SetTimestamp(*timestamp);
    }
    return event;
}

NEventlog::TInstanceStateReaderListInstances MakeInstanceStateReaderListInstancesEvent(const TString& path, const NYT::TListOptions& listOptions) {
    NEventlog::TInstanceStateReaderListInstances event;
    event.SetPath(path);
    if (listOptions.MaxSize_.Defined()) {
        event.SetMaxSize(*listOptions.MaxSize_);
    }
    if (listOptions.AttributeFilter_.Defined()) {
        event.MutableAttributes()->Reserve(listOptions.AttributeFilter_->Attributes_.size());
        for (const TString& attribute : listOptions.AttributeFilter_->Attributes_) {
            *event.AddAttributes() = attribute;
        }
    }
    return event;
}

NEventlog::TInstanceStateReaderListPathResult MakeInstanceStateReaderListPathResultEvent(const NYT::TNode::TListType& result) {
    NEventlog::TInstanceStateReaderListPathResult event;
    event.MutableElements()->Reserve(result.size());
    for (const NYT::TNode& element : result) {
        *event.AddElements() = NYT::NodeToYsonString(element);
    }
    return event;
}

NEventlog::TInstanceStateReaderListInstancesResult MakeInstanceStateReaderListInstancesResultEvent(const TVector<TInstanceInfo>& instanceInfos) {
    NEventlog::TInstanceStateReaderListInstancesResult event;
    event.MutableInstanceInfos()->Reserve(instanceInfos.size());
    for (const TInstanceInfo& info : instanceInfos) {
        *event.AddInstanceInfos() = MakeInstanceInfoEvent(info);
    }
    return event;
}

NEventlog::TInstanceStateReaderFillStatesLookupStateRows MakeInstanceStateReaderFillStatesLookupStateRowsEvent(
    const TString& path,
    const NYT::TNode::TListType& keys,
    const NYT::TLookupRowsOptions& lookupOptions
) {
    NEventlog::TInstanceStateReaderFillStatesLookupStateRows event;
    *event.MutableLookupRequest() = MakeLookupRowsRequestEvent(path, keys, lookupOptions);
    return event;
}

NEventlog::TInstanceStateReaderFillStatesLookupStateRowsResult MakeInstanceStateReaderFillStatesLookupStateRowsResultEvent(const NYT::TNode::TListType& states) {
    NEventlog::TInstanceStateReaderFillStatesLookupStateRowsResult event;
    event.MutableStates()->Reserve(states.size());
    for (const NYT::TNode& state : states) {
        *event.AddStates() = NYT::NodeToYsonString(state);
    }
    return event;
}

NEventlog::TInstanceStateReaderFillStatesLookupVersionInfosRows MakeInstanceStateReaderFillStatesLookupVersionInfosRowsEvent(
    const TString& path,
    const NYT::TNode::TListType& keys,
    const NYT::TLookupRowsOptions& lookupOptions
) {
    NEventlog::TInstanceStateReaderFillStatesLookupVersionInfosRows event;
    *event.MutableLookupRequest() = MakeLookupRowsRequestEvent(path, keys, lookupOptions);
    return event;
}

NEventlog::TInstanceStateReaderFillStatesLookupVersionInfosRowsResult MakeInstanceStateReaderFillStatesLookupVersionInfosRowsResultEvent(
    const NYT::TNode::TListType& versionInfos
) {
    NEventlog::TInstanceStateReaderFillStatesLookupVersionInfosRowsResult event;
    event.MutableVersionInfos()->Reserve(versionInfos.size());
    for (const NYT::TNode& versionInfo : versionInfos) {
        *event.AddVersionInfos() = NYT::NodeToYsonString(versionInfo);
    }
    return event;
}

NEventlog::TInstanceStateReaderFillStatesResult MakeInstanceStateReaderFillStatesResult(const TVector<TInstanceInfo>& instanceInfos) {
    NEventlog::TInstanceStateReaderFillStatesResult event;
    event.MutableInstanceInfos()->Reserve(instanceInfos.size());
    for (const TInstanceInfo& info : instanceInfos) {
        *event.AddInstanceInfos() = MakeInstanceInfoEvent(info);
    }
    return event;
}

NEventlog::TInstanceStateWriterUpdateTimestampInfo MakeInstanceStateWriterUpdateTimestampInfoEvent(
    const NYT::TYPath& path,
    const NYT::TNode::TListType& rows,
    const NYT::TInsertRowsOptions& insertOptions
) {
    NEventlog::TInstanceStateWriterUpdateTimestampInfo event;
    *event.MutableInsertRowsRequest() = MakeInsertRowsRequestEvent(path, rows, insertOptions);
    return event;
}

NEventlog::TInstanceStateWriterSetTimestamp MakeInstanceStateWriterSetTimestampEvent(
    const NYT::TYPath& path,
    const NYT::TNode::TListType& rows,
    const NYT::TInsertRowsOptions& insertOptions
) {
    NEventlog::TInstanceStateWriterSetTimestamp event;
    *event.MutableInsertRowsRequest() = MakeInsertRowsRequestEvent(path, rows, insertOptions);
    return event;
}

NEventlog::TInstanceStateWriterCreateStateTable MakeInstanceStateWriterCreateStateTableEvent(
    const NYT::TYPath& path,
    const NYT::ENodeType& nodeType,
    const NYT::TCreateOptions& createOptions
) {
    NEventlog::TInstanceStateWriterCreateStateTable event;
    *event.MutableCreateRequest() = MakeCreateRequestEvent(path, nodeType, createOptions);
    return event;
}

NEventlog::TInstanceStateWriterWaitForMountedStateTable MakeInstanceStateWriterWaitForMountedStateTableEvent(
    const NYT::TYPath& path,
    const NYT::ETabletState& targetTabletsState,
    const NYT::TWaitForTabletsStateOptions& waitForTabletsStateOptions
) {
    NEventlog::TInstanceStateWriterWaitForMountedStateTable event;
    *event.MutableWaitRequest() = MakeWaitForTabletsStateRequestEvent(path, targetTabletsState, waitForTabletsStateOptions);
    return event;
}

NEventlog::TInstanceStateWriterCreateVersionsTable MakeInstanceStateWriterCreateVersionsTableEvent(
    const NYT::TYPath& path,
    const NYT::ENodeType& nodeType,
    const NYT::TCreateOptions& createOptions
) {
    NEventlog::TInstanceStateWriterCreateVersionsTable event;
    *event.MutableCreateRequest() = MakeCreateRequestEvent(path, nodeType, createOptions);
    return event;
}

NEventlog::TInstanceStateWriterWaitForMountedVersionsTable MakeInstanceStateWriterWaitForMountedVersionsTableEvent(
    const NYT::TYPath& path,
    const NYT::ETabletState& targetTabletsState,
    const NYT::TWaitForTabletsStateOptions& waitForTabletsStateOptions
) {
    NEventlog::TInstanceStateWriterWaitForMountedVersionsTable event;
    *event.MutableWaitRequest() = MakeWaitForTabletsStateRequestEvent(path, targetTabletsState, waitForTabletsStateOptions);
    return event;
}

NEventlog::TInstanceStateWriterCreateInstanceDocument MakeInstanceStateWriterCreateInstanceDocumentEvent(
    const NYT::TYPath& path,
    const NYT::ENodeType& nodeType,
    const NYT::TCreateOptions& createOptions
) {
    NEventlog::TInstanceStateWriterCreateInstanceDocument event;
    *event.MutableCreateRequest() = MakeCreateRequestEvent(path, nodeType, createOptions);
    return event;
}

NEventlog::TInstanceStateWriterRemoveInstanceDocument MakeInstanceStateWriterRemoveInstanceDocumentEvent(
    const NYT::TYPath& path,
    const NYT::TRemoveOptions& removeOptions
) {
    NEventlog::TInstanceStateWriterRemoveInstanceDocument event;
    *event.MutableRemoveRequest() = MakeRemoveRequestEvent(path, removeOptions);
    return event;
}

NEventlog::TYPUpdatesCoordinatorClientRequestTargetStateResponse MakeYPUpdatesCoordinatorClientRequestTargetStateResponseEvent(
    const NApi::TRspGetTargetState& response
) {
    NEventlog::TYPUpdatesCoordinatorClientRequestTargetStateResponse event;
    *event.MutableResponse() = MakeGetTargetStateResponseEvent(response);
    return event;
}

NEventlog::TYPUpdatesCoordinatorClientUpdateTargetState MakeYPUpdatesCoordinatorClientUpdateTargetStateEvent(
    const TTimestampClientInfo& timestampInfo
) {
    NEventlog::TYPUpdatesCoordinatorClientUpdateTargetState event;
    *event.MutableTimestampInfo() = MakeTimestampInfoEvent(timestampInfo);
    return event;
}

NEventlog::TYPUpdatesCoordinatorGetInstanceName MakeYPUpdatesCoordinatorGetInstanceNameEvent(
    const TMaybe<TString>& optionsInstanceName
) {
    NEventlog::TYPUpdatesCoordinatorGetInstanceName event;
    if (optionsInstanceName.Defined()) {
        event.SetInstanceNameInOptions(*optionsInstanceName);
    }
    return event;
}

NEventlog::TProviderClientGetTargetStateRequest MakeProviderClientGetTargetStateRequestEvent(
    const NApi::TReqGetTargetState& request,
    const NInfra::TAttributes& attributes
) {
    NEventlog::TProviderClientGetTargetStateRequest event;
    *event.MutableRequest() = MakeGetTargetStateRequestEvent(request, attributes);
    return event;
}

NEventlog::TProviderClientGetTargetStateResponse MakeProviderClientGetTargetStateResponseEvent(
    const NApi::TRspGetTargetState& response,
    const NInfra::TAttributes& attributes
) {
    NEventlog::TProviderClientGetTargetStateResponse event;
    *event.MutableResponse() = MakeGetTargetStateResponseEvent(response, attributes);
    return event;
}

} // namespace NYPUpdatesCoordinator
