#include "provider.h"

#include <infra/libs/yp_updates_coordinator/instance_state/reader/reader.h>
#include <infra/libs/yp_updates_coordinator/instance_state/writer/writer.h>
#include <infra/libs/yp_updates_coordinator/logger/events/events_decl.ev.pb.h>
#include <infra/libs/yp_updates_coordinator/sensors/sensors.h>

#include <infra/libs/sensors/sensor.h>

#include <yp/cpp/yp/client.h>
#include <yp/cpp/yp/token.h>

#include <library/cpp/retry/retry.h>

namespace NYPUpdatesCoordinator {

using namespace NYPUpdatesCoordinator::NDetail;

TProviderOptions MakeProviderOptions(const TProviderConfig& config) {
    TProviderOptions options;
    options.YtProxy = config.GetYtConfig().GetProxy();
    options.CypressRootPath = config.GetYtConfig().GetCypressRootPath();
    if (config.GetYtConfig().HasPrimaryMedium()) {
        options.PrimaryMedium = config.GetYtConfig().GetPrimaryMedium();
    }
    options.Service = config.GetService();
    options.YpClientOptions.SetAddress(config.GetYpClientConfig().GetAddress())
                           .SetToken(NYP::NClient::FindToken())
                           .SetEnableSsl(config.GetYpClientConfig().GetEnableSsl())
                           .SetEnableBalancing(config.GetYpClientConfig().GetEnableBalancing())
                           .SetTimeout(TDuration::Parse(config.GetYpClientConfig().GetTimeout()));
    return options;
}

TProviderConfig MakeProviderConfig(const TServedServiceConfig& serviceConfig) {
    TProviderConfig config = serviceConfig.GetProviderConfig();
    if (!config.HasService()) {
        config.SetService(serviceConfig.GetName());
    }
    if (!config.HasYtConfig()) {
        *config.MutableYtConfig() = serviceConfig.GetYtConfig();
    }
    return config;
}

TProviderOptions MakeProviderOptions(const TServedServiceConfig& serviceConfig) {
    return MakeProviderOptions(MakeProviderConfig(serviceConfig));
}

class TProvider::TImpl {
public:
    TImpl(TProviderOptions options, ICoordinatorPtr coordinator, NInfra::TLogFramePtr initLogFrame)
        : Options_(std::move(options))
        , YpClient_(NYP::NClient::CreateClient(Options_.YpClientOptions))
        , Coordinator_(coordinator)
    {
        Y_ENSURE(Coordinator_);

        Coordinator_->SetYpClient(YpClient_);

        TInstanceStateProviderWriterOptions stateWriterOptions;
        stateWriterOptions.YtProxy = Options_.YtProxy;
        stateWriterOptions.CypressRootPath = Options_.CypressRootPath;
        stateWriterOptions.Service = Options_.Service;
        stateWriterOptions.PrimaryMedium = Options_.PrimaryMedium;
        StateWriter_ = MakeHolder<TInstanceStateProviderWriter>(std::move(stateWriterOptions), initLogFrame);

        TInstanceStateReaderOptions stateReaderOptions;
        stateReaderOptions.YtProxy = Options_.YtProxy;
        stateReaderOptions.CypressRootPath = Options_.CypressRootPath;
        stateReaderOptions.Service = Options_.Service;
        StateReader_ = MakeHolder<TInstanceStateReader>(std::move(stateReaderOptions));

        StateWriter_->WaitForReadiness(initLogFrame);
    }

    ui64 GetTargetState(const NApi::TReqGetTargetState& request, NInfra::TLogFramePtr logFrame, NInfra::TSensorGroup sensorGroup) const {
        sensorGroup = NInfra::TSensorGroup{sensorGroup, NSensors::PROVIDER};
        const TVector<TInstanceInfo> infos = StateReader_->GetInstanceInfos(logFrame);

        TMaybe<ui64> state = Coordinator_->GetTargetState(request, infos, logFrame, sensorGroup);
        if (!state.Defined()) {
            logFrame->LogEvent(NEventlog::TCoordinatorDidNotSetTargetState());
            NInfra::TRateSensor{sensorGroup, NSensors::COORDINATOR_UNDEFINED_TARGET_STATE}.Inc();

            state = GetDefaultTargetState(logFrame);
        }

        Y_ENSURE(state.Defined());

        NInfra::TIntGaugeSensor{sensorGroup, NSensors::SENT_TIMESTAMP}.Set(*state);
        try {
            StateWriter_->UpdateSentState(request.instance_name(), TTimestampProviderInfo(*state).SentTime(Now()), logFrame);
        } catch (...) {
            // don't break on RO YT
        }

        return *state;
    }

private:
    ui64 GetDefaultTargetState(NInfra::TLogFramePtr logFrame) const {
        logFrame->LogEvent(NEventlog::TGetDefaultTargetState());
        return GenerateTimestamp(logFrame);
    }

    ui64 GenerateTimestamp(NInfra::TLogFramePtr logFrame) const {
        try {
            return *DoWithRetry<ui64, NYP::NClient::TResponseError>(
                [this, logFrame] {
                    logFrame->LogEvent(NEventlog::TYpGenerateTimestampAttempt(Options_.YpClientOptions.Address(), Options_.YpClientOptions.EnableBalancing()));
                    const ui64 result = YpClient_->GenerateTimestamp().GetValue(Options_.YpClientOptions.Timeout() * 2);
                    logFrame->LogEvent(NEventlog::TYpGenerateTimestampSuccess(result));
                    return result;
                },
                [logFrame] (const NYP::NClient::TResponseError& ex) {
                    logFrame->LogEvent(ELogPriority::TLOG_WARNING, NEventlog::TYpGenerateTimestampError(ex.what(), ex.Address()));
                },
                TRetryOptions()
                    .WithCount(3)
                    .WithSleep(TDuration::MilliSeconds(100))
                    .WithIncrement(TDuration::MilliSeconds(500)),
                /* throwLast */ true
            );
        } catch (const NYP::NClient::TResponseError& ex) {
            logFrame->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TYpGenerateTimestampFailure(ex.what(), ex.Address()));
            throw;
        }
    }

private:
    const TProviderOptions Options_;

    const NYP::NClient::TClientPtr YpClient_;

    THolder<TInstanceStateProviderWriter> StateWriter_;
    THolder<TInstanceStateReader> StateReader_;

    const ICoordinatorPtr Coordinator_;
};

TProvider::TProvider(TProviderOptions options, ICoordinatorPtr coordinator, NInfra::TLogFramePtr initLogFrame)
    : Impl_(MakeHolder<TImpl>(std::move(options), coordinator, initLogFrame))
{
}

TProvider::~TProvider() = default;

ui64 TProvider::GetTargetState(const NApi::TReqGetTargetState& request, NInfra::TLogFramePtr logFrame, NInfra::TSensorGroup sensorGroup) const {
    return Impl_->GetTargetState(request, logFrame, std::move(sensorGroup));
}

} // namespace NYPUpdatesCoordinator
