#include "service.h"

#include <infra/libs/yp_updates_coordinator/coordinator/location_coordinator.h>
#include <infra/libs/yp_updates_coordinator/grpc/server.h>
#include <infra/libs/yp_updates_coordinator/logger/events/events_decl.ev.pb.h>
#include <infra/libs/yp_updates_coordinator/logger/make_events.h>
#include <infra/libs/yp_updates_coordinator/provider/provider.h>
#include <infra/libs/yp_updates_coordinator/router_api/router_api.h>
#include <infra/libs/yp_updates_coordinator/sensors/sensors.h>

#include <infra/libs/http_service/service.h>
#include <infra/libs/logger/logger.h>
#include <infra/libs/memory_lock/memory_lock.h>
#include <infra/libs/sensors/sensor.h>
#include <infra/libs/sensors/sensor_group.h>

namespace NYPUpdatesCoordinator {

class TService::TImpl {
public:
    TImpl(TService* parent, TServiceConfig config)
        : Config_(std::move(config))
        , BaseSensorGroup_(NSensors::SERVICE_GROUP)
        , Logger_(Config_.GetLoggerConfig())
        , HttpService_(Config_.GetHttpServiceConfig(), CreateRouter(*parent))
        , GrpcServer_(Config_.GetGrpcServerConfig(), *parent)
        , ServiceMainLogFrame_(Logger_.SpawnFrame())
    {
        NInfra::NMemoryLock::LockSelfMemory(Config_.GetMemoryLock(), Logger_.SpawnFrame(), BaseSensorGroup_);
        InitProviders();
    }

    void Start() {
        ServiceMainLogFrame_->LogEvent(NEventlog::TServiceStart());
        GrpcServer_.Start(ServiceMainLogFrame_);
        HttpService_.Start(ServiceMainLogFrame_);
    }

    void Wait() {
        HttpService_.Wait(ServiceMainLogFrame_);
        GrpcServer_.Wait(ServiceMainLogFrame_);
    }

    void Ping(NInfra::TRequestPtr<NApi::TReqPing> request, NInfra::TReplyPtr<NApi::TRspPing> reply) {
        Logger_.SpawnFrame()->LogEvent(NEventlog::TPingRequest(MakeAttributesEvent(request->Attributes())));
        NInfra::TRateSensor{SensorGroup(NSensors::PING_GROUP), NSensors::REQUESTS}.Inc();

        NApi::TRspPing response;
        response.set_data("pong");

        reply->Set(response);
    }

    void GetTargetState(NInfra::TRequestPtr<NApi::TReqGetTargetState> request, NInfra::TReplyPtr<NApi::TRspGetTargetState> reply) {
        NInfra::TLogFramePtr logFrame = Logger_.SpawnFrame();
        logFrame->LogEvent(MakeGetTargetStateRequestEvent(request));

        NInfra::TSensorGroup sensorGroup = SensorGroup(NSensors::GET_TARGET_STATE_GROUP);
        sensorGroup.AddLabels({
            {NSensors::SERVICE, request->Get().service()},
            {NSensors::INSTANCE_NAME, request->Get().instance_name()},
        });
        NInfra::TRateSensor{sensorGroup, NSensors::REQUESTS}.Inc();

        NApi::TRspGetTargetState response;
        try {
            response = GetTargetState(request->Get(), logFrame, sensorGroup);
        } catch (...) {
            logFrame->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TGetTargetStateFailure(CurrentExceptionMessage()));
            NInfra::TRateSensor{sensorGroup, NSensors::FAILURES}.Inc();
            throw;
        }

        logFrame->LogEvent(MakeGetTargetStateResponseEvent(response));
        NInfra::TRateSensor{sensorGroup, NSensors::SUCCESSES, {{
            {NSensors::STATUS, NApi::TRspGetTargetState::EGetTargetStateStatus_Name(response.status())}
        }}}.Inc();
        reply->Set(std::move(response));
    }

    void ReopenLog(NInfra::TRequestPtr<NApi::TReqReopenLog> request, NInfra::TReplyPtr<NApi::TRspReopenLog> reply) {
        Logger_.SpawnFrame()->LogEvent(NEventlog::TReopenLogRequest(MakeAttributesEvent(request->Attributes())));
        NInfra::TRateSensor{SensorGroup(NSensors::REOPEN_LOG_GROUP), NSensors::REQUESTS}.Inc();

        Logger_.ReopenLog();

        NApi::TRspReopenLog response;
        response.set_data("ok");
        reply->Set(response);
    }

    void Sensors(NInfra::TRequestPtr<NApi::TReqSensors> request, NInfra::TReplyPtr<NApi::TRspSensors> reply) {
        Logger_.SpawnFrame()->LogEvent(NEventlog::TSensorsRequest(MakeAttributesEvent(request->Attributes())));
        NInfra::TRateSensor{SensorGroup(NSensors::SENSORS_GROUP), NSensors::REQUESTS}.Inc();

        NApi::TRspSensors result;
        TStringOutput stream(*result.MutableData());
        NInfra::SensorRegistryPrint(stream, NInfra::ESensorsSerializationType::JSON);
        reply->Set(result);
    }

    void Shutdown(NInfra::TRequestPtr<NApi::TReqShutdown> request, NInfra::TReplyPtr<NApi::TRspShutdown> reply) {
        Logger_.SpawnFrame()->LogEvent(NEventlog::TShutdownRequest(MakeAttributesEvent(request->Attributes())));
        NInfra::TRateSensor{SensorGroup(NSensors::SHUTDOWN_GROUP), NSensors::REQUESTS}.Inc();

        HttpService_.ShutDown();
        GrpcServer_.Shutdown();

        NApi::TRspShutdown response;
        response.set_data("ok");
        reply->Set(response);
    }

private:
    NApi::TRspGetTargetState GetTargetState(const NApi::TReqGetTargetState& request, NInfra::TLogFramePtr logFrame, NInfra::TSensorGroup sensorGroup) {
        NApi::TRspGetTargetState response;

        TProvider* provider = FindProvider(request.service());
        if (!provider) {
            logFrame->LogEvent(ELogPriority::TLOG_WARNING, NEventlog::TProviderForServiceNotFound());
            NInfra::TRateSensor{sensorGroup, NSensors::PROVIDER_NOT_FOUND}.Inc();

            response.set_status(NApi::TRspGetTargetState::UNKNOWN_SERVICE);
            response.set_error_message(TStringBuilder() << "Service \"" << request.service() << "\" not found");
            return response;
        }

        const ui64 targetTimestamp = provider->GetTargetState(request, logFrame, sensorGroup);
        response.set_status(NApi::TRspGetTargetState::OK);
        response.set_timestamp(targetTimestamp);
        return response;
    }

private:
    void InitProviders() {
        Providers_.reserve(Config_.GetServiceConfigs().size());
        for (const TServedServiceConfig& config : Config_.GetServiceConfigs()) {
            Providers_.emplace(
                config.GetName(),
                MakeHolder<TProvider>(
                    MakeProviderOptions(config),
                    MakeIntrusive<TLocationCoordinator>(MakeLocationCoordinatorOptions(config)),
                    ServiceMainLogFrame_
                )
            );
        }
    }

    TProvider* FindProvider(const TString& service) const {
        if (auto* it = Providers_.FindPtr(service)) {
            return it->Get();
        }
        return nullptr;
    }

    NInfra::TSensorGroup SensorGroup(const NInfra::TSensorGroup& group) const {
        return NInfra::TSensorGroup{BaseSensorGroup_, group};
    }

private:
    const TServiceConfig Config_;
    const NInfra::TSensorGroup BaseSensorGroup_;

    NInfra::TLogger Logger_;
    NInfra::THttpService HttpService_;
    TGrpcServer GrpcServer_;

    NInfra::TLogFramePtr ServiceMainLogFrame_;

    THashMap<TString, THolder<TProvider>> Providers_;
};

TService::TService(TServiceConfig config)
    : Impl_(MakeHolder<TImpl>(this, std::move(config)))
{
}

TService::~TService() = default;

void TService::Start() {
    Impl_->Start();
}

void TService::Wait() {
    Impl_->Wait();
}

void TService::Ping(NInfra::TRequestPtr<NApi::TReqPing> request, NInfra::TReplyPtr<NApi::TRspPing> reply) {
    Impl_->Ping(request, reply);
}

void TService::GetTargetState(NInfra::TRequestPtr<NApi::TReqGetTargetState> request, NInfra::TReplyPtr<NApi::TRspGetTargetState> reply) {
    Impl_->GetTargetState(request, reply);
}

void TService::ReopenLog(NInfra::TRequestPtr<NApi::TReqReopenLog> request, NInfra::TReplyPtr<NApi::TRspReopenLog> reply) {
    Impl_->ReopenLog(request, reply);
}

void TService::Sensors(NInfra::TRequestPtr<NApi::TReqSensors> request, NInfra::TReplyPtr<NApi::TRspSensors> reply) {
    Impl_->Sensors(request, reply);
}

void TService::Shutdown(NInfra::TRequestPtr<NApi::TReqShutdown> request, NInfra::TReplyPtr<NApi::TRspShutdown> reply) {
    Impl_->Shutdown(request, reply);
}

} // namespace NYPUpdatesCoordinator
