import argparse
import json
import logging
import os
import sys

import yt.wrapper as yt


YP_CLUSTERS = [
    'sas-test',
    'man-pre',
    'sas',
    'man',
    'vla',
    'myt',
    'iva',
    'xdc',
]

META_VERSIONS_SCHEMA = [
    {"name": "instance",        "type": "string", "sort_order": "ascending"},
    {"name": "timestamp",       "type": "uint64", "sort_order": "ascending"},
    {"name": "receive_time",    "type": "uint64", "lock": "client",      "aggregate": "min"},
    {"name": "update_time",     "type": "uint64", "lock": "client",      "aggregate": "min"},
    {"name": "sent_time",       "type": "uint64", "lock": "coordinator", "aggregate": "min"},
    {"name": "update_status",   "type": "any",    "lock": "client"},
    {"name": "update_attempts", "type": "uint64", "lock": "client",      "aggregate": "sum"},
]

META_STATE_SCHEMA = [
    {"name": "instance",           "type": "string", "sort_order": "ascending"},
    {"name": "current_timestamp",  "type": "uint64", "lock": "client"},
    {"name": "target_timestamp",   "type": "uint64", "lock": "client"},
    {"name": "sent_timestamp",     "type": "uint64", "lock": "coordinator"},
]

LOCATION_VERSIONS_SCHEMA = [
    {"name": "timestamp",   "type": "uint64", "sort_order": "ascending"},
    {"name": "update_time", "type": "uint64"},
]

TABLES = {
    "meta_versions":      {"path": "meta/versions", "schema": META_VERSIONS_SCHEMA},
    "meta_state":         {"path": "meta/state",    "schema": META_STATE_SCHEMA},
    "location_versions":  {"path": "coordinator/locations/{location}/versions", "schema": LOCATION_VERSIONS_SCHEMA},
}


def alter_table(yt_client, table_path, schema):
    logging.info(f'Unmounting {table_path}...')
    yt_client.unmount_table(table_path, sync=True)
    logging.info(f'Altering {table_path}...')
    logging.info(f'New schema:\n{json.dumps(schema, indent=2)}')
    yt_client.alter_table(table_path, schema=schema)
    logging.info(f'Mounting {table_path}...')
    yt_client.mount_table(table_path, sync=True)
    logging.info(f'Done for {table_path}')


def migrate(yp_cluster, cypress_root, service, table, locations):
    logging.info(f'Migrating table {table} for service {service} in YP-{yp_cluster.upper()}...')
    yt_client = yt.YtClient(f'yp-{yp_cluster}', token=os.getenv('YT_TOKEN'))

    table_info = TABLES[table]
    if table == 'location_versions':
        for location in locations:
            table_path = yt.ypath_join(cypress_root, service, table_info['path'].format(location=location))
            alter_table(yt_client, table_path, table_info['schema'])
    else:
        table_path = yt.ypath_join(cypress_root, service, table_info['path'])
        alter_table(yt_client, table_path, table_info['schema'])


def parse_args(argv):
    parser = argparse.ArgumentParser()
    parser.add_argument('--yp-cluster', required=True, choices=YP_CLUSTERS,
                        help="YP cluster name")
    parser.add_argument('--service', required=True,
                        help="Coordinated service name")
    parser.add_argument('--tables', required=True, nargs='+', choices=TABLES.keys(),
                        help="Tables to migrate")
    parser.add_argument('--cypress-root', required=True,
                        help="Cypress root path")
    parser.add_argument('--locations', nargs='+',
                        help="Service location (required if locations_versions table is set")
    args = parser.parse_args(argv)

    if 'location_versions' in args.tables:
        assert args.locations, 'List of locations must be set'

    return args


def main(argv):
    logging.basicConfig(level=logging.DEBUG)
    args = parse_args(argv)
    for table in args.tables:
        migrate(args.yp_cluster, args.cypress_root, args.service, table, args.locations)


if __name__ == '__main__':
    main(sys.argv[1:])
