from __future__ import print_function, division

import os
import py
import random
import re
import time
import threading
import gzip

from logging.handlers import BaseRotatingHandler


__all__ = ['RotatingHandler']


class RotatingHandler(BaseRotatingHandler):
    """
    Rotating logging handler with mixed logic of TimingRotatingFileHandler
    and base RotatingFileHandler.

    Rotates logfile at :arg tuple when: (e.g. (1, 0, 0) means 1:00AM every
    night). Rotated file name in this case will be <filename>.%Y-%m-%d.

    Additionally rotates forcibly by size. If size of logfile will be bigger than
    :arg int maxBytes: it will forcibly rotate.

    So, this could produce this files:

        filename => filename.1111-11-11  < current log file (symlink)
        filename.1111-11-11        < file rotated by size  (#1 of 11 day)
        filename.1111-11-11.001    < also rotated by size  (#2 of 11 day)
        filename.1111-11-11.002    < rotated at midnight   (#3 of 11 day)
        filename.1111-11-10        < rotated at midnight
        filename.1111-11-9         < rotated at midnight

    This way date in filename suffix always means day logfile belongs to.

    If program will be turned off in 01 day at 23:59, rotate will not be performed.
    But if you start it again at 03 day, it will rotate first logfile at mark it
    with 01 day suffix. This is done by looking into logfile modification time, and
    if it does not match current day -- rollover to that day.

    As a bonus - automatically reopen logfile if it's inode changed. So rotating by
    external tools is supported as well.
    """

    suffix = '%Y-%m-%d'
    suffix2 = '%03d'
    extMatch = re.compile(r'^\d{4}-\d{2}-\d{2}(\.\d{3})?(.gz)?$')

    def __init__(self, filename, maxBytes=10 * 1024 * 1024, backupCount=5, timeDiff=True):
        assert isinstance(maxBytes, int)
        assert maxBytes > 0
        assert isinstance(backupCount, int)
        assert backupCount >= 0
        assert isinstance(timeDiff, (bool, int))

        self._filename = py.path.local(filename)
        self._currentInode = None
        self._timeDiff = timeDiff
        self._stat = None
        self._dirpath = self._filename.dirpath()
        self._maxBytes = maxBytes
        self._backupCount = backupCount
        self._compressThread = None
        self._lock = threading.Lock()
        dfn = self._getDfn()
        BaseRotatingHandler.__init__(self, dfn, mode='a')

        if self._filename.check(exists=1) or self._filename.check(link=1):
            self._filename.remove()

        self._filename.mksymlinkto(self._dirpath.bestrelpath(py.path.local(dfn)))

    def shouldRollover(self, record):
        if not os.path.exists(self.baseFilename):
            self._reopen()
            self._cleanup()
            self._compress()
            return False

        self._getStat()
        if self._stat and self._stat.st_size >= self._maxBytes:
            return True

        if self._needReopen():
            self._reopen()
            self._cleanup()
            self._compress()

    def doRollover(self):
        self._prepareFileSlot(self._getDfn())
        self._reopen()
        self._cleanup()
        self._compress()

    def _needReopen(self):
        if self._getDfn() != self.baseFilename:
            return True

        prevInode = self._currentInode
        self._getStat()
        return self._currentInode != prevInode or self._currentInode is None

    def _getDfn(self, base=None):
        timeTuple = time.gmtime(int(time.time()) - self._getTimeDiff())
        dfn = '%s.%s' % (self._filename.strpath, time.strftime(self.suffix, timeTuple))
        return dfn

    def _getStat(self):
        try:
            self._stat = os.stat(self.baseFilename)
            self._currentInode = self._stat.st_ino
        except OSError:
            self._stat = None
            self._currentInode = None
        return self._stat

    def _open(self):
        stream = BaseRotatingHandler._open(self)
        self._getStat()
        return stream

    def _reopen(self):
        if self.stream:
            self.stream.close()
            self.stream = None
        self.baseFilename = self._getDfn(self._filename.strpath)
        self.stream = BaseRotatingHandler._open(self)

        if self._filename.check(exists=1) or self._filename.check(link=1):
            self._filename.remove()

        # To make replacing symlink operation atomic, we create
        # new symlink and rename (atomically) new to old.

        filename_tmp = py.path.local(self._filename + '_tmp')
        if filename_tmp.check(exists=1) or filename_tmp.check(link=1):
            filename_tmp.remove()

        filename_tmp.mksymlinkto(self._dirpath.bestrelpath(py.path.local(self.baseFilename)))
        filename_tmp.rename(self._filename)

    def _prepareFileSlot(self, dfn):
        """
        Find next filename we should use for rotation.
        This will fix any "clashes" (e.g. missing by sequence logs)
        """
        with self._lock:
            backups = self._getBackupNames()
            currentBackups = []
            for fn in backups:
                if fn.strpath.startswith(dfn):
                    currentBackups.append(fn)

            randomSuffix = ''.join(random.sample('abcdefghijklmnopqrstuvwxyz', 5))

            newBackups = []
            for idx, fn in enumerate(currentBackups, 1):
                tfn = py.path.local(dfn + '.' + (self.suffix2 % (idx, )) + '_' + randomSuffix)
                fn.rename(tfn)
                newBackups.append((tfn, fn.ext))

            for backup, ext in newBackups:
                fn = backup.strpath[:-(len(randomSuffix) + 1)]
                if ext == '.gz':
                    fn += '.gz'
                backup.rename(fn)

            return dfn

    def _getTimeDiff(self):
        if self._timeDiff is True:
            return time.altzone
        elif self._timeDiff is False:
            return 0
        else:
            return self._timeDiff

    def _cmp_name(self, a):
        a = a.strpath
        if a.endswith('.gz'):
            a = a[:-3]
        return a

    def _getBackupNames(self):
        prefix = self._filename.strpath + '.'
        plen = len(prefix)
        result = []

        for filename in self._dirpath.listdir():
            if filename.strpath[:plen] == prefix:
                suffix = filename.strpath[plen:]
                if self.extMatch.match(suffix):
                    result.append(filename)

        result.sort(key=lambda item: self._cmp_name(item))
        return result

    def _cleanup(self):
        with self._lock:
            result = self._getBackupNames()

            prefix = self._filename.strpath + '.'
            plen = len(prefix)

            while len(result) > self._backupCount:
                days = []
                for filename in result:
                    if filename.strpath[:plen] == prefix:
                        suffix = filename.strpath[plen:]
                        match = self.extMatch.match(suffix)
                        if match:
                            if suffix.endswith('.gz'):
                                suffix = suffix[:-3]
                            if match.groups()[0] is not None:
                                suffix = suffix[:-len(match.groups()[0])]
                            if suffix not in days:
                                days.append(suffix)

                oldestDay = sorted(days)[0]

                left = len(result) - self._backupCount
                for filename in reversed(result):
                    suffix = filename.strpath[plen:]
                    if suffix.startswith(oldestDay):
                        filename.remove()
                        result.remove(filename)
                        left -= 1

                    if left == 0:
                        break

    def _compress(self):
        def _runner():
            try:
                while True:
                    with self._lock:
                        toCompress = [
                            x for x in self._getBackupNames()
                            if x.ext != '.gz' and x.strpath != self.baseFilename
                        ]
                        if not toCompress:
                            break

                        fn = toCompress[0]
                        fngz = fn.dirpath().join(fn.basename + '.gz')
                        try:
                            fpgz = gzip.open(fngz.strpath, mode='wb')
                            with fn.open(mode='rb') as fp:
                                while 1:
                                    data = fp.read(4 * 1024 * 1024)
                                    if not data:
                                        break
                                    fpgz.write(data)
                        except py.error.ENOENT:
                            continue
                        finally:
                            fpgz.close()

                            try:
                                fn.remove()
                            except:
                                pass

            finally:
                self._compressThread = None

        if self._compressThread is None or not self._compressThread.is_alive():
            thr = threading.Thread(target=_runner)
            thr.daemon = True
            self._compressThread = thr
            thr.start()

    def handleError(self, record):
        BaseRotatingHandler.handleError(self, record)
        import sys
        ei = sys.exc_info()
        if isinstance(ei[1], IOError):
            # If we got any IOError -- exit completely!
            os._exit(1)
